##########################################################################
#
# StudioFactory
#
# The desktop Audio/Video studio.
# Copyright (C) 2002-2006  Peter Wendrich (pwsoft@syntiac.com)
# Homepage: http://www.syntiac.com/studiofactory.html
#
##########################################################################
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
##########################################################################
#
# SynFactory patch editor and management
#
##########################################################################

ccModule('SynFactory patch editor');

rd('#define THE_SYNFACTORY_VERSION "20070813"');

cc('showPatchConfig', 'guiShowWindow(patchConfigWindow);');
cc('routeMidiMessage', 'patchProcessMidi(aMidiMessage);');

cc('ProjectStruct', 'SynFactoryPatchPtr patches;');
cc('ProjectStruct', 'int maxPatches;');

setvar('int', 'thePatchColorForBackground', '0x668888');
setvar('int', 'thePatchColorForModules', '0xAACCCC');
setvar('int', 'thePatchColorForKnobs', '0x66AAAA');
setvar('int', 'thePatchColorForHighlight', '0xCCEEEE');
setvar('int', 'thePatchColorForShadow', '0x000000');
setvar('int', 'theCurrentCableType', '2');
setvar('bool', 'theMainWindowDoubleBuffered', 'false');

cc('thePatchColorForBackground', 'mainWindowRefresh=true;');
cc('thePatchColorForModules', 'mainWindowRefresh=true;');
cc('thePatchColorForKnobs', 'mainWindowRefresh=true;');
cc('thePatchColorForHighlight', 'mainWindowRefresh=true;');
cc('thePatchColorForShadow', 'mainWindowRefresh=true;');
cc('theCurrentCableType', 'mainWindowRefresh=true;');
cc('theMainWindowDoubleBuffered', 'guiDoubleBuffer(mainWindow, theMainWindowDoubleBuffered);');

configAddColorSelector('globalConfig', 'STRING_COLORS', 'STRING_COLOR_PATCH_BACKGROUND', 'thePatchColorForBackground');
configAddColorSelector('globalConfig', 'STRING_COLORS', 'STRING_COLOR_PATCH_MODULES', 'thePatchColorForModules');
configAddColorSelector('globalConfig', 'STRING_COLORS', 'STRING_COLOR_PATCH_KNOBS', 'thePatchColorForKnobs');
configAddColorSelector('globalConfig', 'STRING_COLORS', 'STRING_COLOR_PATCH_HIGHLIGHT', 'thePatchColorForHighlight');
configAddColorSelector('globalConfig', 'STRING_COLORS', 'STRING_COLOR_PATCH_SHADOW', 'thePatchColorForShadow');

rv('const int outputXMargin=32;');
rv('const int outputXMargin2=8;');
rv('const int minXSize=64;');
rv('const int knobXOffset=48;');
rv('const int knobXSpacing=40;');
rv('const int ioYSpacing=14;');
rv('const int ioXSize=10;');
rv('const int ioYSize=10;');
rv('const int maxAutoSize=8;');

rv('int cableTypeList[4]={STRING_CABLE_TYPE_0, STRING_CABLE_TYPE_1, STRING_CABLE_TYPE_2, STRING_};');
rv('bool showResizeMarkers;');
rd('typedef const char *(*TranslationFunc)(int value, int mode);');

# !!! forward declaration of refreshObject
rs(<<EOF);
static void refreshObject(SynFactoryPatchPtr aPatch, SynFactoryObject *aObject);

EOF



rc(<<EOF);
static char TranslationStr[128];
static const char *translateFreq(int value, int mode) {
	double freq=exposc(value)/97392.0;
	switch(mode&24) {
	case 8: freq*=2; break;
	case 16: freq*=4; break;
	case 24: freq*=8; break;
	default: break;
	}
	if (freq>=1000) {
		sprintf(TranslationStr, "%dHz%01d", (int)(freq), (int)((freq-(int)(freq))*10));
		return TranslationStr;
	}
	if (freq>=100) {
		sprintf(TranslationStr, "%dHz%02d", (int)(freq), (int)((freq-(int)(freq))*100));
		return TranslationStr;
	}
	if (freq>=10) {
		sprintf(TranslationStr, "%dHz%03d", (int)(freq), (int)((freq-(int)(freq))*1000));
		return TranslationStr;
	}
	if (freq>=1) {
		sprintf(TranslationStr, "%dHz%04d", (int)(freq), (int)((freq-(int)(freq))*10000));
		return TranslationStr;
	}
	sprintf(TranslationStr, "Hz%05d", (int)(freq*100000));
	return TranslationStr;
}

static const char *translateBpm(int value, int mode) {
	double bpm=exposc(value)/1623.2;
	switch(mode&24) {
	case 8: bpm*=2; break;
	case 16: bpm*=4; break;
	case 24: bpm*=8; break;
	default: break;
	}

	if (bpm>=1000) {
		return ">999BPM";
	}
	if (bpm>=100) {
		sprintf(TranslationStr, "%d.%02dBPM", (int)(bpm), (int)((bpm-(int)(bpm))*100));
		return TranslationStr;
	}
	if (bpm>=10) {
		sprintf(TranslationStr, "%d.%03dBPM", (int)(bpm), (int)((bpm-(int)(bpm))*1000));
		return TranslationStr;
	}
	sprintf(TranslationStr, "%d.%04dBPM", (int)(bpm), (int)((bpm-(int)(bpm))*10000));
	return TranslationStr;
}

static const char *translateTrig(int value, int mode) {
	double time=(double)(exposc(value))/(double)(PLAYBACKRATE*32.0);
	if (time>=1000) {
		sprintf(TranslationStr, "%ds", (int)(time));
		return TranslationStr;
	}
	if (time>=100) {
		sprintf(TranslationStr, "%ds%01d", (int)(time), (int)((time-(int)(time))*10));
		return TranslationStr;
	}
	if (time>=10) {
		sprintf(TranslationStr, "%ds%02d", (int)(time), (int)((time-(int)(time))*100));
		return TranslationStr;
	}
	if (time>=1) {
		sprintf(TranslationStr, "%ds%03d", (int)(time), (int)((time-(int)(time))*1000));
		return TranslationStr;
	}
	sprintf(TranslationStr, "s%04d", (int)(time*10000));
	return TranslationStr;
}

static const char *translateAHD(int value, int mode) {
	double time=(double)(0x7FFFFFFF/(double)(exposc(32768-value)))/(double)(PLAYBACKRATE);
	if (time>=1000) {
		sprintf(TranslationStr, "%ds", (int)(time));
		return TranslationStr;
	}
	if (time>=100) {
		sprintf(TranslationStr, "%ds%01d", (int)(time), (int)((time-(int)(time))*10));
		return TranslationStr;
	}
	if (time>=10) {
		sprintf(TranslationStr, "%ds%02d", (int)(time), (int)((time-(int)(time))*100));
		return TranslationStr;
	}
	if (time>=1) {
		sprintf(TranslationStr, "%ds%03d", (int)(time), (int)((time-(int)(time))*1000));
		return TranslationStr;
	}
	sprintf(TranslationStr, "s%04d", (int)(time*10000));
	return TranslationStr;
}

static const char *translateCC(int value, int mode) {
	int v=value/256;
	if (v<0) v=0;
	sprintf(TranslationStr, "%3d (%02Xh)", v, v);
	return TranslationStr;
}
EOF


my @objectTypeStructDef=();
my @objectLoadNames=();
my @objectTitles=();
my @objectMenu=();
my @objectKnobs;
my @objectKnobsValue;
my @objectKnobsTranslation;
my @objectInputs;
my @objectOutputs;
my $objectAutoSize;
my $objectRoutine='';
my $objectHelpText;
my $objectKeys='';
my $objectBufferSize='';
my $objectBufferKeep='';
my $subMenuFlag=0;

open FILE, "studiofactory_objects.txt" || die "couldn't open studiofactory_objects.txt";
while(<FILE>) {
	s/\r//;
	s/\n//;
	if (!(/^\s*\#/)) {
		if (/^(.+?)\s+((\".+?\")|NULL)\s+((\".+?\")|NULL)\s+(.)\s+(.)\s+(.+?)\s+(.+?)\s+((\".+?\")|NULL)((\s+.*)?)/) {
			my $enumName=$1;
			my $names=$12;
			push(@objectTypeStructDef, $enumName);    
			push(@objectLoadNames, $2);
			push(@objectTitles, $4);
			if (!($6 eq '-')) {
				$objectKeys .= "\t\t\t\tcase '$6': n=addSynFactoryObject(currentPatch, $1, false); break;\n";
			}
			if ($7 eq 'A') {
				$objectAutoSize .= "\tcase $enumName: return true;\n";
			} else {
				if (!($8 eq 'NULL')) {
					$objectRoutine .= "\n\tcase $enumName: routine=dspRoutine_$8; break;";
				}
			}
			if (!($9 eq 'NULL')) {
				$objectHelpText .= "\tcase $enumName: return STRING_OBJECTHELP_$9;\n";
			}
			if (!($10 eq 'NULL')) {
				push(@objectMenu, "{$1+1024, {$10}}");
			}
			while (defined($names) && $names =~ s/([KIO])(\d+)\:(\".*?\")(,(\d+))?\s*//) {
				if ($1 eq 'K') {
					$objectKnobs[$2] .= "\t\tcase $enumName: return $3; break;\n";
					if (defined $5) {
						$objectKnobsValue[$2] .= "\t\tcase $enumName: return $5; break;\n";
					}
				}        
				if ($1 eq 'I') {
					$objectInputs[$2] .= "\t\tcase $enumName: return $3; break;\n";
				}        
				if ($1 eq 'O') {
					$objectOutputs[$2] .= "\t\tcase $enumName: return $3; break;\n";
				}
			}
			while (defined($names) && $names =~ s/T(\d+)\:(\w+)//) {
				$objectKnobsTranslation[$1] .= "\t\tcase $enumName: return $2; break;\n";
			}
			while (defined($names) && $names =~ s/B\:(.+?);//) {
				$objectBufferSize .= "\n\tcase $enumName: return $1;";
			}
			while (defined($names) && $names =~ s/BS\:(.+?);//) {
				$objectBufferSize .= "\n\tcase $enumName: return $1;";
				$objectBufferKeep .= "\n\tcase $enumName: return true;";
			}
		}
		if (/^(\".*?\")$/) {
			if ($subMenuFlag) {
				push(@objectMenu, "{0, {NULL}}");
			}
			push(@objectMenu, "{-1, {$1}}");
			$subMenuFlag=1;
		}
		if (/^\-\-\-$/) {
			push(@objectMenu, "{-2, {NULL}}");
		}
	}
}
close FILE;
push(@objectLoadNames, 'NULL'); # terminate list
genEnumStruct("enum", "ObjectType", \@objectTypeStructDef, ',');
genTable('const char * const', 'objectLoadNames', \@objectLoadNames);
genTable('const char * const', 'objectTitles', \@objectTitles);
if ($subMenuFlag) {
	push(@objectMenu, "{0, {NULL}}");
}
push(@objectMenu, "{-2, {NULL}}");
push(@objectMenu, "{0, {NULL}}");
genTable("MenuDef", 'synFactoryMenuDef', \@objectMenu);
#  genMenuDef('synFactoryMenuDef', 'studiofactory_mainmenu.txt');
	
my @SynFactoryObjectStructDef=(
	'bool selected;',
	'ObjectType objectType;',
	'int dsp;',
	'bool showLabel;',
	'int mode;',
	'int x;',
	'int y;',
	'int xs;',
	'int ys;',
	'int labelXs;',
	'int maxInput;',
	'GuiWindowPtr window;',
	'char *name;',
	'int mainColor;',
	'int borderColor;',
	'int currentOutput;',
	'struct {',
		'int fromObject;',
		'int fromOutput;',
		'int cableColor;',
	'} connections[maxObjectIo];'
);
genEnumStruct("struct", "SynFactoryObject", \@SynFactoryObjectStructDef);

my @SynFactoryPatchStructDef=(
	'bool selected;',
	'bool unfolded;',
	'char *patchName;',
	'int offsX;',
	'int offsY;',
	'SynFactoryObjectPtr objects;',
	'int maxObjects;',
	'int midiClockPort;',
	'int midiPort;',
	'int midiChannel;',
	'int sustainCC;',
	'int sustainValue;'
);
genEnumStruct("struct", "SynFactoryPatch", \@SynFactoryPatchStructDef);

rc("static StringIndex objectHelpText(ObjectType type) {\n  switch(type) {\n${objectHelpText}\tdefault: return STRING_NOHELP_AVAILABLE;\n\t}\n}\n\n");
rc("static bool autoSize(ObjectType type) {\n\tswitch(type) {\n${objectAutoSize}\tdefault: return false;\n  }\n}\n\n");
rc("static int objectBufferSize(ObjectType type) {\n  switch(type) {${objectBufferSize}\n  default: return 0;\n  }\n}\n\n");
rc("static bool objectBufferKeep(ObjectType type) {\n  switch(type) {${objectBufferKeep}\n  default: return false;\n  }\n}\n\n");

rc("static const char *knobName(ObjectType type, int nr) {");
rc("\tswitch(nr) {");
for(my $i=0;$i<100;$i++) {
	if (defined($objectKnobs[$i])) {
		rc("\tcase $i:");
		rc("\t\tswitch(type) {");
		rc($objectKnobs[$i]);
		rc("\t\tdefault: break;");
		rc("\t\t}");
		rc("\t\tbreak;");
	}
}
rc("\t}\n\treturn NULL;\n}\n\n");

rc("static int knobDefaultValue(ObjectType type, int nr) {");
rc("\tswitch(nr) {");
for(my $i=0;$i<100;$i++) {
	if (defined($objectKnobsValue[$i])) {
		rc("\tcase $i:");
		rc("\t\tswitch(type) {");
		rc($objectKnobsValue[$i]);
		rc("\t\tdefault: break;");
		rc("\t\t}");
		rc("\t\tbreak;");
	}
}
rc("\tdefault:\n\t\tbreak;\n");
rc("\t}\n\treturn 0;\n}\n\n");

rc("static TranslationFunc knobTranslateFunction(ObjectType type, int nr) {");
rc("\tswitch(nr) {");
for(my $i=0;$i<100;$i++) {
	if (defined($objectKnobsTranslation[$i])) {
		rc("\tcase $i:");
		rc("\t\tswitch(type) {");
		rc($objectKnobsTranslation[$i]);
		rc("\t\tdefault: break;");
		rc("\t\t}");
		rc("\t\tbreak;");
	}
}
rc("\tdefault:\n\t\tbreak;\n");
rc("\t}\n\treturn NULL;\n}\n\n");

rc("static const char *inputName(ObjectType type, int nr) {");
rc("\tif (autoSize(type) && nr<maxAutoSize) return \"\";\n");
rc("\tswitch(nr) {");
for(my $i=0;$i<100;$i++) {
	if (defined($objectInputs[$i])) {
		rc("\tcase $i:");
		rc("\t\tswitch(type) {");
		rc($objectInputs[$i]);
		rc("\t\tdefault: break;\n\t\t}");
		rc("\t\tbreak;\n");
	}
}
rc("\t}\n\treturn NULL;\n}\n\n");

rc("static const char *outputName(ObjectType type, int nr) {");
rc("\tswitch(nr) {");
for(my $i=0;$i<100;$i++) {
	if (defined($objectOutputs[$i])) {
		rc("\tcase $i:");
		rc("\t\tswitch(type) {");
		rc($objectOutputs[$i]);
#      $rc .= "    default: return NULL;\n\t\t}\n";
		rc("\t\tdefault: break;\n\t\t}");
		rc("\t\tbreak;\n");
	}
}
rc("\t}\n  return NULL;\n}\n\n");

rc(<<EOF);
static void drawPot(ContextPtr aContext, int x, int y, const char *name, int value, int mode, int color, int highlight, int shadow, bool halfRangePot, TranslationFunc transfunc1, TranslationFunc transfunc2) {
	char tmpstr[64];
	int insidex, insidey, outsidex, outsidey;

	guiSelectPotFont(aContext);
	guiDrawTransparent(aContext);
	guiSetTextAlignment(aContext, alignTopCenter);

	guiSelectPen1Color(aContext, shadow);
	guiSelectFillColor(aContext, shadow);
	guiDrawEllipse(aContext, x+2, y+2, x + 26, y + 26);

	guiSelectPen1Color(aContext, color);
	guiSelectFillColor(aContext, color);
	guiDrawEllipse(aContext, x, y, x + 24, y + 24);

	if (halfRangePot) {
		insidex  = x+12+(int)(sin((value-16384)/7000.0)*  5.0);
		insidey  = y+12+(int)(cos((value-16384)/7000.0)*- 5.0);
		outsidex = x+12+(int)(sin((value-16384)/7000.0)* 12.0);
		outsidey = y+12+(int)(cos((value-16384)/7000.0)*-12.0);
	} else {
		insidex  = x+12+(int)(sin(value/14000.0)*  5.0);
		insidey  = y+12+(int)(cos(value/14000.0)*- 5.0);
		outsidex = x+12+(int)(sin(value/14000.0)* 12.0);
		outsidey = y+12+(int)(cos(value/14000.0)*-12.0);
	}
	guiSelectPen3Color(aContext, highlight);
	guiDrawLine(aContext,insidex, insidey, outsidex, outsidey);
	
	guiSelectPen1Color(aContext, shadow);

	// Pot name
	if (name) {
		guiDrawText(aContext,
			x + 2,
			y + 26,
			x + 26,
			y + 26 + 100,
			name);
	}

	// Pot value
	sprintf(tmpstr, "%d", value);
	guiDrawText(aContext,
		x + 2, y + ((!!name)?38:26),
		x + 26, y + ((!!name)?38:26) + 100,
		tmpstr);

	// Secondary (translated) pot value
	if (transfunc1) {
		const char *convertedstr=(*transfunc1)(value, mode);
		guiDrawText(aContext,
			x + 2, y + ((!!name)?50:38),
			x + 26, y + ((!!name)?50:38) + 100,
			convertedstr);
	}
	if (transfunc2) {
		const char *convertedstr=(*transfunc2)(value, mode);
		guiDrawText(aContext,
			x + 2, y + ((!!name)?62:50),
			x + 26, y + ((!!name)?62:50) + 100,
			convertedstr);
	}
}

static DspRoutinePtr addRoutines[maxAutoSize]={
	dspRoutine_ADD1,  dspRoutine_ADD2,  dspRoutine_ADD3,  dspRoutine_ADD4,  dspRoutine_ADD5,  dspRoutine_ADD6,  dspRoutine_ADD7, dspRoutine_ADD8,
//   dspRoutine_ADD9,  dspRoutine_ADD10, dspRoutine_ADD11, dspRoutine_ADD12, dspRoutine_ADD13, dspRoutine_ADD14, dspRoutine_ADD15,
//  dspRoutine_ADD16, dspRoutine_ADD17, dspRoutine_ADD18, dspRoutine_ADD19, dspRoutine_ADD20, dspRoutine_ADD21, dspRoutine_ADD22, dspRoutine_ADD23,
//  dspRoutine_ADD24, dspRoutine_ADD25, dspRoutine_ADD26, dspRoutine_ADD27, dspRoutine_ADD28, dspRoutine_ADD29, dspRoutine_ADD30, dspRoutine_ADD31,
};
static DspRoutinePtr andRoutines[maxAutoSize]={
	dspRoutine_AND1,  dspRoutine_AND2,  dspRoutine_AND3,  dspRoutine_AND4,  dspRoutine_AND5,  dspRoutine_AND6,  dspRoutine_AND7, dspRoutine_AND8, 
//   dspRoutine_AND9,  dspRoutine_AND10, dspRoutine_AND11, dspRoutine_AND12, dspRoutine_AND13, dspRoutine_AND14, dspRoutine_AND15,
//  dspRoutine_AND16, dspRoutine_AND17, dspRoutine_AND18, dspRoutine_AND19, dspRoutine_AND20, dspRoutine_AND21, dspRoutine_AND22, dspRoutine_AND23,
//  dspRoutine_AND24, dspRoutine_AND25, dspRoutine_AND26, dspRoutine_AND27, dspRoutine_AND28, dspRoutine_AND29, dspRoutine_AND30, dspRoutine_AND31,
};
static DspRoutinePtr orRoutines[maxAutoSize]={
	dspRoutine_OR1,  dspRoutine_OR2,  dspRoutine_OR3,  dspRoutine_OR4,  dspRoutine_OR5,  dspRoutine_OR6,  dspRoutine_OR7, dspRoutine_OR8, 
//    dspRoutine_OR9,  dspRoutine_OR10, dspRoutine_OR11, dspRoutine_OR12, dspRoutine_OR13, dspRoutine_OR14, dspRoutine_OR15,
//   dspRoutine_OR16, dspRoutine_OR17, dspRoutine_OR18, dspRoutine_OR19, dspRoutine_OR20, dspRoutine_OR21, dspRoutine_OR22, dspRoutine_OR23,
//   dspRoutine_OR24, dspRoutine_OR25, dspRoutine_OR26, dspRoutine_OR27, dspRoutine_OR28, dspRoutine_OR29, dspRoutine_OR30, dspRoutine_OR31,
};
static void setDspRoutine(int aProject, int aPatch, int obj) {
	DspRoutinePtr routine;
	int i;

	switch(projects[aProject].patches[aPatch].objects[obj].objectType) {$objectRoutine
	case OBJ_ADD:
		routine=dspRoutine_Dummy;
		for(i=maxAutoSize-1;i>=0;i--) {
			if (projects[aProject].patches[aPatch].objects[obj].connections[i].fromObject) {
				routine=addRoutines[i];
				break;
			}
		}
		break;
	case OBJ_AND:
		routine=dspRoutine_Dummy;
		for(i=maxAutoSize-1;i>=0;i--) {
			if (projects[aProject].patches[aPatch].objects[obj].connections[i].fromObject) {
				routine=andRoutines[i];
				break;
			}
		}
		break;
	case OBJ_OR:
		routine=dspRoutine_Dummy;
		for(i=maxAutoSize-1;i>=0;i--) {
			if (projects[aProject].patches[aPatch].objects[obj].connections[i].fromObject) {
				routine=orRoutines[i];
				break;
			}
		}
		break;
	default:
		routine=NULL;
		break;
	}
	if (routine == dspRoutine_Dummy) {
		// Clear output of autosize modules if not connected (ADD, AND, OR)
		projects[aProject].dsp[projects[aProject].patches[aPatch].objects[obj].dsp].io[0].output=0;
	}
	projects[aProject].dsp[projects[aProject].patches[aPatch].objects[obj].dsp].dspRoutine = routine;
	if (thePlayingProject == aProject) {
		~cc~recalcDsp~
	}
}

static void setMode(int aProject, int aPatch, int aObject, int aMode) {
	if (projects[aProject].patches[aPatch].objects[aObject].mode != aMode) {

		DspRoutinePtr routine=projects[aProject].dsp[projects[aProject].patches[aPatch].objects[aObject].dsp].dspRoutine;

		switch(projects[aProject].patches[aPatch].objects[aObject].objectType) {
		case OBJ_OSC:
			switch(aMode) {
// With AA filter
			case  0:  routine=dspRoutine_OSC_SIN_L; break;
			case  1:  routine=dspRoutine_OSC_TRI_L; break;
			case  2:  routine=dspRoutine_OSC_SAW_AA_L; break;
			case  3:  routine=dspRoutine_OSC_SPS_AA_L; break;
			case  4:  routine=dspRoutine_OSC_PLS_AA_L; break;
			case  5:  routine=dspRoutine_OSC_RND_L; break;

			case  8:  routine=dspRoutine_OSC_SIN_M; break;
			case  9:  routine=dspRoutine_OSC_TRI_M; break;
			case 10:  routine=dspRoutine_OSC_SAW_AA_M; break;
			case 11:  routine=dspRoutine_OSC_SPS_AA_M; break;
			case 12:  routine=dspRoutine_OSC_PLS_AA_M; break;
			case 13:  routine=dspRoutine_OSC_RND_M; break;

			case 16:  routine=dspRoutine_OSC_SIN_H; break;
			case 17:  routine=dspRoutine_OSC_TRI_H; break;
			case 18:  routine=dspRoutine_OSC_SAW_AA_H; break;
			case 19:  routine=dspRoutine_OSC_SPS_AA_H; break;
			case 20:  routine=dspRoutine_OSC_PLS_AA_H; break;
			case 21:  routine=dspRoutine_OSC_RND_H; break;

// No AA filter (bit 5)
			case 32:  routine=dspRoutine_OSC_SIN_L; break;
			case 33:  routine=dspRoutine_OSC_TRI_L; break;
			case 34:  routine=dspRoutine_OSC_SAW_L; break;
			case 35:  routine=dspRoutine_OSC_SPS_L; break;
			case 36:  routine=dspRoutine_OSC_PLS_L; break;
			case 37:  routine=dspRoutine_OSC_RND_L; break;

			case 40:  routine=dspRoutine_OSC_SIN_M; break;
			case 41:  routine=dspRoutine_OSC_TRI_M; break;
			case 42:  routine=dspRoutine_OSC_SAW_M; break;
			case 43:  routine=dspRoutine_OSC_SPS_M; break;
			case 44:  routine=dspRoutine_OSC_PLS_M; break;
			case 45:  routine=dspRoutine_OSC_RND_M; break;

			case 48:  routine=dspRoutine_OSC_SIN_H; break;
			case 49:  routine=dspRoutine_OSC_TRI_H; break;
			case 50:  routine=dspRoutine_OSC_SAW_H; break;
			case 51:  routine=dspRoutine_OSC_SPS_H; break;
			case 52:  routine=dspRoutine_OSC_PLS_H; break;
			case 53:  routine=dspRoutine_OSC_RND_H; break;
			default:  routine=dspRoutine_Dummy; break;
			}
			break;
		case OBJ_PERLINOSC:
			switch(aMode) {
			case  0:  routine=dspRoutine_PERLIN_L; break;
			case  8:  routine=dspRoutine_PERLIN_M; break;
			case 16:  routine=dspRoutine_PERLIN_H; break;
			default:  routine=dspRoutine_Dummy; break;
			}
			break;
		case OBJ_PMOSC:
			switch(aMode) {
			case  0:  routine=dspRoutine_PMOSC_L; break;
			case  8:  routine=dspRoutine_PMOSC_M; break;
			case 16:  routine=dspRoutine_PMOSC_H; break;
			default:  routine=dspRoutine_Dummy; break;
			}
			break;
		default:
			break;
		}

		projects[aProject].dsp[projects[aProject].patches[aPatch].objects[aObject].dsp].dspRoutine=routine;
		projects[aProject].patches[aPatch].objects[aObject].mode = aMode;
		setProjectModified(aProject);
		refreshObject(projects[aProject].patches + aPatch, projects[aProject].patches[aPatch].objects + aObject);

		if (thePlayingProject == aProject) {
			~cc~recalcDsp~
		}
	}
}

static int calcObjectXs(ObjectType type) {
	int i;
	int xs=knobXOffset;
	int margin=outputXMargin2;

	switch(type) {
	case OBJ_DRM4:
	case OBJ_DRM8:
		margin = outputXMargin;
		xs += (16*16);
		break;
	default:
		for(i=0;i<maxObjectIo;i++) {
			if (outputName(type, i) && strlen(outputName(type, i))>0) {
				margin = outputXMargin;
			}
			if (knobName(type, i)) {
				xs = knobXOffset + knobXSpacing * ((i>>1)+1);
			}
		}
		break;
	}
	xs += margin;
	if (xs < minXSize) {
		xs = minXSize;
	}

	return xs;
}

static int calcObjectYs(ObjectType type, int autoSizeOffset) {
	int i;
	int maxIo=autoSizeOffset;

	for(i=0;i<maxObjectIo;i++) {
		if (autoSizeOffset==0 && i>maxIo && (inputName(type, i) || outputName(type,i))) {
			maxIo=i;
		}
		if (maxIo<2 && (i&1)==0 && knobName(type, i)) {
			maxIo=2;
		}
		if (maxIo<3 && (i&1)==0 && knobName(type, i) && knobTranslateFunction(type, i)) {
			maxIo=3;
		}
		if (maxIo<6 && (i&1)==1 && knobName(type, i)) {
			maxIo=6;
		}
	}

	return (maxIo+3)*ioYSpacing+ioYSize;
}

static void adjustAutoSize(int aProject, int aPatch, int aObject) {
	int max=1;
	int i;

	for(i=1;i<maxAutoSize-1;i++) {
		if (projects[aProject].patches[aPatch].objects[aObject].connections[i].fromObject) {
			max=i+1;
		}
	}
	if (projects[aProject].patches[aPatch].objects[aObject].connections[maxAutoSize-1].fromObject) {
		max=maxAutoSize-1;
	}
	
	projects[aProject].patches[aPatch].objects[aObject].maxInput=max;
	projects[aProject].patches[aPatch].objects[aObject].ys=calcObjectYs(projects[aProject].patches[aPatch].objects[aObject].objectType, max);
	setDspRoutine(aProject, aPatch, aObject);
}

static int addSynFactoryObject(int patch, ObjectType type, bool keepSelection) {
	int n=projects[theCurrentProject].patches[patch].maxObjects;
	int dsp=0;

	logprintf("addSynFactoryObject type %d\\n", (int)type);

	if ((type != OBJ_BOX) && (type != OBJ_ROUNDBOX) && (type != OBJ_ELLIPSE) && (type != OBJ_TEXT) && (type != OBJ_HTML)) {
		dsp=nextFreeDsp();
		if (0 == dsp) {
			return -1;
		}
		if (objectBufferSize(type)) {
			projects[theCurrentProject].dsp[dsp].buffer = (signed short*)calloc(1, objectBufferSize(type));
			if (NULL == projects[theCurrentProject].dsp[dsp].buffer) {
				return -1;
			}
		}
		for(int i=0;i<maxObjectIo;i++) {
			projects[theCurrentProject].dsp[dsp].io[i].knob=knobDefaultValue(type, i);      
		}

		projects[theCurrentProject].dsp[dsp].inUse=true;
	}

	projects[theCurrentProject].patches[patch].maxObjects++;
	projects[theCurrentProject].patches[patch].objects=(SynFactoryObjectPtr)realloc(projects[theCurrentProject].patches[patch].objects, projects[theCurrentProject].patches[patch].maxObjects*sizeof(SynFactoryObject));
	memset(&(projects[theCurrentProject].patches[patch].objects[n]), 0, sizeof(SynFactoryObject));
	projects[theCurrentProject].patches[patch].objects[n].xs=calcObjectXs(type);
	projects[theCurrentProject].patches[patch].objects[n].ys=calcObjectYs(type, (autoSize(type))?1:0);
	projects[theCurrentProject].patches[patch].objects[n].objectType=type;
	projects[theCurrentProject].patches[patch].objects[n].maxInput=1; /* only for autoSize objects ADD AND OR */
	projects[theCurrentProject].patches[patch].objects[n].dsp=dsp;
	if (type==OBJ_NOTESCRIPT || type==OBJ_TEXT || type==OBJ_HTML) {
		projects[theCurrentProject].patches[patch].objects[n].window=createNoteScriptEditorWindow(theCurrentProject, patch, n, "edit");
	}
	if (type==OBJ_SCOPE) {
		projects[theCurrentProject].patches[patch].objects[n].window=createMultiScopeWindow(&(projects[theCurrentProject].dsp[projects[theCurrentProject].patches[patch].objects[n].dsp]));
	}
	setDspRoutine(theCurrentProject, patch, n);
	setProjectModified(theCurrentProject);
	selectProjectBrowserItem(theCurrentProject, patch, n, -1, keepSelection);

	/* Need to refresh, because refreshObject can't work as it doesn't has it size calcuated */
	mainWindowRefresh=true;

	logprintf("addSynFactoryObject type %d returns %d\\n", (int)type, n);
	return n;
}


static void disconnectCable(int aProject, int aPatch, int aObject, int aPort) {
	logprintf("disconnectCable project %d patch %d object %d port %d\\n", aProject, aPatch, aObject, aPort);
	projects[aProject].patches[aPatch].objects[aObject].connections[aPort].fromObject = 0;
	projects[aProject].dsp[projects[aProject].patches[aPatch].objects[aObject].dsp].io[aPort].input=&(projects[aProject].dsp[0].io[0].output);
	setProjectModified(aProject);
	mainWindowRefresh=true;

	if (autoSize(projects[aProject].patches[aPatch].objects[aObject].objectType)) {
		adjustAutoSize(aProject, aPatch, aObject);
	} else {
		if (thePlayingProject == aProject) {
			~cc~recalcDsp~
		}
	}
}

// Connect from currently active output to specified object and input.
static void connectCable(int aProject, int aPatch, int aObject, int aPort, int aColor) {
	int i;

	logprintf("connectCable project %d patch %d object %d port %d\\n", aProject, aPatch, aObject, aPort);
	for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
		if (projects[aProject].patches[aPatch].objects[i].currentOutput) {
			logprintf("Found output %d, %d\\n", i, projects[aProject].patches[aPatch].objects[i].currentOutput);
			projects[aProject].patches[aPatch].objects[aObject].connections[aPort].fromObject = i+1;
			projects[aProject].patches[aPatch].objects[aObject].connections[aPort].fromOutput = projects[aProject].patches[aPatch].objects[i].currentOutput;
			projects[aProject].patches[aPatch].objects[aObject].connections[aPort].cableColor = aColor;
			projects[aProject].dsp[projects[aProject].patches[aPatch].objects[aObject].dsp].io[aPort].input=&(projects[theCurrentProject].dsp[projects[aProject].patches[aPatch].objects[i].dsp].io[projects[aProject].patches[aPatch].objects[i].currentOutput-1].output);
			setProjectModified(aProject);
			mainWindowRefresh=true;
			if (autoSize(projects[aProject].patches[aPatch].objects[aObject].objectType)) {
				adjustAutoSize(aProject, aPatch, aObject);
			} else {
				if (thePlayingProject == aProject) {
					~cc~recalcDsp~
				}
			}
			break;
		}
	}
}

// Toggle visability of the extra label for all selected objects
static void toggleLabel(void) {
	if (theCurrentProject>=0 && currentPatch>=0) {
		for(int i=0;i<projects[theCurrentProject].patches[currentPatch].maxObjects;i++) {
			if (projects[theCurrentProject].patches[currentPatch].objects[i].dsp && projects[theCurrentProject].patches[currentPatch].objects[i].selected) {
				projects[theCurrentProject].patches[currentPatch].objects[i].showLabel=!projects[theCurrentProject].patches[currentPatch].objects[i].showLabel;
				setProjectModified(theCurrentProject);
				mainWindowRefresh=true;
			}
		}
	}
}

// delete all selected objects or everything when deleteAll is true
static void deleteSynFactoryObjects(int aProject, int aPatch, bool deleteAll) {
	int i,j;
	int s=0;
	int d=0;
	
	while (s<projects[aProject].patches[aPatch].maxObjects) {
		if (projects[aProject].patches[aPatch].objects[s].selected || deleteAll) {
			logprintf("delete object from project %d patch %d object %d (type %d)\\n", aProject, aPatch, s, (int)projects[aProject].patches[aPatch].objects[s].objectType);
			// Remove all connections from this object output to others (his inputs are not relevant)
			for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
				for(j=0;j<maxObjectIo;j++) {
					if (projects[aProject].patches[aPatch].objects[i].connections[j].fromObject == s+1) {
						disconnectCable(aProject, aPatch, i, j);
					}
				}
			}
			if (projects[aProject].patches[aPatch].objects[s].window) {
				guiDestroyWindow(projects[aProject].patches[aPatch].objects[s].window);
			}
			if (projects[aProject].patches[aPatch].objects[s].name) {
				free(projects[aProject].patches[aPatch].objects[s].name);
			}
			if (projects[aProject].patches[aPatch].objects[s].dsp) {
				lockMutex(theAudioMutex);
				projects[aProject].dsp[projects[aProject].patches[aPatch].objects[s].dsp].dspRoutine=dspRoutine_Dummy;
				projects[aProject].dsp[projects[aProject].patches[aPatch].objects[s].dsp].inUse=false;
				if (projects[aProject].patches[aPatch].objects[s].objectType==OBJ_NOTESCRIPT && projects[aProject].dsp[projects[aProject].patches[aPatch].objects[s].dsp].scriptInfo->script) {
					free(projects[aProject].dsp[projects[aProject].patches[aPatch].objects[s].dsp].scriptInfo->script);
				}
				if (projects[aProject].dsp[projects[aProject].patches[aPatch].objects[s].dsp].buffer) {
					free(projects[aProject].dsp[projects[aProject].patches[aPatch].objects[s].dsp].buffer);
				}
				if (thePlayingProject == aProject) {
					~cc~recalcDsp~
				}
				unlockMutex(theAudioMutex);
			}
		} else {
			if (d!=s) {
				projects[aProject].patches[aPatch].objects[d]=projects[aProject].patches[aPatch].objects[s];
				// Rewire all objects inputs to new index of object
				for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
					for(j=0;j<maxObjectIo;j++) {
						if (projects[aProject].patches[aPatch].objects[i].connections[j].fromObject == s+1) {
							projects[aProject].patches[aPatch].objects[i].connections[j].fromObject = d+1;
						}
					}
				}
			}
			d++;
		}
		s++;
	}
	if (d != projects[aProject].patches[aPatch].maxObjects) {
		projects[aProject].patches[aPatch].maxObjects=d;
		projects[aProject].patches[aPatch].objects=(SynFactoryObjectPtr)realloc(projects[aProject].patches[aPatch].objects, projects[aProject].patches[aPatch].maxObjects*sizeof(SynFactoryObject));
		setProjectModified(aProject);
		projectBrowserWindowRefresh=true;
		mainWindowRefresh=true;
	}
}

/*static void moveOneUpSynFactoryObject(SynFactoryPatch *patch, int objectIndex) {
	SynFactoryObject tmp;
	
	if (objectIndex>patch->maxObjects-1) {
		tmp=patch->objects[objectIndex];
		patch->objects[objectIndex]=patch->objects[objectIndex+1];
		patch->objects[objectIndex]=tmp;
	}  
}*/

static int createSynfactoryPatch(void) {
	char tmpName[256];
	int n=-1;

	logprintf("createSynfactoryPatch\\n");

	if (projects==NULL) {
		createProject(newProjectName);
	}

	if (thePlayingProject==theCurrentProject) {
		lockMutex(theAudioMutex);
		lockMutex(midiMutex);
	}

	n=projects[theCurrentProject].maxPatches;
	projects[theCurrentProject].maxPatches++;
	projects[theCurrentProject].patches=(SynFactoryPatchPtr)realloc(projects[theCurrentProject].patches, projects[theCurrentProject].maxPatches*sizeof(SynFactoryPatch));
	memset(&(projects[theCurrentProject].patches[n]), 0, sizeof(SynFactoryPatch));
	sprintf(tmpName, "patch %d", ++projects[theCurrentProject].patchCount);
	projects[theCurrentProject].patches[n].patchName=strdup(tmpName);
	if (thePlayingProject==theCurrentProject) {
		unlockMutex(midiMutex);
		unlockMutex(theAudioMutex);
	}
	currentPatch=n;
	setProjectModified(theCurrentProject);
	patchConfigWindowRefresh=true;
	mainWindowRefresh=true;

	logprintf("createSynfactoryPatch returns %d\\n", n);
	return n;
}

typedef enum _MouseMode {
	MOUSE_NONE,
	MOUSE_DRAGSCREEN,          // Right mouse button with movement / Popup menu when just click
	MOUSE_DRAGSCREENZOOM,      // Drag screen and press left mouse button
	MOUSE_DRAGOBJECT,          // Left mouse button with movement.
	MOUSE_DRAGOBJECTANDSCREEN, // While drag of object press right mousebutton to drag screen
	MOUSE_DRAGCABLE,           // Left mouse button with drawing cable to other object.
	MOUSE_DRAGCABLEANDSCREEN,  // While drag of cable press right mousebutton to drag screen
	MOUSE_DRAGRECT,            // Paint selection rectangle
	MOUSE_DRAGRECTANDSCREEN,   // Paint selection rectangle and drag screen with right mouse button

	MOUSE_KNOB,
	MOUSE_KNOBSLOW,

	MOUSE_RESIZE_TOP_LEFT,
	MOUSE_RESIZE_TOP_RIGHT,
	MOUSE_RESIZE_BOTTOM_LEFT,
	MOUSE_RESIZE_BOTTOM_RIGHT,

	MOUSE_EDIT_LABEL,
//  MOUSE_
} MouseMode;
static MouseMode mouseMode = MOUSE_NONE;
static int startX; // first location of drawing operation (second location is current mouse pos)
static int startY; // first location of drawing operation (second location is current mouse pos)
static int oldMouseX;
static int oldMouseY;
static int oldMouseClientX;
static int oldMouseClientY;
static int mouseDownTicks=0;
static int redrawDelay;
static bool XorFlag;
static int editObject;
static int knobIndex;
static void storeMouse(ContextPtr aContext) {
	oldMouseX = aContext->mouseX;
	oldMouseY = aContext->mouseY;
	oldMouseClientX = aContext->mouseClientX;
	oldMouseClientY = aContext->mouseClientY;
}

static void refreshObject(SynFactoryPatchPtr aPatch, SynFactoryObject *aObject) {
	if (theCurrentProject>=0 && currentPatch>=0 && aPatch && projects[theCurrentProject].patches+currentPatch==aPatch && aObject) {
		switch(aObject->objectType) {
		case OBJ_TEXT:
			guiRefreshWindow(mainWindow, aObject->x-(aPatch->offsX+7), aObject->y-(aPatch->offsY+7), aObject->x+aObject->xs+8-aPatch->offsX, aObject->y+aObject->ys+8-aPatch->offsY);
			break;
		default:
			guiRefreshWindow(mainWindow, aObject->x-aPatch->offsX, aObject->y-aPatch->offsY, aObject->x+aObject->xs-aPatch->offsX, aObject->y+aObject->ys-aPatch->offsY);
			break;
		}
	}
}

static bool dragScreen(ContextPtr aContext, SynFactoryPatchPtr patch) {
	int deltaX = aContext->mouseX - oldMouseX;
	int deltaY = aContext->mouseY - oldMouseY;

	if (mouseMode==MOUSE_DRAGSCREENZOOM) {
		patch->offsX -= deltaX*5;
		patch->offsY -= deltaY*5;
	} else {
		patch->offsX -= deltaX;
		patch->offsY -= deltaY;
	}

	storeMouse(aContext);
	return (0!=deltaX || 0!=deltaY);
}

static void updateDragScreen(ContextPtr aContext, SynFactoryPatchPtr patch) {
	mouseDownTicks++;
	if (dragScreen(aContext, patch)) {
		redrawDelay = 4;
		mainWindowRefresh=true;
	} else {
		// Repaint window with all details if mouse hasn't moved some time.
		if (redrawDelay > 0) {
			redrawDelay--;
			if (redrawDelay == 0) {
				mainWindowRefresh=true;
			}
		}
	}
}

static void selectOutput(SynFactoryPatchPtr patch, int object, int output) {
	int i;
	
	// reset all other outputs
	for(i=0;i<patch->maxObjects;i++) {
		if (patch->objects[i].currentOutput) {
			refreshObject(patch, &patch->objects[i]);
			patch->objects[i].currentOutput=0;
		}
	}

	patch->objects[object].currentOutput=output;
	refreshObject(patch, &patch->objects[object]);
}

static void selectObjectsInArea(SynFactoryPatchPtr patch, int x1, int y1, int x2, int y2) {
	int i;
	bool flag=ctrlPressed();

	for(i=patch->maxObjects-1;i>=0;i--) {
		SynFactoryObject *curObj=&(patch->objects[i]);
		if (x1<curObj->x && x2>curObj->x+curObj->xs
		&& y1<curObj->y && y2>curObj->y+curObj->ys) {
			// !!!TODO: remove theCurrentProject, currentPatch
			selectProjectBrowserItem(theCurrentProject, currentPatch, i, -1, flag);
			flag=true;
		}
	}
}

static int findObject(int aProject, int aPatch, int x, int y, MouseMode *newMouseMode) {
	int i;

	for(i=projects[aProject].patches[aPatch].maxObjects-1;i>=0;i--) {
		SynFactoryObject *curObj=&(projects[aProject].patches[aPatch].objects[i]);
		if (curObj->dsp>0 || showResizeMarkers) {
			if (MOUSE_NONE==mouseMode && curObj->showLabel
			&& x>=curObj->x && x<curObj->x+curObj->labelXs
			&& y>=curObj->y-18 && y<curObj->y-2) {
				*newMouseMode=MOUSE_EDIT_LABEL;
				return i;
			}
			if (x>=curObj->x && x<curObj->x+curObj->xs
			&& y>=curObj->y && y<curObj->y+curObj->ys) {
				if (y < curObj->y+2*ioYSpacing) {
					*newMouseMode = MOUSE_DRAGOBJECT;
				} else {
					int j;

					for(j=0;j<maxObjectIo;j++) {
						// inputs
						if ((x >= curObj->x) &&
								(x <= curObj->x+ioXSize) &&
								(y >= curObj->y+(j+2)*ioYSpacing) &&
								(y <= curObj->y+(j+2)*ioYSpacing+ioYSize) &&
								inputName(curObj->objectType, j)) {
							if ((curObj->connections[j].fromObject==0)
									|| (MOUSE_DRAGCABLE==mouseMode)) {
								connectCable(aProject, aPatch, i, j, theCurrentCableColorIndex);
							} else {
								disconnectCable(aProject, aPatch, i, j);
							}
						}
						// outputs
						if ((x >= curObj->x+curObj->xs-ioXSize) &&
								(x <= curObj->x+curObj->xs) &&
								(y >= curObj->y+(j+2)*ioYSpacing) &&
								(y <= curObj->y+(j+2)*ioYSpacing+ioYSize) &&
								outputName(curObj->objectType, j)) {
							selectOutput(projects[aProject].patches + aPatch, i, j+1);
							*newMouseMode = MOUSE_DRAGCABLE;
							break;
						}
						// knobs
						if ((x >= curObj->x + (j>>1)*knobXSpacing + knobXOffset) &&
								(x <= curObj->x + (j>>1)*knobXSpacing + knobXOffset +30) &&
								(y >= curObj->y + ioYSpacing*(2 + ((j&1)?4:0))) &&
								(y <= curObj->y + ioYSpacing*(2 + ((j&1)?4:0)) + 30) &&
								knobName(curObj->objectType, j)) {
							editObject=i;
							knobIndex=j;
							*newMouseMode = MOUSE_KNOB;
							break;
						}
					}
				}

				// OSC specific actions
				if (curObj->objectType == OBJ_OSC) {
					if ((x >= curObj->x +  50) &&
							(x <  curObj->x +  70) &&
							(y >= curObj->y + ioYSpacing*6 + 20) &&
							(y <  curObj->y + ioYSpacing*6 + 40)) {
						setMode(aProject, aPatch,i, 0 + (curObj->mode&56));
					}
					if ((x >= curObj->x +  70) &&
							(x <  curObj->x +  90) &&
							(y >= curObj->y + ioYSpacing*6 + 20) &&
							(y <  curObj->y + ioYSpacing*6 + 40)) {
						setMode(aProject, aPatch,i, 1 + (curObj->mode&56));
					}
					if ((x >= curObj->x +  90) &&
							(x <  curObj->x + 110) &&
							(y >= curObj->y + ioYSpacing*6 + 20) &&
							(y <  curObj->y + ioYSpacing*6 + 40)) {
						setMode(aProject, aPatch,i, 2 + (curObj->mode&56));
					}
					if ((x >= curObj->x + 110) &&
							(x <  curObj->x + 130) &&
							(y >= curObj->y + ioYSpacing*6 + 20) &&
							(y <  curObj->y + ioYSpacing*6 + 40)) {
						setMode(aProject, aPatch,i, 3 + (curObj->mode&56));
					}
					if ((x >= curObj->x + 130) &&
							(x <  curObj->x + 150) &&
							(y >= curObj->y + ioYSpacing*6 + 20) &&
							(y <  curObj->y + ioYSpacing*6 + 40)) {
						setMode(aProject, aPatch,i, 4 + (curObj->mode&56));
					}
					if ((x >= curObj->x + 150) &&
							(x <  curObj->x + 170) &&
							(y >= curObj->y + ioYSpacing*6 + 20) &&
							(y <  curObj->y + ioYSpacing*6 + 40)) {
						setMode(aProject, aPatch,i, 5 + (curObj->mode&56));
					}
					if ((x >= curObj->x +  90) &&
							(x <  curObj->x +  110) &&
							(y >= curObj->y + ioYSpacing*6 -  3) &&
							(y <  curObj->y + ioYSpacing*6 + 17)) {
						setMode(aProject, aPatch,i, curObj->mode^32);
					}
					if ((x >= curObj->x + 110) &&
							(x <  curObj->x + 130) &&
							(y >= curObj->y + ioYSpacing*6 -  3) &&
							(y <  curObj->y + ioYSpacing*6 + 17)) {
						setMode(aProject, aPatch,i, (curObj->mode&39));
					}
					if ((x >= curObj->x + 130) &&
							(x <  curObj->x + 150) &&
							(y >= curObj->y + ioYSpacing*6 -  3) &&
							(y <  curObj->y + ioYSpacing*6 + 17)) {
						setMode(aProject, aPatch,i, (curObj->mode&39) + 8);
					}
					if ((x >= curObj->x + 150) &&
							(x <  curObj->x + 170) &&
							(y >= curObj->y + ioYSpacing*6 -  3) &&
							(y <  curObj->y + ioYSpacing*6 + 17)) {
						setMode(aProject, aPatch,i, (curObj->mode&39) + 16);
					}
				}
				if (curObj->objectType == OBJ_PERLINOSC || curObj->objectType == OBJ_PMOSC) {
					if ((x >= curObj->x + 110) &&
							(x <  curObj->x + 130) &&
							(y >= curObj->y + ioYSpacing*6 -  3) &&
							(y <  curObj->y + ioYSpacing*6 + 17)) {
						setMode(aProject, aPatch,i, (curObj->mode&7));
					}
					if ((x >= curObj->x + 130) &&
							(x <  curObj->x + 150) &&
							(y >= curObj->y + ioYSpacing*6 -  3) &&
							(y <  curObj->y + ioYSpacing*6 + 17)) {
						setMode(aProject, aPatch,i, (curObj->mode&7) + 8);
					}
					if ((x >= curObj->x + 150) &&
							(x <  curObj->x + 170) &&
							(y >= curObj->y + ioYSpacing*6 -  3) &&
							(y <  curObj->y + ioYSpacing*6 + 17)) {
						setMode(aProject, aPatch,i, (curObj->mode&7) + 16);
					}
				}

				// DRUM specific actions
				if (curObj->objectType == OBJ_DRM4 || curObj->objectType == OBJ_DRM8) {
					int col = (x + 10 - (curObj->x+knobXOffset)) / 16;
					int row = (y - (curObj->y+ioYSpacing*2)) / (ioYSpacing*2);

					if (col>=0 && col<16 && (y >= curObj->y+ioYSpacing*2) && row<((curObj->objectType == OBJ_DRM4)?4:8)) {
						((unsigned char *)projects[theCurrentProject].dsp[curObj->dsp].buffer)[row*16+col]^=1;
						refreshObject(projects[aProject].patches + aPatch, curObj);
					}
				}

				// take first object found
				return i;
			} else  if (curObj->dsp==0) {
				// Resize icons
				editObject=i;
				if (x>curObj->x-6 && x<curObj->x && y>curObj->y-6 && y<curObj->y) {
					*newMouseMode = MOUSE_RESIZE_TOP_LEFT;
					return i;
				}
				if (x>curObj->x+curObj->xs && x<curObj->x+curObj->xs+6 && y>curObj->y-6 && y<curObj->y) {
					*newMouseMode = MOUSE_RESIZE_TOP_RIGHT;
					return i;
				}
				if (x>curObj->x-6 && x<curObj->x && y>curObj->y+curObj->ys && y<curObj->y+curObj->ys+6) {
					*newMouseMode = MOUSE_RESIZE_BOTTOM_LEFT;
					return i;
				}
				if (x>curObj->x+curObj->xs && x<curObj->x+curObj->xs+6 && y>curObj->y+curObj->ys && y<curObj->y+curObj->ys+6) {
					*newMouseMode = MOUSE_RESIZE_BOTTOM_RIGHT;
					return i;
				}
			}
		}
	}
	return -1;
}

static void synfactoryLeftClick(ContextPtr aContext, int aProject, int aPatch) {
	SynFactoryPatchPtr patch = projects[aProject].patches + aPatch;
	int obj=findObject(aProject, aPatch, aContext->mouseClientX + patch->offsX, aContext->mouseClientY + patch->offsY, &mouseMode);
	if (obj<0) {
		// mouse pressed on background activate select rectangle
//    storeMouse(aContext);
//    SetCapture(aContext->currentWindow);
		startX=aContext->mouseClientX+projects[aProject].patches[aPatch].offsX;
		startY=aContext->mouseClientY+projects[aProject].patches[aPatch].offsY;
		mouseMode=MOUSE_DRAGRECT;
	} else {
		selectProjectBrowserItem(aProject, aPatch, obj, -1, ctrlPressed());
	}

	if (MOUSE_EDIT_LABEL == mouseMode) {
		 mouseMode = MOUSE_NONE;
	}
	if (MOUSE_NONE != mouseMode) {
		storeMouse(aContext);
		SetCapture(aContext->currentWindow);
		if (MOUSE_KNOB == mouseMode) {
			guiHideMouse(aContext);
//      oldMouseX=aContext->mouseX;
//      oldMouseY=aContext->mouseY;
			SetCursorPos(GetSystemMetrics(SM_CXFULLSCREEN)/2, GetSystemMetrics(SM_CYFULLSCREEN)/2);
		}
	}
}

static void processPatchDoubleClick(ContextPtr aContext, int aProject, int aPatch) {
	MouseMode dummyMouseMode=MOUSE_NONE;
	int obj=findObject(aProject, aPatch, aContext->mouseClientX + projects[aProject].patches[aPatch].offsX, aContext->mouseClientY + projects[aProject].patches[aPatch].offsY, &dummyMouseMode);
//  mouseMode=MOUSE_NONE;

	if (obj<0) {
		// mouse pressed on background activate select rectangle
//    storeMouse(aContext);
//    SetCapture(aContext->currentWindow);
		startX=aContext->mouseClientX + projects[aProject].patches[aPatch].offsX;
		startY=aContext->mouseClientY + projects[aProject].patches[aPatch].offsY;
		mouseMode=MOUSE_DRAGRECT;
	} else {
		if (projects[aProject].patches[aPatch].objects[obj].window) {
			guiShowWindow(projects[aProject].patches[aPatch].objects[obj].window);
		}
	}
}

static bool dragObjects(ContextPtr aContext, SynFactoryPatchPtr patch) {
	int deltaX = aContext->mouseX - oldMouseX;
	int deltaY = aContext->mouseY - oldMouseY;

	if (0!=deltaX
	|| 0!=deltaY) {
		int i;

		for(i=0;i<patch->maxObjects;i++) {
			if (patch->objects[i].selected) {
				patch->objects[i].x += deltaX;
				patch->objects[i].y += deltaY;
				mainWindowRefresh=true;
			}
		}
	}

	oldMouseX = aContext->mouseX;
	oldMouseY = aContext->mouseY;
	return (0!=deltaX || 0!=deltaY);
}

static void updateDragObjects(ContextPtr aContext, SynFactoryPatchPtr patch) {
	dragObjects(aContext, patch);
}

static void finishDragObjects(ContextPtr aContext, SynFactoryPatchPtr patch) {
	dragObjects(aContext, patch);
	ReleaseCapture();
	mouseMode = MOUSE_NONE;  
}

static void finishDragCable(ContextPtr aContext, int aProject, int aPatch) {
	ReleaseCapture();
	findObject(aProject, aPatch, aContext->mouseClientX + projects[aProject].patches[aPatch].offsX, aContext->mouseClientY + projects[aProject].patches[aPatch].offsY, &mouseMode);
	mouseMode = MOUSE_NONE;  

	// !!!FIXME This can be removed, when current XOR cable is removed in other way.
	// connect/disconnect cable already refreshes if necessary
	mainWindowRefresh=true;
}

static void updateModuleLabel(SynFactoryPatchPtr patch, int obj, int chr) {
	if (chr==8) {
		if (patch->objects[obj].name) {
			char *oldName=patch->objects[obj].name;
			int len=strlen(oldName);
			if (len<2) {
				patch->objects[obj].name=NULL;
			} else {
				patch->objects[obj].name=(char *)malloc(len);
				memcpy(patch->objects[obj].name, oldName, len);
				patch->objects[obj].name[len-1]='\\0';
			}
			free(oldName);
		}
	}
	if (chr>=32) {
		if (patch->objects[obj].name) {
			char *oldName=patch->objects[obj].name;
			int len=strlen(oldName);
			patch->objects[obj].name=(char *)malloc(len+2);
			strcpy(patch->objects[obj].name, oldName);
			patch->objects[obj].name[len]=(char)chr;
			patch->objects[obj].name[len+1]='\\0';
			free(oldName);
		} else {
			patch->objects[obj].name=(char *)malloc(2);
			patch->objects[obj].name[0]=(char)chr;
			patch->objects[obj].name[1]='\\0';
		}
	}

	patch->objects[obj].labelXs=0;
	mainWindowRefresh=true; // !!!FIXME shouldn't be necessary to refresh complete screen
}

static void updateKnob(ContextPtr aContext, SynFactoryPatchPtr patch) {
	int incr;
	int temp;

	// Only redraw on vertical movement
	if (aContext->mouseY != GetSystemMetrics(SM_CYFULLSCREEN)/2) {
		incr = GetSystemMetrics(SM_CYFULLSCREEN)/2 - aContext->mouseY;
		if (MOUSE_KNOBSLOW == mouseMode) {
			incr = incr / 4;
		} else {
			incr = incr * 16;
		}

		temp = projects[theCurrentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob + incr;
		if (temp > 32767) temp = 32767;
		if (temp < -32768) temp = -32768;
		projects[theCurrentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob = temp;
		setProjectModified(theCurrentProject);
		SetCursorPos(GetSystemMetrics(SM_CXFULLSCREEN)/2, GetSystemMetrics(SM_CYFULLSCREEN)/2);

		refreshObject(patch, &patch->objects[editObject]);
	}
}

static void resizeObject(ContextPtr aContext, SynFactoryPatchPtr patch) {
	int deltaX = aContext->mouseX - oldMouseX;
	int deltaY = aContext->mouseY - oldMouseY;

	if (0!=deltaX
	|| 0!=deltaY) {
		switch(mouseMode) {
		case MOUSE_RESIZE_TOP_LEFT:
			patch->objects[editObject].x+=deltaX;
			patch->objects[editObject].xs-=deltaX;
			patch->objects[editObject].y+=deltaY;
			patch->objects[editObject].ys-=deltaY;
			if (patch->objects[editObject].xs<0) {
				patch->objects[editObject].xs=0;
			}
			if (patch->objects[editObject].ys<0) {
				patch->objects[editObject].ys=0;
			}
			break;
		case MOUSE_RESIZE_TOP_RIGHT:
			patch->objects[editObject].xs+=deltaX;
			patch->objects[editObject].y+=deltaY;
			patch->objects[editObject].ys-=deltaY;
			if (patch->objects[editObject].xs<0) {
				patch->objects[editObject].x+=patch->objects[editObject].xs;
				patch->objects[editObject].xs=0;
			}
			if (patch->objects[editObject].ys<0) {
				patch->objects[editObject].ys=0;
			}
			break;
		case MOUSE_RESIZE_BOTTOM_LEFT:
			patch->objects[editObject].x+=deltaX;
			patch->objects[editObject].xs-=deltaX;
			patch->objects[editObject].ys+=deltaY;
			if (patch->objects[editObject].xs<0) {
				patch->objects[editObject].xs=0;
			}
			if (patch->objects[editObject].ys<0) {
				patch->objects[editObject].y+=patch->objects[editObject].ys;
				patch->objects[editObject].ys=0;
			}
			break;
		case MOUSE_RESIZE_BOTTOM_RIGHT:
			patch->objects[editObject].xs+=deltaX;
			patch->objects[editObject].ys+=deltaY;
			if (patch->objects[editObject].xs<0) {
				patch->objects[editObject].x+=patch->objects[editObject].xs;
				patch->objects[editObject].xs=0;
			}
			if (patch->objects[editObject].ys<0) {
				patch->objects[editObject].y+=patch->objects[editObject].ys;
				patch->objects[editObject].ys=0;
			}
			break;
		default:
			break;
		}
		mainWindowRefresh=true;
	}

	oldMouseX = aContext->mouseX;
	oldMouseY = aContext->mouseY; 
}

static void drawObjectsZoomed(ContextPtr aContext, int aProject, int aPatch) {
	int i,j;
	int offsX=(aContext->guiClientRect.right-aContext->guiClientRect.left)/5*2-projects[aProject].patches[aPatch].offsX/5;
	int offsY=(aContext->guiClientRect.bottom-aContext->guiClientRect.top)/5*2-projects[aProject].patches[aPatch].offsY/5;
	

	for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
		SynFactoryObject *curObj=projects[aProject].patches[aPatch].objects + i;
		guiSelectPen1Color(aContext, 0);
		guiSelectFillColor(aContext, thePatchColorForModules);
		guiDrawRoundRect(aContext, (curObj->x)/5+offsX, (curObj->y)/5+offsY, (curObj->x+curObj->xs)/5+offsX, (curObj->y+curObj->ys)/5+offsY, 4, 4);
		guiSelectFillColor(aContext, thePatchColorForKnobs);
		for(j=0;j<maxObjectIo;j++) {
			if (knobName(curObj->objectType, j)) {
				guiDrawEllipse(aContext, (curObj->x+(j>>1)*knobXSpacing+knobXOffset)/5+offsX, (curObj->y+ioYSpacing*(2+(j&1)*4))/5+offsY, (curObj->x+(j>>1)*knobXSpacing+knobXOffset)/5+offsX+6, (curObj->y+ioYSpacing*(2+(j&1)*4))/5+offsY+6);
			}
		}
	}        

	guiSelectFillColor(aContext, -1);
	guiSelectPen1Color(aContext, 15);
	guiDrawRect(aContext,
		(aContext->guiClientRect.right-aContext->guiClientRect.left)/5*2,
		(aContext->guiClientRect.bottom-aContext->guiClientRect.top)/5*2,
		(aContext->guiClientRect.right-aContext->guiClientRect.left)/5*3,
		(aContext->guiClientRect.bottom-aContext->guiClientRect.top)/5*3);
}

static void drawResizeMarkers(ContextPtr aContext, int left, int top, int right, int bottom, bool selected) {
	guiSelectPen3Color(aContext, (selected)?thePatchColorForHighlight:thePatchColorForShadow);
	guiSelectFillColor(aContext, (selected)?thePatchColorForShadow:thePatchColorForModules);
	guiDrawRect(aContext, left-6, top-6, left, top);
	guiDrawRect(aContext, right, top-6, right+6, top);
	guiDrawRect(aContext, left-6, bottom, left, bottom+6);
	guiDrawRect(aContext, right, bottom, right+6, bottom+6);
}

static void drawObject(ContextPtr aContext, int aProject, int aPatch, int aObject) {
	int i;
	SynFactoryObjectPtr object = projects[aProject].patches[aPatch].objects + aObject;

	// Check if object is completely off screen. Only draw modules
	// that are partly or completely visible.
	if (object->x+object->xs-projects[aProject].patches[aPatch].offsX<aContext->guiClientRect.left
	|| object->x-projects[aProject].patches[aPatch].offsX>aContext->guiClientRect.right
	|| object->y+object->ys-projects[aProject].patches[aPatch].offsY<aContext->guiClientRect.top
	|| object->y-projects[aProject].patches[aPatch].offsY>aContext->guiClientRect.bottom) {
		return;
	}

	if (object->dsp>0) {
		if (object->labelXs<48) {
			if (object->name) {
				int xs;
				guiSelectPotFont(aContext);
				guiCalcTextSize(aContext, &xs, NULL, object->name, strlen(object->name));
//				DrawText(aContext->hdc, object->name, -1, &myRect, DT_CALCRECT);
				object->labelXs=xs+12;  
			}
			if (object->labelXs<48) {
				object->labelXs=48;
			}
		}

		guiSelectPen1Color(aContext, thePatchColorForShadow);
		if (0 == redrawDelay) {
			guiSelectFillColor(aContext, thePatchColorForShadow);
			guiDrawRoundRect(aContext, object->x+2-projects[aProject].patches[aPatch].offsX, object->y+2-projects[aProject].patches[aPatch].offsY, object->x+object->xs+2-projects[aProject].patches[aPatch].offsX, object->y+object->ys+2-projects[aProject].patches[aPatch].offsY, 12, 12);
			if (object->showLabel) {
				guiDrawRoundRect(aContext, object->x+2-projects[aProject].patches[aPatch].offsX, -18+object->y-projects[aProject].patches[aPatch].offsY, object->x+object->labelXs+2-projects[aProject].patches[aPatch].offsX, -2+object->y-projects[aProject].patches[aPatch].offsY, 12, 12);
			}
		}

		guiSelectFillColor(aContext, thePatchColorForModules);
		guiDrawRoundRect(aContext, object->x-projects[aProject].patches[aPatch].offsX,object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, object->y+object->ys-projects[aProject].patches[aPatch].offsY, 12, 12);
		if (object->showLabel) {
			guiDrawRoundRect(aContext, object->x-projects[aProject].patches[aPatch].offsX, -20+object->y-projects[aProject].patches[aPatch].offsY, object->x+object->labelXs-projects[aProject].patches[aPatch].offsX, -4+object->y-projects[aProject].patches[aPatch].offsY, 12, 12);
			if (object->name) {
				guiSetTextAlignment(aContext, alignTopLeft);
				guiSelectPen1Color(aContext, thePatchColorForShadow);
				guiSelectPotFont(aContext);
				guiDrawTransparent(aContext);
				guiDrawText(aContext, object->x+6-projects[aProject].patches[aPatch].offsX, -18+object->y-projects[aProject].patches[aPatch].offsY, object->name, strlen(object->name));
			}

			// Draw connection between label and module
			int myX=object->x+min(object->labelXs/2,object->xs/2)-projects[aProject].patches[aPatch].offsX;
			int myY=object->y-projects[aProject].patches[aPatch].offsY;
			guiSelectPen1Color(aContext, -1);
			guiDrawRect(aContext, myX-4, myY-5, myX+4, myY+1);
			guiSelectPen1Color(aContext, thePatchColorForShadow);
			guiDrawLine(aContext, myX-5, myY-4, myX-5, myY);
			guiDrawLine(aContext, myX+4, myY-4, myX+4, myY);
		}

		if (objectTitles[object->objectType]) {
			if (object->selected) {
				guiSelectFillColor(aContext, thePatchColorForHighlight);
				guiDrawRoundRect(aContext, object->x-projects[aProject].patches[aPatch].offsX,object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, (object->y-projects[aProject].patches[aPatch].offsY) + 2*ioYSpacing-ioYSize, 12, 12);
			}
			guiDrawTransparent(aContext);
			guiSelectTitleFont(aContext);
			guiSetTextAlignment(aContext, alignTopLeft);
			guiDrawText(aContext, (object->x-projects[aProject].patches[aPatch].offsX)+4, (object->y-projects[aProject].patches[aPatch].offsY)+2, objectTitles[object->objectType], strlen(objectTitles[object->objectType]));
		}

		// Inputs
		guiSelectIOFont(aContext);
		guiSetTextAlignment(aContext, alignLineLeft);
		for(i=0;i<maxObjectIo;i++) {
			if (inputName(object->objectType, i) && (autoSize(object->objectType)==false || object->maxInput>=i)) {
				guiSelectFillColor(aContext, -1);
				guiSelectPen1Color(aContext, thePatchColorForShadow);
				guiDrawRect(aContext,
					(object->x-projects[aProject].patches[aPatch].offsX),
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing,
					(object->x-projects[aProject].patches[aPatch].offsX)+ioXSize,
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing+ioYSize);

				guiDrawText(aContext,
					(object->x-projects[aProject].patches[aPatch].offsX)+ioXSize+3,
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing,
					(object->x-projects[aProject].patches[aPatch].offsX)+ioXSize+64,
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing+ioYSize,
					inputName(object->objectType, i));
			}
		}

		// Outputs
		guiSetTextAlignment(aContext, alignLineRight);
		for(i=0;i<maxObjectIo;i++) {
			if (outputName(object->objectType, i)) {
				if (object->currentOutput == i+1) {
					guiSelectFillColor(aContext, thePatchColorForShadow);
				} else {
					guiSelectFillColor(aContext, -1);
				}
				guiSelectPen1Color(aContext, thePatchColorForShadow);
				guiDrawRect(aContext,
					(object->x-projects[aProject].patches[aPatch].offsX)+object->xs-ioXSize,
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing,
					(object->x-projects[aProject].patches[aPatch].offsX)+object->xs,
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing+ioYSize);
				guiDrawText(aContext,
					(object->x+object->xs-projects[aProject].patches[aPatch].offsX)-(ioXSize+64),
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing,
					(object->x+object->xs-projects[aProject].patches[aPatch].offsX)-(ioXSize+3),
					(object->y-projects[aProject].patches[aPatch].offsY)+(i+2)*ioYSpacing+ioYSize,
					outputName(object->objectType, i));
			}
		}

		// Draw the knobs
		for(i=0;i<maxObjectIo;i++) {
			if (knobName(object->objectType, i)) {
				drawPot(aContext, (object->x-projects[aProject].patches[aPatch].offsX)+(i>>1)*knobXSpacing+knobXOffset, (object->y-projects[aProject].patches[aPatch].offsY)+ioYSpacing*(2+(i&1)*4), knobName(object->objectType, i), projects[theCurrentProject].dsp[object->dsp].io[i].knob, object->mode, thePatchColorForKnobs, thePatchColorForHighlight, thePatchColorForShadow, false, knobTranslateFunction(object->objectType, i), NULL);
			}
		}
	}

	// Object specific drawing stuff
	switch(object->objectType) {
	case OBJ_OSC:
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX +  50, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6+20, (HICON)(((object->mode&7)==0)?ICON_SIN1:ICON_SIN0));
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX +  70, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6+20, (HICON)(((object->mode&7)==1)?ICON_TRI1:ICON_TRI0));
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX +  90, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6+20, (HICON)(((object->mode&7)==2)?ICON_SAW1:ICON_SAW0));
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 110, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6+20, (HICON)(((object->mode&7)==3)?ICON_SPS1:ICON_SPS0));
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 130, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6+20, (HICON)(((object->mode&7)==4)?ICON_PLS1:ICON_PLS0));
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 150, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6+20, (HICON)(((object->mode&7)==5)?ICON_RND1:ICON_RND0));

		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX +  90, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6-3, (HICON)(((object->mode&32)==0)?ICON_AA1:ICON_AA0));
//		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 110, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6-3, (HICON)(((object->mode&24)==0)?ICON_L1:ICON_L0));
//		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 130, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6-3, (HICON)(((object->mode&24)==8)?ICON_M1:ICON_M0));
//		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 150, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6-3, (HICON)(((object->mode&24)==16)?ICON_H1:ICON_H0));
//		break;
	case OBJ_PERLINOSC:
	case OBJ_PMOSC:
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 110, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6-3, (HICON)(((object->mode&24)==0)?ICON_L1:ICON_L0));
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 130, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6-3, (HICON)(((object->mode&24)==8)?ICON_M1:ICON_M0));
		DrawIcon(aContext->hdc, object->x-projects[aProject].patches[aPatch].offsX + 150, object->y-projects[aProject].patches[aPatch].offsY + ioYSpacing*6-3, (HICON)(((object->mode&24)==16)?ICON_H1:ICON_H0));
		break;
	case OBJ_ADD:
		guiSelectTitleFont(aContext);
		guiSelectPen1Color(aContext, thePatchColorForShadow);
		guiSetTextAlignment(aContext, alignBottomCenter);
		guiDrawText(aContext, 40+object->x-projects[aProject].patches[aPatch].offsX, ioYSpacing*3+object->y-projects[aProject].patches[aPatch].offsY, "+", 1);
		break;
	case OBJ_MUL:
		guiSelectTitleFont(aContext);
		guiSelectPen1Color(aContext, thePatchColorForShadow);
		guiSetTextAlignment(aContext, alignBottomCenter);
		guiDrawText(aContext, 40+object->x-projects[aProject].patches[aPatch].offsX, ioYSpacing*3+object->y-projects[aProject].patches[aPatch].offsY, "X", 1);
		break;
	case OBJ_OR:
		guiSelectTitleFont(aContext);
		guiSelectPen1Color(aContext, thePatchColorForShadow);
		guiSetTextAlignment(aContext, alignBottomCenter);
		guiDrawText(aContext, 40+object->x-projects[aProject].patches[aPatch].offsX, ioYSpacing*3+object->y-projects[aProject].patches[aPatch].offsY, ">0", 2);
		break;
	case OBJ_AND:
		guiSelectTitleFont(aContext);
		guiSelectPen1Color(aContext, thePatchColorForShadow);
		guiSetTextAlignment(aContext, alignBottomCenter);
		guiDrawText(aContext, 40+object->x-projects[aProject].patches[aPatch].offsX, ioYSpacing*3+object->y-projects[aProject].patches[aPatch].offsY, "&", 1);
		break;
	case OBJ_DRM4:
	case OBJ_DRM8:
		if (0 == redrawDelay) {
			int col;
			int row;

			for(row=0; row<((object->objectType==OBJ_DRM4)?4:8); row++) {
				for(col=0; col<16; col++) {
					guiSelectFillColor(aContext, (((unsigned char *)projects[theCurrentProject].dsp[object->dsp].buffer)[row * 16 + col])?thePatchColorForShadow:thePatchColorForModules);
					guiDrawRect(aContext,
						(object->x-projects[aProject].patches[aPatch].offsX)+knobXSpacing+col*16,
						(object->y-projects[aProject].patches[aPatch].offsY)+ioYSpacing*2*(row+1),
						(object->x-projects[aProject].patches[aPatch].offsX)+knobXSpacing+col*16+10,
						(object->y-projects[aProject].patches[aPatch].offsY)+ioYSpacing*2*(row+1)+12);
				}
			}
		}
		break;
	case OBJ_TEXT:
		guiSelectPotFont(aContext);
		guiSelectPen1Color(aContext, -1);
		guiSelectFillColor(aContext, object->mainColor);
		guiDrawRect(aContext, object->x-projects[aProject].patches[aPatch].offsX, object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, object->y+object->ys-projects[aProject].patches[aPatch].offsY);
		if (object->name) {
			guiSetTextAlignment(aContext, alignTopLeft);
			guiSelectPen1Color(aContext, object->borderColor);
			guiDrawOpaque(aContext);
			guiDrawText(aContext, object->x-projects[aProject].patches[aPatch].offsX, object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, object->y+object->ys-projects[aProject].patches[aPatch].offsY, object->name, strlen(object->name));
		}
		if (showResizeMarkers) {
			drawResizeMarkers(aContext, object->x-projects[aProject].patches[aPatch].offsX, object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, object->y+object->ys-projects[aProject].patches[aPatch].offsY, object->selected);
		}
		break;
	case OBJ_BOX:
		if (showResizeMarkers) {
			drawResizeMarkers(aContext, object->x-projects[aProject].patches[aPatch].offsX, object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, object->y+object->ys-projects[aProject].patches[aPatch].offsY, object->selected);
		}
		break;
	case OBJ_ROUNDBOX:
		if (showResizeMarkers) {
			drawResizeMarkers(aContext, object->x-projects[aProject].patches[aPatch].offsX, object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, object->y+object->ys-projects[aProject].patches[aPatch].offsY, object->selected);
		}
		break;
	case OBJ_ELLIPSE:
		if (showResizeMarkers) {
			drawResizeMarkers(aContext, object->x-projects[aProject].patches[aPatch].offsX, object->y-projects[aProject].patches[aPatch].offsY, object->x+object->xs-projects[aProject].patches[aPatch].offsX, object->y+object->ys-projects[aProject].patches[aPatch].offsY, object->selected);
		}
		break;
	default:
		break;
	}
}

static void drawCable(ContextPtr aContext, int x1, int y1, int x2, int y2, int color, bool xorMode) {
	if ((1==theCurrentCableType) || (0==theCurrentCableType && xorMode)) {
		// Colored cables
		guiSelectPen3Color(aContext, cablePenColors[color]);
		guiDrawLine(aContext, x1, y1, x2, y2);
		guiSelectPen1Color(aContext, cableFillColors[color]);
		guiDrawLine(aContext, x1, y1, x2, y2);
	} else if (2==theCurrentCableType) {
		// Hanging colored cables
		POINT curve[4];
		curve[0].x = x1;
		curve[0].y = y1;
		curve[3].x = x2;
		curve[3].y = y2;
		curve[1].x = (curve[0].x + curve[3].x) / 2;
		curve[1].y = max(curve[0].y, curve[3].y) + (abs(curve[0].x - curve[3].x) / 8) + (abs(curve[0].y - curve[3].y) / 16);
		curve[2].x = curve[1].x;
		curve[2].y = curve[1].y;

		guiSelectPen3Color(aContext, cablePenColors[color]);
		PolyBezier(aContext->hdc, curve, 4);
		guiSelectPen1Color(aContext, cableFillColors[color]);
		PolyBezier(aContext->hdc, curve, 4);
	} else {
		// Black cables
		guiSelectPen1Color(aContext, 0);
		guiDrawLine(aContext, x1, y1, x2, y2);
	}
}

static void drawCables(ContextPtr aContext, int aProject, int aPatch, int aObject) {
	int i;
	int patchOffsX = projects[aProject].patches[aPatch].offsX;
	int patchOffsY = projects[aProject].patches[aPatch].offsY;

	for(i=0;i<maxObjectIo;i++) {
		if (projects[aProject].patches[aPatch].objects[aObject].connections[i].fromObject) {
			drawCable(aContext,
				-patchOffsX + projects[aProject].patches[aPatch].objects[projects[aProject].patches[aPatch].objects[aObject].connections[i].fromObject-1].x+projects[aProject].patches[aPatch].objects[projects[aProject].patches[aPatch].objects[aObject].connections[i].fromObject-1].xs-ioXSize/2,
				-patchOffsY + projects[aProject].patches[aPatch].objects[projects[aProject].patches[aPatch].objects[aObject].connections[i].fromObject-1].y+(projects[aProject].patches[aPatch].objects[aObject].connections[i].fromOutput+1)*ioYSpacing+ioYSize/2,
				-patchOffsX + projects[aProject].patches[aPatch].objects[aObject].x+ioXSize/2,
				-patchOffsY + projects[aProject].patches[aPatch].objects[aObject].y+(i+2)*ioYSpacing+ioYSize/2,
				projects[aProject].patches[aPatch].objects[aObject].connections[i].cableColor,
				false);
		}
	}
}

static void drawXorCable(ContextPtr aContext, SynFactoryPatchPtr patch) {
	int i;
	int oldROP2=GetROP2(aContext->hdc);

	SetROP2(aContext->hdc, R2_XORPEN);
	for(i=0;i<patch->maxObjects;i++) {
		if (patch->objects[i].currentOutput) {
			if (XorFlag) {
				drawCable(aContext, -patch->offsX+patch->objects[i].x+patch->objects[i].xs-ioXSize/2, -patch->offsY+patch->objects[i].y+(patch->objects[i].currentOutput+1)*ioYSpacing+ioYSize/2, oldMouseClientX, oldMouseClientY, 8, true);
				storeMouse(aContext);
			}
			drawCable(aContext, -patch->offsX+patch->objects[i].x+patch->objects[i].xs-ioXSize/2, -patch->offsY+patch->objects[i].y+(patch->objects[i].currentOutput+1)*ioYSpacing+ioYSize/2, oldMouseClientX, oldMouseClientY, 8, true);
			XorFlag=true;
		}
	}
	SetROP2(aContext->hdc, oldROP2);
}


static void drawXorRect(ContextPtr aContext, SynFactoryPatchPtr patch) {
	int oldROP2=GetROP2(aContext->hdc);

	SetROP2(aContext->hdc, R2_XORPEN);
	guiSelectPen1Color(aContext, 0x888888);
	guiSelectFillColor(aContext, -1);
	guiDrawRect(aContext,
		min(startX-patch->offsX, oldMouseClientX),
		min(startY-patch->offsY, oldMouseClientY),
		max(startX-patch->offsX, oldMouseClientX),
		max(startY-patch->offsY, oldMouseClientY));
	SetROP2(aContext->hdc, oldROP2);
	XorFlag=!XorFlag;
}

static void showXorRect(ContextPtr aContext, SynFactoryPatchPtr patch) {
	if (XorFlag) {
		drawXorRect(aContext, patch);
	}
	storeMouse(aContext);
	drawXorRect(aContext, patch);
}

static void processSynFactory(ContextPtr aContext, int aProject, int aPatch) {
	SynFactoryPatchPtr patch = &(projects[aProject].patches[aPatch]);
	// !!!TODO: remove dependency on ptr "patch"

	switch (aContext->guiEvent) {
	case GUIEVENT_REFRESH: {
		guiSelectFillColor(aContext, thePatchColorForBackground);
		guiSelectPen1Color(aContext, -1);
		guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom);
//		guiDraw3DRect(aContext, &(aContext->guiClientRect), 0);
		XorFlag=false;

		SelectObject(aContext->hdc, Fonts[0x18]);
		const char * const satisfactory="SynFactory";
		guiDrawTransparent(aContext);
		guiSelectPen1Color(aContext, thePatchColorForModules);
		guiDrawText(aContext, 16, 16, satisfactory, strlen(satisfactory));
		if (projects[aProject].patches[aPatch].patchName) {
			guiDrawText(aContext, 16, 48, projects[aProject].patches[aPatch].patchName, strlen(projects[aProject].patches[aPatch].patchName));
		}
//      guiSelectPen1Color(aContext, 0);
//      guiDrawText(aContext, 14, 14, satisfactory,strlen(satisfactory));


		int i;

		if (mouseMode==MOUSE_DRAGSCREENZOOM) {
			drawObjectsZoomed(aContext, aProject, aPatch);
		} else {
			for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
				drawObject(aContext, aProject, aPatch, i);
			}
			for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
				drawCables(aContext, aProject, aPatch, i);
			}
		}
		if (mouseMode==MOUSE_DRAGRECT || mouseMode==MOUSE_DRAGRECTANDSCREEN) {
			showXorRect(aContext, projects[aProject].patches + aPatch);
		}
	} break;
	case GUIEVENT_MENU:
		if (aContext->id>1024) {
			int n=addSynFactoryObject(aPatch, (ObjectType)(aContext->id-1024), false);
			if (n>=0) {
				projects[aProject].patches[aPatch].objects[n].x=oldMouseClientX+projects[aProject].patches[aPatch].offsX;
				projects[aProject].patches[aPatch].objects[n].y=oldMouseClientY+projects[aProject].patches[aPatch].offsY;
			}
		} else {
			switch(aContext->id) {
			case MENU_SELECTALL: {
					int i;

					for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
						selectProjectBrowserItem(aProject, aPatch, i, -1, (i!=0) || ctrlPressed());
					}
				} break;
			case MENU_MODULEHELP: {
					int i;
					for(i=0;i<projects[aProject].patches[aPatch].maxObjects;i++) {
						if (projects[aProject].patches[aPatch].objects[i].selected) {
							displayHelp("Context help", objectHelpText(projects[aProject].patches[aPatch].objects[i].objectType));
							break;
						}
					}
				} break;
			default:
				break;
			}
		}
		break;
	case GUIEVENT_LOSTFOCUS:
		switch(mouseMode) {
		case MOUSE_DRAGOBJECT:
			finishDragObjects(aContext, patch);
			break;
		case MOUSE_DRAGCABLE:
		case MOUSE_DRAGCABLEANDSCREEN:
			finishDragCable(aContext, aProject, aPatch);
			break;
		case MOUSE_DRAGSCREEN:
		case MOUSE_DRAGSCREENZOOM:
		case MOUSE_DRAGRECT:
		case MOUSE_DRAGRECTANDSCREEN:
			ReleaseCapture();
			mouseMode=MOUSE_NONE;
			if (redrawDelay) {
				mainWindowRefresh=true;
				redrawDelay=0;
			}
			break;
		case MOUSE_KNOB:
		case MOUSE_KNOBSLOW:
			SetCursorPos(oldMouseX, oldMouseY);
			guiShowMouse(aContext);
			ReleaseCapture();
			mouseMode=MOUSE_NONE;
			break;
		case MOUSE_RESIZE_TOP_LEFT:
		case MOUSE_RESIZE_TOP_RIGHT:
		case MOUSE_RESIZE_BOTTOM_LEFT:
		case MOUSE_RESIZE_BOTTOM_RIGHT:
			resizeObject(aContext, patch);
			ReleaseCapture();
			mouseMode=MOUSE_NONE;
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_MOUSEBUTTON1DOWN:
		switch(mouseMode) {
		case MOUSE_NONE:
			synfactoryLeftClick(aContext, aProject, aPatch);
			break;
		case MOUSE_DRAGSCREEN:
			mouseMode=MOUSE_DRAGSCREENZOOM;
			mainWindowRefresh=true;
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_MOUSEBUTTON1UP:
		switch(mouseMode) {
		case MOUSE_DRAGSCREENZOOM:
			mouseMode=MOUSE_DRAGSCREEN;
			mainWindowRefresh=true;
			break;
		case MOUSE_DRAGOBJECT:
			finishDragObjects(aContext, patch);
			break;
		case MOUSE_DRAGCABLE:
			finishDragCable(aContext, aProject, aPatch);
			break;
		case MOUSE_DRAGOBJECTANDSCREEN:
			mouseMode=MOUSE_DRAGSCREEN;
			mainWindowRefresh=true;
			break;
		case MOUSE_DRAGRECT:
			ReleaseCapture();
			if (XorFlag) {
				drawXorRect(aContext, patch);
			}
			selectObjectsInArea(patch, min(startX,oldMouseClientX+patch->offsX), min(startY,oldMouseClientY+patch->offsY), max(startX,oldMouseClientX+patch->offsX), max(startY,oldMouseClientY+patch->offsY));
			mouseMode=MOUSE_NONE;
			break;
		case MOUSE_DRAGRECTANDSCREEN:
			if (XorFlag) {
				drawXorRect(aContext, patch);
			}
			selectObjectsInArea(patch, min(startX,oldMouseClientX+patch->offsX), min(startY,oldMouseClientY+patch->offsY), max(startX,oldMouseClientX+patch->offsX), max(startY,oldMouseClientY+patch->offsY));
			mouseMode=MOUSE_DRAGSCREEN;
			break;
		case MOUSE_KNOB:
		case MOUSE_KNOBSLOW:
			SetCursorPos(oldMouseX, oldMouseY);
			guiShowMouse(aContext);
			ReleaseCapture();
			mouseMode=MOUSE_NONE;
			break;
		case MOUSE_RESIZE_TOP_LEFT:
		case MOUSE_RESIZE_TOP_RIGHT:
		case MOUSE_RESIZE_BOTTOM_LEFT:
		case MOUSE_RESIZE_BOTTOM_RIGHT:
			resizeObject(aContext, patch);
			ReleaseCapture();
			mouseMode=MOUSE_NONE;
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_MOUSEBUTTON2DOWN:
		mouseDownTicks = 0;
		switch(mouseMode) {
		case MOUSE_NONE:
			storeMouse(aContext);
			SetCapture(aContext->currentWindow);
			mouseMode = MOUSE_DRAGSCREEN;
			break;
		case MOUSE_DRAGOBJECT:
			storeMouse(aContext);
			mouseMode=MOUSE_DRAGOBJECTANDSCREEN;
			break;
		case MOUSE_DRAGCABLE:
			storeMouse(aContext);
			mouseMode=MOUSE_DRAGCABLEANDSCREEN;
			break;
		case MOUSE_DRAGRECT:
			storeMouse(aContext);
			mouseMode=MOUSE_DRAGRECTANDSCREEN;
			break;
		case MOUSE_KNOB:
			mouseMode=MOUSE_KNOBSLOW;
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_MOUSEBUTTON2UP:
		switch(mouseMode) {
		case MOUSE_DRAGSCREEN:
		case MOUSE_DRAGSCREENZOOM:
			ReleaseCapture();
			mouseMode=MOUSE_NONE;
			if (redrawDelay) {
				mainWindowRefresh=true;
				redrawDelay=0;
			}
			if (mouseDownTicks<3) {
				TrackPopupMenu(synFactoryMenu[theCurrentLanguage], TPM_RIGHTBUTTON, aContext->mouseX, aContext->mouseY, 0, mainWindow, NULL);
			}
			break;
		case MOUSE_DRAGOBJECTANDSCREEN:
			mouseMode=MOUSE_DRAGOBJECT;
			if (redrawDelay) {
				mainWindowRefresh=true;
				redrawDelay=0;
			}
			break;
		case MOUSE_DRAGCABLEANDSCREEN:
			mouseMode=MOUSE_DRAGCABLE;
			if (redrawDelay) {
				mainWindowRefresh=true;
				redrawDelay=0;
			}
			break;
		case MOUSE_DRAGRECTANDSCREEN:
			mouseMode=MOUSE_DRAGRECT;
			if (redrawDelay) {
				mainWindowRefresh=true;
				redrawDelay=0;
			}
			break;
		case MOUSE_KNOBSLOW:
			mouseMode=MOUSE_KNOB;
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_MOUSEBUTTON1DCLK:
		switch(mouseMode) {
		case MOUSE_NONE:
			processPatchDoubleClick(aContext, aProject, aPatch);
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_MOUSEBUTTON2DCLK:
		switch(mouseMode) {
		case MOUSE_KNOB:
		case MOUSE_KNOBSLOW:
			projects[theCurrentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob=0;
			refreshObject(patch, &patch->objects[editObject]);
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_TIMERTICK:
		switch(mouseMode) {
		case MOUSE_DRAGOBJECT:
			updateDragObjects(aContext, patch);
			break;
		case MOUSE_DRAGCABLE:
			drawXorCable(aContext, patch);
			break;
		case MOUSE_DRAGSCREEN:
		case MOUSE_DRAGSCREENZOOM:
		case MOUSE_DRAGOBJECTANDSCREEN:
			updateDragScreen(aContext, patch);
			break;
		case MOUSE_DRAGCABLEANDSCREEN:
			updateDragScreen(aContext, patch);
			drawXorCable(aContext, patch);
			break;
		case MOUSE_DRAGRECT:
			showXorRect(aContext, patch);
			break;
		case MOUSE_DRAGRECTANDSCREEN:
			updateDragScreen(aContext, patch);
			showXorRect(aContext, patch);
			break;
		case MOUSE_KNOB:
		case MOUSE_KNOBSLOW:
			updateKnob(aContext, patch);
			break;
		case MOUSE_RESIZE_TOP_LEFT:
		case MOUSE_RESIZE_TOP_RIGHT:
		case MOUSE_RESIZE_BOTTOM_LEFT:
		case MOUSE_RESIZE_BOTTOM_RIGHT:
			resizeObject(aContext, patch);
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_KEYDOWN:
		switch(aContext->id) {
		case VK_F1: {
				int i;
				for(i=0;i<patch->maxObjects;i++) {
					if (patch->objects[i].selected) {
						displayHelp("Context help", objectHelpText(patch->objects[i].objectType));
						break;
					}
				}
			} break;
		case VK_RETURN:
			if (MOUSE_NONE == mouseMode) {
				storeMouse(aContext);
				TrackPopupMenu(synFactoryMenu[theCurrentLanguage], TPM_RIGHTBUTTON, aContext->mouseX, aContext->mouseY, 0, mainWindow, NULL);
			}
			break;
		case VK_DELETE:
			deleteSynFactoryObjects(aProject, aPatch, false);
			break;
		default:
			break;
		}
		break;
	case GUIEVENT_CHAR: {
			MouseMode tempMouseMode=MOUSE_NONE;
			int n=-1;

			int obj=findObject(aProject, aPatch, aContext->mouseClientX + patch->offsX, aContext->mouseClientY + patch->offsY, &tempMouseMode);

			if (MOUSE_NONE==tempMouseMode && !aContext->keyRepeatFlag) {
				switch(aContext->id) {
$objectKeys
				default:
					break;
				}
				if (n>=0) {
					patch->objects[n].x=aContext->mouseClientX+patch->offsX;
					patch->objects[n].y=aContext->mouseClientY+patch->offsY;
				}
			}
			if (MOUSE_EDIT_LABEL==tempMouseMode && (aContext->id==8 || (aContext->id>=32 && aContext->id<127))) {
				updateModuleLabel(patch, obj, aContext->id);
			}
			if ((aContext->id=='[' || aContext->id==']' || aContext->id=='{' || aContext->id=='}' || aContext->id=='-' || aContext->id==8 || (aContext->id>='0' && aContext->id<='9'))
			&& (MOUSE_NONE==mouseMode && MOUSE_KNOB==tempMouseMode)) {
				int temp = projects[theCurrentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob;
				if (aContext->id=='-') {
					temp=-temp;
				}
				if (aContext->id==8) {
					temp/=10;
				}
				if (aContext->id>='0' && aContext->id<='9') {
					temp*=10;
					if (temp<0) {
						temp-=aContext->id-'0';
					} else {
						temp+=aContext->id-'0';
					}
				}
				if (aContext->id=='[') {
					temp--;
				}
				if (aContext->id==']') {
					temp++;
				}
				if (aContext->id=='{') {
					temp-=10;
				}
				if (aContext->id=='}') {
					temp+=10;
				}

				if (temp > 32767) temp = 32767;
				if (temp < -32768) temp = -32768;
				projects[theCurrentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob = temp;          
				mainWindowRefresh=true;
			}
		} break;
	default:
		break;
	}
}

EOF
rv('HANDLE ICON_SIN0, ICON_SIN1, ICON_TRI0, ICON_TRI1, ICON_SAW0, ICON_SAW1;');
rv('HANDLE ICON_SPS0, ICON_SPS1, ICON_PLS0, ICON_PLS1, ICON_RND0, ICON_RND1;');
rv('HANDLE ICON_L0, ICON_L1, ICON_M0, ICON_M1, ICON_H0, ICON_H1;');
rv('HANDLE ICON_AA0, ICON_AA1;');

rv('HMENU synFactoryMenu[noofLanguages];');
cc('mainInit', "for(i=0;languageCodes[i];i++) { synFactoryMenu[i] = makeMenu(synFactoryMenuDef, mainMenuDef, i, true); }");
cc('mainInit', "ICON_SIN0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOSIN0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_SIN1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOSIN1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_TRI0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOTRI0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_TRI1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOTRI1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_SAW0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOSAW0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_SAW1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOSAW1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_SPS0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOSPS0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_SPS1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOSPS1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_PLS0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOPLS0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_PLS1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOPLS1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_RND0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCORND0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_RND1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCORND1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_L0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOL0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_L1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOL1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_M0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOM0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_M1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOM1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_H0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOH0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_H1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOH1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_AA0=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOAA0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cc('mainInit', "ICON_AA1=LoadImage(theInstance, MAKEINTRESOURCE(IDI_VCOAA1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
cci('mainTerm', "for(i=0;languageCodes[i];i++) { guiDestroyMenu(synFactoryMenu[i]); }");

rc(<<EOF);
static void patchConfigWindowHandler(ContextPtr aContext) {
	static int maxxsize=0;
	static int maxy=0;
	static int maxysize=0;

	switch(aContext->guiEvent) {
	case GUIEVENT_REFRESH: {
			int y=-guiGetVScrollPos(aContext->currentWindow);
			int x=-guiGetHScrollPos(aContext->currentWindow);

			guiClearAreaList(aContext);

			guiSelectFillColor(aContext, 0xFFFFFF);
			guiSelectPen1Color(aContext, -1);
			guiDrawRect(
				aContext,
				aContext->guiClientRect.left,
				aContext->guiClientRect.top,
				aContext->guiClientRect.right,
				aContext->guiClientRect.bottom);

			if (theCurrentProject>=0 && currentPatch>=0) {
				guiSelectPotFont(aContext);
				guiDrawTransparent(aContext);

				configWindowHeader(aContext,  x+4, &y, STRING_PATCH_MIDI_PORT);
				int j;
				guiSelectPen1Color(aContext, 0);
				for(j=0;j<27;j++) {
					if (j==projects[theCurrentProject].patches[currentPatch].midiPort) {
						guiDrawText(aContext, x+30+(j%9)*25, y+(j/9)*14, midiSelectedPortNames+j*5, 5);
					} else {
						guiDrawText(aContext, x+30+(j%9)*25, y+(j/9)*14, midiUnSelectedPortNames+j*5, 5);
					}
					configWindowAddHitArea(aContext, x+30+(j%9)*25, y+(j/9)*14, x+30+25+(j%9)*25);
				}
				y+=42;

				configWindowHeader(aContext,  x+4, &y, STRING_PATCH_MIDI_CHANNEL);
				guiSelectPen1Color(aContext, 0);
				for(j=0;j<17;j++) {
					if (j==projects[theCurrentProject].patches[currentPatch].midiChannel) {
						guiDrawText(aContext, x+30+(j%9)*25, y+(j/9)*14, midiSelectedChannelAllNames+j*5, 5);
					} else {
						guiDrawText(aContext, x+30+(j%9)*25, y+(j/9)*14, midiUnSelectedChannelAllNames+j*5, 5);
					}
					configWindowAddHitArea(aContext, x+30+(j%9)*25, y+(j/9)*14, x+30+25+(j%9)*25);
				}
				y+=28;

				configWindowHeader(aContext,  x+4, &y, STRING_PATCH_SUSTAIN_CC);
				guiSelectPen1Color(aContext, 0);
				guiDrawText(aContext, x+30, y, (projects[theCurrentProject].patches[currentPatch].sustainCC==0)?"[off]":" off ", 5);
				configWindowAddHitArea(aContext, x+30, y, x+30+25);
				guiDrawText(aContext, x+55, y, (projects[theCurrentProject].patches[currentPatch].sustainCC==64)?"[ 64]":"  64 ", 5);
				configWindowAddHitArea(aContext, x+55, y, x+55+25);
				guiDrawText(aContext, x+80, y, (projects[theCurrentProject].patches[currentPatch].sustainCC==65)?"[ 65]":"  65 ", 5);
				configWindowAddHitArea(aContext, x+80, y, x+80+25);
				guiDrawText(aContext, x+105, y, (projects[theCurrentProject].patches[currentPatch].sustainCC==66)?"[ 66]":"  66 ", 5);
				configWindowAddHitArea(aContext, x+105, y, x+105+25);
				guiDrawText(aContext, x+130, y, (projects[theCurrentProject].patches[currentPatch].sustainCC==67)?"[ 67]":"  67 ", 5);
				configWindowAddHitArea(aContext, x+130, y, x+130+25);
				guiDrawText(aContext, x+155, y, (projects[theCurrentProject].patches[currentPatch].sustainCC==68)?"[ 68]":"  68 ", 5);
				configWindowAddHitArea(aContext, x+155, y, x+155+25);
				guiDrawText(aContext, x+180, y, (projects[theCurrentProject].patches[currentPatch].sustainCC==69)?"[ 69]":"  69 ", 5);
				configWindowAddHitArea(aContext, x+180, y, x+180+25);
				y+=14;


	//    guiSelectFillColor(aContext, 0xFFFFFF);
	//    guiSelectPen1Color(aContext, -1);
	//    guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom);
				if (/*y+guiGetVScrollPos(aContext->currentWindow)!=maxy ||*/ (aContext->guiClientRect.right-aContext->guiClientRect.left)!=maxxsize) {
//          maxx=250;
					maxxsize=aContext->guiClientRect.right-aContext->guiClientRect.left;
					guiHScrollRange(aContext->currentWindow, 0, 250/*maxy*/, maxxsize);
				}
				if (y+guiGetVScrollPos(aContext->currentWindow)!=maxy || (aContext->guiClientRect.bottom-aContext->guiClientRect.top)!=maxysize) {
					maxy=y+guiGetVScrollPos(aContext->currentWindow);
					maxysize=aContext->guiClientRect.bottom;
					guiVScrollRange(aContext->currentWindow, 0, maxy, maxysize);
				}
			}

		} break;
	case GUIEVENT_MOUSEBUTTON1DOWN: {
			int index=0;
			int i;

			for(i=0;i<27;i++) {
				index++;
				if (index==aContext->id) {
					projects[theCurrentProject].patches[currentPatch].midiPort=i;
					patchConfigWindowRefresh=true;
				}
			}
			for(i=0;i<17;i++) {
				index++;
				if (index==aContext->id) {
					projects[theCurrentProject].patches[currentPatch].midiChannel=i;
					patchConfigWindowRefresh=true;
				}
			}
			if (++index==aContext->id) {
				projects[theCurrentProject].patches[currentPatch].sustainCC=0;
				patchConfigWindowRefresh=true;
			}
			if (++index==aContext->id) {
				projects[theCurrentProject].patches[currentPatch].sustainCC=64;
				patchConfigWindowRefresh=true;
			}
			if (++index==aContext->id) {
				projects[theCurrentProject].patches[currentPatch].sustainCC=65;
				patchConfigWindowRefresh=true;
			}
			if (++index==aContext->id) {
				projects[theCurrentProject].patches[currentPatch].sustainCC=66;
				patchConfigWindowRefresh=true;
			}
			if (++index==aContext->id) {
				projects[theCurrentProject].patches[currentPatch].sustainCC=67;
				patchConfigWindowRefresh=true;
			}
			if (++index==aContext->id) {
				projects[theCurrentProject].patches[currentPatch].sustainCC=68;
				patchConfigWindowRefresh=true;
			}
			if (++index==aContext->id) {
				projects[theCurrentProject].patches[currentPatch].sustainCC=69;
				patchConfigWindowRefresh=true;
			}
		} break;
	case GUIEVENT_KEYDOWN:
		switch(aContext->id) {
		case VK_ESCAPE:
			guiHideWindow(aContext->currentWindow);
			break;
		default:
			break;
		} break;
	case GUIEVENT_CLOSE:
		guiHideWindow(aContext->currentWindow);
		break;
	default:
		break;
	}
}
EOF

genWindow('patchConfigWindow', 'PatchConfigWindow', '"Patch Settings"', 'patchConfigWindowHandler', undef, undef, 1, 1, undef);

cc('stfLoad', 'if (strcmp(loadInfo->headerTitle, "PatchCableType")==0) { int c; readStfInteger(loadInfo, &c); setTheCurrentCableType(c); }');
cc('stfSave', 'storeStfBegin(loadInfo, "PatchCableType"); storeStfInteger(loadInfo, theCurrentCableType); storeStfEnd(loadInfo);');

cc('stfLoad', 'if (strcmp("DefColorPatch", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) setThePatchColorForBackground(c); }');
cc('stfSave', 'storeStfBegin(loadInfo, "DefColorPatch"); storeStfInteger(loadInfo, thePatchColorForBackground); storeStfEnd(loadInfo);');
cc('stfLoad', 'if (strcmp("DefColorPatchModules", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) setThePatchColorForModules(c); }');
cc('stfSave', 'storeStfBegin(loadInfo, "DefColorPatchModules"); storeStfInteger(loadInfo, thePatchColorForModules); storeStfEnd(loadInfo);');
cc('stfLoad', 'if (strcmp("DefColorPatchKnobs", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) setThePatchColorForKnobs(c); }');
cc('stfSave', 'storeStfBegin(loadInfo, "DefColorPatchKnobs"); storeStfInteger(loadInfo, thePatchColorForKnobs); storeStfEnd(loadInfo);');
cc('stfLoad', 'if (strcmp("DefColorPatchHighlight", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) setThePatchColorForHighlight(c); }');
cc('stfSave', 'storeStfBegin(loadInfo, "DefColorPatchHighlight"); storeStfInteger(loadInfo, thePatchColorForHighlight); storeStfEnd(loadInfo);');
cc('stfLoad', 'if (strcmp("DefColorPatchShadow", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) setThePatchColorForShadow(c); }');
cc('stfSave', 'storeStfBegin(loadInfo, "DefColorPatchShadow"); storeStfInteger(loadInfo, thePatchColorForShadow); storeStfEnd(loadInfo);');
