##########################################################################
#
# StudioFactory
#
# The desktop Audio/Video studio.
# Copyright (C) 2002-2007  Peter Wendrich (pwsoft@syntiac.com)
# Homepage: http://www.syntiac.com/studiofactory.html
#
##########################################################################
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
##########################################################################
#
# Generation of source files and compilation routines
#
##########################################################################

use strict;
use warnings;
use Fcntl;

my $theStudioFactoryVersion = '0.43';


cc('theDcFilterEnabled', 'markMenuItem(mainMenu, MENU_DCFILTER, theDcFilterEnabled);');



sub genStringFromFile {
	my $stringName = shift;
	my $fileName = shift;

	rv("static const char $stringName\[\]=");
	open(FILE, $fileName) || die "Can't open $fileName";
	while(<FILE>) {
		s/\r//;
		s/\n//;
		if (length($_)>0) {
			rv_("\n\"$_\"");
		}
	}
	close(FILE);
	rv_(';');
}

##########################################################################
#
# Text Editor Object
#
##########################################################################
sub genTextEditorGdi {
	my @TextEditorStructDef=('GuiWindowPtr window;', 'char *textBuffer;', 'int cursorPos;', 'int bufferSize;', 'int bufferMaxSize;', 'void (*updateFunc)(int updateUserId, const char *text);', 'int updateUserId;');
	genEnumStruct("struct", "TextEditor", \@TextEditorStructDef);

	rc(<<EOF);
static void initTextEditor(TextEditorPtr editor, const char *text) {
	if (editor->textBuffer) free(editor->textBuffer);
	if (text!=NULL) {
		editor->bufferSize=strlen(text);
		editor->bufferMaxSize=(editor->bufferSize+255)&(~255); // round up to nearest 256 marker
		editor->textBuffer=(char*)malloc(editor->bufferMaxSize);
		strcpy(editor->textBuffer, text);
	} else {
		editor->textBuffer=NULL;
		editor->bufferSize=0;
		editor->bufferMaxSize=0;
	}
	editor->cursorPos=0;
}

static void showTextEditor(ContextPtr aContext, TextEditorPtr editor, int left, int top, int right, int bottom) {
}

static void hideTextEditor(ContextPtr aContext, TextEditorPtr editor) {
	guiRefreshWindow(aContext->currentWindow);
}
EOF
}

#
# !!! Temp solution use SF2.0 text editor
#
sub genSynTextEditor {
	rc(readCompleteFile("editor_functions.txt"));

	cc('mainInit', 'InitNoteScriptEditor();');
	cci('mainTerm', 'DeInitNoteScriptEditor();');
}


##########################################################################
#
# DSP Engine
#
##########################################################################
sub genDsp {

	ccModule('DSP engine');

	rd('static const int maxObjectIo=32;');
	rd('static const int maxDspObjects=1024;');
	rd('static const int scriptReturnStackSize=256;');
	my @ScriptInfoStructDef=('int rSP;', 'int dSP;', 'int workVars[26];', 'int returnStack[scriptReturnStackSize];', 'char *script;');
	genEnumStruct("struct", "ScriptInfo", \@ScriptInfoStructDef);
	my @DspObjectStructDef=(
		'bool inUse;',
		'void (*dspRoutine)(_DspObject *);',
		'struct {',
		'int *input;',
		'int intern;',
		'int output;',
		'int knob;',
		'} io[maxObjectIo];',
		'union {',
		'ScriptInfo *scriptInfo;',
		'signed short *buffer;',
		'};'
	);
	genEnumStruct("struct", "DspObject", \@DspObjectStructDef);
	rd('typedef void (*DspRoutinePtr)(DspObject *);');


	rv('short EXPTBL16[32768];');
	rv('short SINTBL[65536];');
	rv('int EXPTBL32[32768];');
	rv('int DSP_Left;');
	rv('int DSP_Right;');
	rv('int dspInputLeft;');
	rv('int dspInputRight;');

rv_(<<EOF);
static const unsigned char PERLINTBL[256]={
	 151,160,137,91,90,15,
	 131,13,201,95,96,53,194,233,7,225,140,36,103,30,69,142,8,99,37,240,21,10,23,
	 190, 6,148,247,120,234,75,0,26,197,62,94,252,219,203,117,35,11,32,57,177,33,
	 88,237,149,56,87,174,20,125,136,171,168, 68,175,74,165,71,134,139,48,27,166,
	 77,146,158,231,83,111,229,122,60,211,133,230,220,105,92,41,55,46,245,40,244,
	 102,143,54, 65,25,63,161, 1,216,80,73,209,76,132,187,208, 89,18,169,200,196,
	 135,130,116,188,159,86,164,100,109,198,173,186, 3,64,52,217,226,250,124,123,
	 5,202,38,147,118,126,255,82,85,212,207,206,59,227,47,16,58,17,182,189,28,42,
	 223,183,170,213,119,248,152, 2,44,154,163, 70,221,153,101,155,167, 43,172,9,
	 129,22,39,253, 19,98,108,110,79,113,224,232,178,185, 112,104,218,246,97,228,
	 251,34,242,193,238,210,144,12,191,179,162,241, 81,51,145,235,249,14,239,107,
	 49,192,214, 31,181,199,106,157,184, 84,204,176,115,121,50,45,127, 4,150,254,
	 138,236,205,93,222,114,67,29,24,72,243,141,128,195,78,66,215,61,156,180};

//
// Remap pots in correct order for Sequencer
//
static const int SeqPotRemap[16]={
	0,2,4,6,8,10,12,14,
	1,3,5,7,9,11,13,15
};

static int dspRnd1;
static int dspRnd2;

EOF

rc(<<EOF);
//
// Generate SIN and EXP tables
//
void initDspTables(void) {
	double pi = 3.141592653589793;
	int cnt;

	// Generate EXP table
	for(cnt=0;cnt<32768;cnt++) {
		EXPTBL16[cnt]=(short)(exp(cnt*log(32767.0)/32767.0));
		EXPTBL32[cnt]=  (int)((exp(cnt*log(32768.0)/32767.0)-1)*8192.0);
	}

	// Generate SIN table
	for(cnt=-32768;cnt<32768;cnt++) {
		SINTBL[cnt+32768]=(short)(32767.0*sin((double)cnt*(pi/32768.0)));
	}
}


// Find unused Dsp object. We start allocating from 1 until maxDspObjects-1, because 0 is placeholder for constants and last is used for loop termination.
static int nextFreeDsp(void) {
	int n=0;
	int i;

	for(i=1;i<maxDspObjects-1;i++) {
		if (!projects[theCurrentProject].dsp[i].inUse) {
			n=i;
			memset(&(projects[theCurrentProject].dsp[i]), 0, sizeof(DspObject));
			for(i=0;i<maxObjectIo;i++) {
				projects[theCurrentProject].dsp[n].io[i].input=&(projects[theCurrentProject].dsp[0].io[0].output);
			}
			break;
		}
	}
	return n;
}
EOF

	rc(readCompleteFile("dsp_functions.txt"));


rc(<<EOF);

//
// Reset all object to default state
//
static void dspResetAll(DspObject *objectList) {
	DspObject *currentObject=objectList;

	for(currentObject=objectList;;currentObject++) {
		if (currentObject->dspRoutine) {
			int i;

			for(i=0;i<maxObjectIo;i++) {
				currentObject->io[i].output=0;
				currentObject->io[i].intern=0;
			}
		} else {
			break;
		}
	}

#if 0
	for(;;) {
		if (CurObj->dspRoutine==NULL) {
			// No more modules
			tempoCounter=999999; // Force first step directly after restart
			ticksCounter=0;
			curPlayTime=0;
			curTempo=882;
			curTicks=6;
			curSongPos=0;
			curRow=-1;
			return;
		}
		if (CurObj->dspRoutine==DSPRoutine_FLA) {
			memset(CurObj->Buffer, 0 , objectInfo[OBJ_FLA].BufferSize);
		}
		if (CurObj->dspRoutine==DSPRoutine_DLY) {
			memset(CurObj->Buffer, 0 , objectInfo[OBJ_DLY].BufferSize);
		}
		if (CurObj->dspRoutine==DSPRoutine_SDY) {
			memset(CurObj->Buffer, 0 , objectInfo[OBJ_SDY].BufferSize);
		}
		if (CurObj->dspRoutine==DSPRoutine_SCRIPT) {
			CurObj->ScriptInfo->RSP=0;
			memset(&(CurObj->ScriptInfo->ReturnStack), 0, NOTESCRIPT_RETURNSTACKSIZE*sizeof(int));
			memset(&(CurObj->ScriptInfo->workVars), 0, 26*sizeof(int));
		}
		for(cnt=0; cnt<MAX_IO; cnt++) {
			CurObj->Out[cnt]=0;
		}
		for (cnt=0; cnt<MAX_INTERN; cnt++) {
			CurObj->Intern[cnt] = 0;
		}

		CurObj++;
	}
#endif
}


//
// Calculate wave block for pause/stop mode. Only zero's are output to the
// soundcard.
//
void dspCalculateZeroBlock(signed short *MemoryBlock, signed short *EndAddr) {
	do {
		*MemoryBlock = 0; // Fill with zero
		MemoryBlock++;
	} while (MemoryBlock != EndAddr);
	if (audioInReady) {
		int noofBuffers=audioInputRingWritePos-audioInputRingReadPos;
		if (noofBuffers<0) noofBuffers+=audioInputRingSize;
		if (noofBuffers>5) {
			DSP_BlockUse[audioInputRing[audioInputRingReadPos++]]=AUDIOBLOCK_UNUSED;
			if (audioInputRingReadPos>=audioInputRingSize) audioInputRingReadPos=0;
		}
		if (noofBuffers>4) {
			DSP_BlockUse[audioInputRing[audioInputRingReadPos++]]=AUDIOBLOCK_UNUSED;
			if (audioInputRingReadPos>=audioInputRingSize) audioInputRingReadPos=0;
		}
	}
}

static void dspCalculateBlock(DspObject *objectList, signed short *MemoryBlock, signed short *EndAddr) {
	int max_left=0;
	int max_right=0;
	DspObject *currentObject;

	dspInputLeft=0;
	dspInputRight=0;
	do {
		DSP_Left=0;
		DSP_Right=0;

		if (audioInReady) {
			int noofBuffers=audioInputRingWritePos-audioInputRingReadPos;
			if (noofBuffers<0) noofBuffers+=audioInputRingSize;
			if (noofBuffers>0) {
				// The noofBuffers>5 test throws away input blocks if we are to slow to process (high CPU load then sometimes playback is slower as realtime)
				if (noofBuffers>5 || audioInputBufferPos*2>=DSP_BlockSize[audioInputRing[audioInputRingReadPos]]) {
					audioInputBufferPos=0;
					DSP_BlockUse[audioInputRing[audioInputRingReadPos++]]=AUDIOBLOCK_UNUSED;
					if (audioInputRingReadPos>=audioInputRingSize) audioInputRingReadPos=0;
				}
				dspInputLeft=((signed short *)(DSP_DriverBlock[audioInputRing[audioInputRingReadPos]]))[audioInputBufferPos*2];
				dspInputRight=((signed short *)(DSP_DriverBlock[audioInputRing[audioInputRingReadPos]]))[(audioInputBufferPos++)*2+1];
				
				// Audio input <-> output Sync trying to keep 3 buffers in the ring
				if ((audioInputBufferSubPos++)>1024) {
					audioInputBufferSubPos=0;
					if (noofBuffers<3) audioInputBufferPos--;
					if (noofBuffers>3) audioInputBufferPos++;
				}
			} else {
				dspInputLeft=0;
				dspInputRight=0;
			}
		}

//		for(currentObject=objectList;;currentObject++) {
//			if (currentObject->dspRoutine) {
//				(currentObject->dspRoutine)(currentObject);
//			} else {
//				break;
//			}
//		}

		if (theDspOptActiveFlag) {
			__asm {
				mov eax, theDspOptBuffer
				call eax
			}
		} else {
			for(currentObject=objectList;currentObject->dspRoutine;currentObject++) {
				(currentObject->dspRoutine)(currentObject);
			}
		}

		// Clip audio and put in driver buffer
		{
			int audio_sample_l=limit(DSP_Left);
			int audio_sample_r=limit(DSP_Right);
			max_left=max(max_left, abs(audio_sample_l));
			max_right=max(max_right, abs(audio_sample_r));
			*(MemoryBlock++) = (short)(audio_sample_l);
			*(MemoryBlock++) = (short)(audio_sample_r);
			if (DSP_ScopePosition<(4*SCOPEBUFSIZE)) {
				DSP_ScopeL[DSP_ScopePosition>>2]=(short)audio_sample_l;
				DSP_ScopeR[DSP_ScopePosition>>2]=(short)audio_sample_r;
				DSP_ScopePosition++;
			}
		}
	} while (MemoryBlock != EndAddr);
	updateVuMeter(max_left, max_right);
}

int dc_left = 0;
int dc_right = 0;
static void dspCalculateBlockDcFilter(DspObject *objectList, signed short *MemoryBlock, signed short *EndAddr) {
	int max_left=0;
	int max_right=0;
	DspObject *currentObject;

	dspInputLeft=0;
	dspInputRight=0;
	do {
		DSP_Left=0;
		DSP_Right=0;

		if (audioInReady) {
			int noofBuffers=audioInputRingWritePos-audioInputRingReadPos;
			if (noofBuffers<0) noofBuffers+=audioInputRingSize;
			if (noofBuffers>0) {
				// The noofBuffers>5 test throws away input blocks if we are to slow to process (high CPU load then sometimes playback is slower as realtime)
				if (noofBuffers>5 || audioInputBufferPos*2>=DSP_BlockSize[audioInputRing[audioInputRingReadPos]]) {
					audioInputBufferPos=0;
					DSP_BlockUse[audioInputRing[audioInputRingReadPos++]]=AUDIOBLOCK_UNUSED;
					if (audioInputRingReadPos>=audioInputRingSize) audioInputRingReadPos=0;
				}
				dspInputLeft=((signed short *)(DSP_DriverBlock[audioInputRing[audioInputRingReadPos]]))[audioInputBufferPos*2];
				dspInputRight=((signed short *)(DSP_DriverBlock[audioInputRing[audioInputRingReadPos]]))[(audioInputBufferPos++)*2+1];
				
				// Audio input <-> output Sync trying to keep 3 buffers in the ring
				if ((audioInputBufferSubPos++)>1024) {
					audioInputBufferSubPos=0;
					if (noofBuffers<3) audioInputBufferPos--;
					if (noofBuffers>3) audioInputBufferPos++;
				}
			} else {
				dspInputLeft=0;
				dspInputRight=0;
			}
		}

//		for(currentObject=objectList;;currentObject++) {
//			if (currentObject->dspRoutine) {
//				(currentObject->dspRoutine)(currentObject);
//			} else {
//				break;
//			}
//		}

		if (theDspOptActiveFlag) {
			__asm {
				mov eax, theDspOptBuffer
				call eax
			}
		} else {
			for(currentObject=objectList;currentObject->dspRoutine;currentObject++) {
				(currentObject->dspRoutine)(currentObject);
			}
		}

		// Clip audio and put in driver buffer
		{
			dc_left = dc_left + (DSP_Left * 16384 - dc_left) / 1024;
			dc_right = dc_right + (DSP_Right * 16384 - dc_right) / 1024;
			int audio_sample_l=limit(DSP_Left - dc_left / 16384);
			int audio_sample_r=limit(DSP_Right - dc_right / 16384);
			max_left=max(max_left, abs(audio_sample_l));
			max_right=max(max_right, abs(audio_sample_r));
			*(MemoryBlock++) = (short)(audio_sample_l);
			*(MemoryBlock++) = (short)(audio_sample_r);
			if (DSP_ScopePosition<(4*SCOPEBUFSIZE)) {
				DSP_ScopeL[DSP_ScopePosition>>2]=(short)audio_sample_l;
				DSP_ScopeR[DSP_ScopePosition>>2]=(short)audio_sample_r;
				DSP_ScopePosition++;
			}
		}
	} while (MemoryBlock != EndAddr);
	updateVuMeter(max_left, max_right);
}

EOF

	cc('mainInit', "initDspTables();");
}

sub genLoadWindow {
	my $name = shift;
	my $saveName = shift;
	my $fixedSize = shift;

	if (defined($fixedSize)) {
	cc('synLoad', <<EOF);
	if ((section==SECTION_WINDOWS) && strcmp(loadInfo->headerTitle, \"$saveName\")==0) {
		int v,x,y;

		readInteger(loadInfo, &v,  true);
		readInteger(loadInfo, &x,  true);
		readInteger(loadInfo, &y,  true);
		guiPositionWindow($name, x, y);
		if (0==v) guiHideWindow($name);
		if (0!=v) guiShowWindow($name);
	}
EOF
	} else {
	cc('synLoad', <<EOF);
	if ((section==SECTION_WINDOWS) && strcmp(loadInfo->headerTitle, \"$saveName\")==0) {
		int v,x,y,xs,ys;

		readInteger(loadInfo, &v,  true);
		readInteger(loadInfo, &x,  true);
		readInteger(loadInfo, &y,  true);
		readInteger(loadInfo, &xs, true);
		readInteger(loadInfo, &ys, true);
		guiPositionSizeWindow($name, x, y, xs, ys);		
		if (0==v) guiHideWindow($name);
		if (0!=v) guiShowWindow($name);
	}
EOF
	}
}

my $generatedLoadAndSaveWindowStf=0;
sub genWindow {
	my $name = shift;
	my $saveName = shift;
	my $title = shift;
	my $handlerFunction = shift;
	my $windowXSize = shift;
	my $windowYSize = shift;
	my $vscroll = shift;
	my $hscroll = shift;
	my $fixedSize = shift;

	genLoadWindow($name, $saveName, $fixedSize);

	if ($generatedLoadAndSaveWindowStf == 0) {
	rc(<<EOF);
void loadWindowStf(LoadInfo *loadInfo, const char *aTitle, GuiWindowPtr windowHandle, bool fixedSize) {
	if (strcmp(aTitle, loadInfo->headerTitle)==0) {
		if (fixedSize) {
			int v,x,y;

			readStfInteger(loadInfo, &v);
			readStfInteger(loadInfo, &x);
			readStfInteger(loadInfo, &y);
			guiPositionWindow(windowHandle, x, y);
			if (0==v) guiHideWindow(windowHandle);
			if (0!=v) guiShowWindow(windowHandle);
		} else {
			int v,x,y,xs,ys;

			readStfInteger(loadInfo, &v);
			readStfInteger(loadInfo, &x);
			readStfInteger(loadInfo, &y);
			readStfInteger(loadInfo, &xs);
			readStfInteger(loadInfo, &ys);
			guiPositionSizeWindow(windowHandle, x, y, xs, ys);
			if (0==v) guiHideWindow(windowHandle);
			if (0!=v) guiShowWindow(windowHandle);
			if (2==v) guiMaximizeWindow(windowHandle);
		}
	}
}

void saveWindowStf(LoadInfo *loadInfo, const char *aTitle, GuiWindowPtr windowHandle, bool fixedSize) {
	int show=0;
	int x,y,w,h;

	guiGetWindowPosition(windowHandle, &x, &y, &w, &h);

	if (guiIsWindowVisible(windowHandle)) {
		if (guiIsWindowMaximized(windowHandle)) {
			show=2;
		} else {
			show=1;
		}
	}
	if (fixedSize) {
		storeStfBegin(loadInfo, aTitle);
		storeStfInteger(loadInfo, show);
		storeStfInteger(loadInfo, x);
		storeStfInteger(loadInfo, y);
		storeStfEnd(loadInfo);
	} else {
		storeStfBegin(loadInfo, aTitle);
		storeStfInteger(loadInfo, show);
		storeStfInteger(loadInfo, x);
		storeStfInteger(loadInfo, y);
		storeStfInteger(loadInfo, w);
		storeStfInteger(loadInfo, h);
		storeStfEnd(loadInfo);
	}
}
EOF
		$generatedLoadAndSaveWindowStf=1;
	}

#  push(@stfSave, "sprintf(myTempBuf, \"windowpos\");");

	# backwards compatibility (versions before 1.13 don't have resizable scopewindow)
	if ($saveName eq 'SCOPEWINDOW2') {
		genLoadWindow($name, 'SCOPEWINDOW', 1);
	}

	if ($vscroll == 1) {
		$vscroll = ' | WS_VSCROLL';
	} else {
		$vscroll = '';
	}
	if ($hscroll == 1) {
		$hscroll = ' | WS_HSCROLL';
	} else {
		$hscroll = '';
	}

	rv("bool ${name}Refresh;");
#	if ($name eq 'mainWindow') {
#		rv("GuiMainWindowPtr $name;");
#	} else {
		rv("GuiWindowPtr $name;");
#	}
	cc('refreshFunctions', "if (${name}Refresh) { guiRefreshWindow(${name}); ${name}Refresh=false; }");

#	if ($platformGraphics eq 'GDI') {
		if (defined($fixedSize)) {
			cc('stfLoad100', "loadWindowStf(loadInfo, \"$saveName\", $name, true);");
			cc('stfSave', "saveWindowStf(loadInfo, \"$saveName\", $name, true);");
			$fixedSize='';
		} else {
			cc('stfLoad100', "loadWindowStf(loadInfo, \"$saveName\", $name, false);");
			cc('stfSave', "saveWindowStf(loadInfo, \"$saveName\", $name, false);");
			$fixedSize=' | WS_THICKFRAME | WS_MINIMIZEBOX | WS_MAXIMIZEBOX';
		}
		if (!defined($windowXSize)) {
			$windowXSize=400;
		} else {
			$windowXSize.='+2*GetSystemMetrics(SM_CXFIXEDFRAME)';
		}
		if (!defined($windowYSize)) {
			$windowYSize=400;
		} else {
			$windowYSize.='+GetSystemMetrics(SM_CYSMCAPTION) + 2*GetSystemMetrics(SM_CYFIXEDFRAME)';
		}
		if ($name eq 'mainWindow') {
			cc('windowInit', "$name = CreateWindowEx(0, mainWindowClass, $title, WS_SYSMENU | WS_CLIPCHILDREN | WS_OVERLAPPED | WS_VISIBLE$fixedSize$vscroll$hscroll, CW_USEDEFAULT, CW_USEDEFAULT, $windowXSize, $windowYSize, NULL, NULL, theInstance, NULL);");
			cc('windowInit', "SetWindowLong($name, WNDEXTRA_EVENTFUNC, (LONG)$handlerFunction);");
			cc('windowInit', "UpdateWindow($name);");
		} else {
			cc('windowInit', "$name = CreateWindowEx(WS_EX_TOOLWINDOW, mainWindowClass, $title, WS_SYSMENU | WS_CLIPCHILDREN | WS_OVERLAPPED$fixedSize$vscroll$hscroll, CW_USEDEFAULT, CW_USEDEFAULT, $windowXSize, $windowYSize, mainWindow, NULL, theInstance, NULL);");
			cc('windowInit', "SetWindowLong($name, WNDEXTRA_EVENTFUNC, (LONG)$handlerFunction);");
		}
#	}
	if (0) {
		if (!defined($windowXSize)) {
			$windowXSize=400;
		}
		if (!defined($windowYSize)) {
			$windowYSize=400;
		}
		if ($name eq 'mainWindow') {
			cc('windowInit', "$name = new GuiMainWindow(NULL, mainWindowHandler);");
			cc('windowInit', "theApplication->setMainWidget($name);");
			cc('windowInit', "$name->show();");
#			cc('windowInit', "new guiMenuBar($name);");
		} else {
			cc('windowInit', 'logprintf("before GuiToolWindow\\n");');
			cc('windowInit', "$name = new GuiToolWindow(mainWindow, $handlerFunction, Qt::WType_TopLevel | Qt::WStyle_Tool | Qt::WType_Dialog | Qt::WResizeNoErase | Qt::WRepaintNoErase);");
			cc('windowInit', 'logprintf("after GuiToolWindow\\n");');
			if (defined($fixedSize)) {
				cc('windowInit', "$name->setFixedSize($windowXSize, $windowYSize);");
			} else {
				cc('windowInit', "$name->resize($windowXSize, $windowYSize);");
			}
#			cc('windowInit', "$name->setEraseColor(QColor(128,0,0));");
			cc('windowInit', "$name->show();");
		}
		cc('windowInit', "guiSetWindowTitle($name, $title);");
	}
#	if ($platformGraphics eq 'Xt') {
#     cc('windowInit', "$name = XtCreateWidget(\"$name\", , mainWindow,,);");
#		cci('windowTerm', "XtDestroyWidget($name);");
#	}
	cci('windowTerm', "guiDestroyWindow($name);");
}

sub genMainWindow {
	genWindow('mainWindow', 'MAINWINDOW', "\"StudioFactory $theStudioFactoryVersion - http://www.syntiac.com\"", 'mainWindowHandler', 400, 400, 0, 0, undef);
	cc('windowInit', "guiRunTimer(mainWindow, 100);");
	cci('windowTerm', "guiSetMenu(mainWindow, NULL);");
}

sub genMainWindowProcessing {
rc(<<EOF);
static void mainWindowHandler(ContextPtr aContext) {
	switch(aContext->guiEvent) {
	case GUIEVENT_GETFOCUS:
	case GUIEVENT_LOSTFOCUS:
		if (theCurrentProject>=0) {
			if (currentPatch>=0) {
				processSynFactory(aContext, theCurrentProject, currentPatch);
			} else if (currentSample>=0) {
				processSampleEditor(aContext);
			} else {
				processTimeline(aContext);
			}
		}
		break;
	case GUIEVENT_REFRESH:
		guiClearAreaList(aContext);
//    lastMainWindowArea=0;
//    processTimeline(aContext);
		if (theCurrentProject>=0 && currentPatch>=0) {
			processSynFactory(aContext, theCurrentProject, currentPatch);
		} else {
			processTimeline(aContext);
		}
		break;
	case GUIEVENT_MENU:
		switch(aContext->id) {
		case MENU_NEW_PROJECT:
			createProject(newProjectName);
			break;
		case MENU_NEW_SYNFACTORY:
			createSynfactoryPatch();
			break;
		case MENU_NEW_SEQUENCER:
			createSequencer();
			break;
//		case MENU_NEW_TRACK:
			// !!! disabled !!! createTrack();
//			break;
		case MENU_OPEN_FILE:
			openFileSelect(aContext, false);
			break;
		case MENU_ADD_FILE:
			openFileSelect(aContext, true);
			break;
		case MENU_CLOSE:
			// Implement !!!
			break;
		case MENU_CLOSE_PROJECT:
			// !!! Check for modified and give warning!
			if (theCurrentProject>=0) {
				deleteProject(theCurrentProject);
			}
			break;
		case MENU_SAVE:
			if (theCurrentProject>=0) {
				if (projects[theCurrentProject].projectPath) {
					saveStfToFilename(aContext, projects[theCurrentProject].projectPath, false, theCurrentProject);
				} else {
					saveFileSelect(aContext, false);
				}
			}
			break;
		case MENU_SAVE_AS:
			saveFileSelect(aContext, false);
			break;
		case MENU_EXIT:
			quitFlag=true;
			break;
		case MENU_CUT:
			saveStfToClipboard();
			deleteSelected();
			break;
		case MENU_COPY:
			saveStfToClipboard();
			break;
		case MENU_PASTE:
			loadFromClipboard();
			break;
		case MENU_DELETE:
			deleteSelected();
			break;
		case MENU_CLONE:
			cloneSelected();
			break;
		case MENU_TOGGLELABEL:
			toggleLabel();
			break;
		case MENU_CABLECOLOR_0:
		case MENU_CABLECOLOR_1:
		case MENU_CABLECOLOR_2:
		case MENU_CABLECOLOR_3:
		case MENU_CABLECOLOR_4:
		case MENU_CABLECOLOR_5:
		case MENU_CABLECOLOR_6:
		case MENU_CABLECOLOR_7:
			setCurrentCableColor(aContext->id-MENU_CABLECOLOR_0);
			break;
		case MENU_PROJECTBROWSER:
			~cc~showProjectBrowser~
			break;
		case MENU_COLORSELECTOR:
			~cc~showColorSelector~
			break;
		case MENU_TRANSPORT:
			~cc~showTransport~
			break;
		case MENU_OUTPUTSCOPE:
			~cc~showAudioOutputScope~
			break;
		case MENU_AUDIOMIXER:
			~cc~showAudioMixer~
			break;
		case MENU_MIDIACTIVITY:
			~cc~showMidiActivityMonitor~
			break;
		case MENU_MIDISCOPE:
			~cc~showMidiScope~
			break;
		case MENU_MIDIKEYBOARD:
			~cc~showVirtualMidiKeyboard~
			break;
		case MENU_REWIND:
			setTheRewindTicker(2);
			resetPlayback();
			break;
		case MENU_STOP:
			setPlayMode(PLAYMODE_STOP);
			break;
		case MENU_PLAY:
			setPlayMode(PLAYMODE_PLAY);
			break;
		case MENU_RECORD:
			if (theRecordMode!=RECORDMODE_STOP) {
				setRecordMode(RECORDMODE_STOP);
			} else {
				setRecordMode(RECORDMODE_RECORD);
			}
			break;
		case MENU_DCFILTER:
			setTheDcFilterEnabled(!theDcFilterEnabled);
			break;
		case MENU_DSPOPT:
			theDspOptEnabledFlag = !theDspOptEnabledFlag;
			markMenuItem(mainMenu, MENU_DSPOPT, theDspOptEnabledFlag);
			markMenuItem(synFactoryMenu, MENU_DSPOPT, theDspOptEnabledFlag);
			~cc~recalcDsp~
			break;
		case MENU_GLOBAL_SETTINGS:
			~cc~showConfigPanel~
			break;
		case MENU_PATCH_SETTINGS:
			~cc~showPatchConfig~
			break;
		case MENU_SAVE_SETTINGS:
			saveFileSelect(aContext, true);
			break;
		case MENU_MENUHELP:
			displayHelp("Menu help", STRING_MENUHELP);
			break;
		case MENU_ABOUT:
			displayHelp("About StudioFactory", aboutHelpText);
			break;
		default:
			if (theCurrentProject>=0 && currentPatch>=0) {
				processSynFactory(aContext, theCurrentProject, currentPatch);
			} else {
				processTimeline(aContext);
			}
			break;
		}
		break;
	case GUIEVENT_MOUSEBUTTON1DOWN:
	case GUIEVENT_MOUSEBUTTON1UP:
	case GUIEVENT_MOUSEBUTTON1DCLK:
	case GUIEVENT_MOUSEBUTTON2DOWN:
	case GUIEVENT_MOUSEBUTTON2UP:
	case GUIEVENT_MOUSEBUTTON2DCLK:
	case GUIEVENT_TIMERTICK:
	case GUIEVENT_CHAR:
		switch(aContext->id) {
		case 3:
			// CTRL+C
			// Ignore repeated events when key is held down
			if (!aContext->keyRepeatFlag) {
				saveStfToClipboard();
			}
			break;
		case 4:
			// CTRL+D
			// Ignore repeated events when key is held down
			if (!aContext->keyRepeatFlag) {
				cloneSelected();
			}
			break;
		case 19:
			// CTRL+S
			// Ignore repeated events when key is held down
			if (!aContext->keyRepeatFlag) {
				if (theCurrentProject>=0) {
					if (projects[theCurrentProject].projectPath) {
						saveStfToFilename(aContext, projects[theCurrentProject].projectPath, false, theCurrentProject);
					} else {
						saveFileSelect(aContext, false);
					}
				}
			}
			break;
		case 22:
			// CTRL+V
			// Ignore repeated events when key is held down
			if (!aContext->keyRepeatFlag) {
				loadFromClipboard();
			}
			break;
		case 24:
			// CTRL+X
			// Ignore repeated events when key is held down
			if (!aContext->keyRepeatFlag) {
				saveStfToClipboard();
				deleteSelected();
			}
			break;
		default:
			if (theCurrentProject>=0 && currentPatch>=0) {
				processSynFactory(aContext, theCurrentProject, currentPatch);
			} else {
				processTimeline(aContext);
			}
		}
//    processTimeline(aContext);

		break;
	case GUIEVENT_KEYDOWN: {
			switch(aContext->id) {
			case VK_F1:
				if (theCurrentProject>=0 && currentPatch>=0) {
					processSynFactory(aContext, theCurrentProject, currentPatch);
				} else {
					processTimeline(aContext);
				}
				break;
			case VK_F2:
				~cc~showProjectBrowser~
				break;
			default:
				if (theCurrentProject>=0 && currentPatch>=0) {
					processSynFactory(aContext, theCurrentProject, currentPatch);
				} else {
					processTimeline(aContext);
				}
				break;
			}
		} break;
	case GUIEVENT_CLOSE:
		quitFlag=true;
		break;
	default:
		break;
	}
}
EOF
}

sub genHelpWindow() {
rv(<<EOF);
const char aboutHelpText[]=
	"<P><FONT SIZE=+2><U>StudioFactory</U></FONT><BR>"
	"The desktop Audio/Video studio</P><HR>"
	"<P><B>Program version: </B>$theStudioFactoryVersion<BR>"
	"<P><B>Synthesizer version: </B>"THE_SYNFACTORY_VERSION"<BR>"
	"<P><B>Website: </B>http://www.syntiac.com/studiofactory.html<BR>"
	"<B>Feedback: </B>studio\@syntiac.com</P>"
	"<P><B>Software design and programming: </B>Peter Wendrich (pwsoft\@syntiac.com)<BR>"
	"<B>Minor changes: </B>Michael W. Brown (http://fractalmbrown.deviantart.com)<BR>"
	"<B>Thanx to: </B>Dave, Antonio, Patrick and Raymond for patches and ideas</P><HR>"
	"<P><B>License: </B>This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.</P>"
	"<P><B>Disclamer: </B>This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.</P>";
EOF

	rv("const char StudioFactoryHelp[]=\"StudioFactory Help\";");
	rv("const char * currentHelpPage=aboutHelpText;");

rc(<<EOF);
static bool displayHelp(const char *title, const char *helppage) {
	if (NULL!=onlineHelpWindow) {
		guiSetWindowTitle(onlineHelpWindow, (NULL!=title)?title:StudioFactoryHelp);
		currentHelpPage=helppage;
		guiShowWindow(onlineHelpWindow);
		guiVScrollWindowTo(onlineHelpWindow, 0);
		guiRefreshWindow(onlineHelpWindow);
		return true;
	}

	return false;
}

static bool displayHelp(const char *title, StringIndex helpstring) {
	if (strings[helpstring][theCurrentLanguage]) {
		return displayHelp(title, strings[helpstring][theCurrentLanguage]);
	}
	/* !!! needfix: Hard link to english as language 1 */
	return displayHelp(title, strings[helpstring][1]);
}

static void onlineHelpHandler(ContextPtr aContext) {
	switch (aContext->guiEvent) {
	case GUIEVENT_REFRESH: {
			int scrollPos=guiGetVScrollPos(aContext->currentWindow);
			int curY=-scrollPos;

			DrawEdge(aContext->hdc, &(aContext->guiClientRect), EDGE_SUNKEN, BF_MIDDLE);
			aContext->guiClientRect.left+=4;
			aContext->guiClientRect.right-=4;
			renderHTMLPage(aContext, &(aContext->guiClientRect), &curY, currentHelpPage);
			guiVScrollRange(aContext->currentWindow, 0, curY+scrollPos, aContext->guiClientRect.bottom);

		} break;
	case GUIEVENT_KEYDOWN:
		switch(aContext->id) {
		case VK_ESCAPE:
			guiHideWindow(aContext->currentWindow);
			break;
		default:
			break;
		} break;
	case GUIEVENT_CLOSE:
		guiHideWindow(aContext->currentWindow);
		break;
	default:
		break;
	}
}
EOF

	genWindow('onlineHelpWindow', 'HELPWINDOW', 'StudioFactoryHelp', 'onlineHelpHandler', undef, undef, 1, 0, undef);
}

##########################################################################
#
# Load/Save routines
#
##########################################################################

sub genLoadSave {
	ccModule('Load/Save');

rc(<<EOF);
//
// dest - buffer used to store the string
// maxlen - maximum length of the string (excluding the terminating '0') so dest must be maxlen+1 in size
// sf - if reading from file this is the file descriptor
// iBuffer - if reading from a character buffer this is a pointer to a pointer to the buffer.
// useComma - Due to backwards compatibility, some commands use comma seperated values.
//
// return value  0 no string read.
//               >0 single string read (returned value is string length).
//
static int readString(LoadInfo *loadInfo, char *dest, int maxlen, bool useComma) {
	char *tmpdest=dest;
	if (tmpdest) {
		int chr;
		if (loadInfo->file) {
			chr=getc(loadInfo->file);
			// Skip leading spaces and control chars
			while ((chr!=EOF) && ((chr<=32) || (chr==',' && useComma))) {
				chr=getc(loadInfo->file);
			}
			while ((chr!=EOF) && (chr>32) && (chr!=',' || (!useComma))) {
				if (maxlen>0) {
					*tmpdest=(char)chr;
					tmpdest++;
					maxlen--;
				}
				chr=getc(loadInfo->file);
			}
		}
		if (loadInfo->buffer) {
			// Skip leading spaces and control chars
			while ((loadInfo->buffer[loadInfo->bufferPos]>0) && (loadInfo->buffer[loadInfo->bufferPos]<=32 || (loadInfo->buffer[loadInfo->bufferPos]==',' && useComma))) {
				loadInfo->bufferPos++;
			}
			while (loadInfo->buffer[loadInfo->bufferPos]>32 && (loadInfo->buffer[loadInfo->bufferPos]!=',' || (!useComma))) {
				if (maxlen>0) {
					*tmpdest=loadInfo->buffer[loadInfo->bufferPos];          
					tmpdest++;
					maxlen--;
				}
				loadInfo->bufferPos++;
			}

			// Just like filebased method we consume last read character
			if (loadInfo->buffer[loadInfo->bufferPos] && (loadInfo->buffer[loadInfo->bufferPos]<=32 || (loadInfo->buffer[loadInfo->bufferPos]==',' && useComma))) {
				loadInfo->bufferPos++;
			}
		}

		*tmpdest='\\0';
		return strlen(dest);
	}
	return 0;
}

//
// Read integer from file or character buffer.
// returns 0 when no integer read.
//         1 when single integer read.
//
static int readInteger(LoadInfo *loadInfo, int *i, bool useComma) {
	char tmpbuf[32];
	if (readString(loadInfo, tmpbuf, 31, useComma)>0) {
		*i=atoi(tmpbuf);
		return 1;
	} else {
		*i=0;
	}
	return 0;
}

//
// Read single char from input file or character buffer.
// returns 0 when no character read.
//         1 when single character read.
//
static int readChar(LoadInfo *loadInfo, int *chr) {
	if (loadInfo->file) {
		*chr=getc(loadInfo->file);
		return 1;
	}
	if (loadInfo->buffer) {
		*chr=loadInfo->buffer[loadInfo->bufferPos++];
		return 1;
	}
	return 0;
}


typedef enum _SynFileSection {
	SECTION_NONE,
	SECTION_SETTINGS,
	SECTION_WINDOWS,
	SECTION_SAMPLES,
	SECTION_PATCHES
} SynFileSection;
static void logNotSynFile(void) {
	logprintf("Not a SynFactory 1.x (SYN or SYF) file.\\n");
}
static void loadSyn(LoadInfo *loadInfo) {
	SynFileSection section=SECTION_NONE;
	int offsetX=0, offsetY=0;
	int patch=currentPatch;
	int module=-1;
	bool createdProject=false;

	if (readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)<=0 ||
			strcmp(loadInfo->headerTitle, "FILETYPE")!=0 ||
			readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)<=0 ||
			strcmp(loadInfo->headerTitle, "SYN")!=0) {
		logNotSynFile();
		return;
	}
	if (readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)>0 &&
			strcmp(loadInfo->headerTitle, "PROGRAM")==0 &&
			readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)>0) {
		logprintf("File generated by '%s'.\\n", loadInfo->headerTitle);
	} else {
		logprintf("Expected 'PROGRAM'.\\n");
		logNotSynFile();
		return;
	}

	loadInfo->loadError=LOADERROR_OK;
	while (readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)>0) {
	if (strcmp(loadInfo->headerTitle, "{{{SETTINGS")==0) {
		section=SECTION_SETTINGS;
	}
	if (strcmp(loadInfo->headerTitle, "}}}SETTINGS")==0) {
		section=SECTION_NONE;
	}
	if (strcmp(loadInfo->headerTitle, "{{{WINDOWS")==0) {
		section=SECTION_WINDOWS;
	}
	if (strcmp(loadInfo->headerTitle, "}}}WINDOWS")==0) {
		section=SECTION_NONE;
	}
	if ((strcmp(loadInfo->headerTitle, "{{{SAMPLES")==0) || (strcmp(loadInfo->headerTitle, "{{{SAMPLEBANK")==0)) {
		section=SECTION_SAMPLES;
	}
	if ((strcmp(loadInfo->headerTitle, "}}}SAMPLES")==0)  || (strcmp(loadInfo->headerTitle, "}}}SAMPLEBANK")==0)) {
		section=SECTION_NONE;
	}
	if (strcmp(loadInfo->headerTitle, "{{{PATCHES")==0) {
		section=SECTION_PATCHES;
	}
	if (strcmp(loadInfo->headerTitle, "}}}PATCHES")==0) {
		patch=-1;
		module=-1;
		section=SECTION_NONE;
	}
	if (strcmp(loadInfo->headerTitle, "{{{PATCH")==0) {
		section=SECTION_PATCHES; // Pre 2.00 SYN files don't have an PATCHES section. We fake one when '{{{PATCH' is seen.
		if (projects==NULL || (!loadInfo->copyPasteFlag && !loadInfo->mergeFlag && !createdProject)) {
			createProject(newProjectName);
			createdProject=true;
			patch=-1;
		}
		if (patch<0) {
			patch=createSynfactoryPatch();
		}
		loadClearRewireTables(loadInfo);
	}
	if (strcmp(loadInfo->headerTitle, "}}}PATCH")==0) {
		loadRewire(loadInfo, patch);
		patch=-1;
		module=-1;
	}
	if (strcmp(loadInfo->headerTitle, "{{{MODULE")==0 && patch>=0) {
		ObjectType type=OBJ_NONE;
		char modname[32];
		int index=0;
		int i;

		readInteger(loadInfo, &index, true);
		readString(loadInfo, modname, 32, true);
		for(i=0;objectLoadNames[i];i++) {
			if (strcmp(objectLoadNames[i], modname)==0) {
				type=(ObjectType)i;
				break;
			}
		}
		// Try aliases
		if (type==OBJ_NONE) {
			if (strcmp("VCF", modname)==0) type=OBJ_OBVCF;
			if (modname[0]=='A' && modname[1]=='D' && modname[2]>'1' && modname[2]<'9' && modname[3]==0) type=OBJ_ADD;
			if (modname[0]=='A' && modname[1]=='N' && modname[2]=='D' && modname[3]>'2' && modname[4]<'9' && modname[5]==0) type=OBJ_AND;
			if (modname[0]=='O' && modname[1]=='R' && modname[2]>'2' && modname[2]<'9' && modname[3]==0) type=OBJ_OR;
		}
		if (type!=OBJ_NONE) {
			module=addSynFactoryObject(patch, type, false);
			if (module>=0 && projects[theCurrentProject].patches[patch].objects[module].dsp) {
				loadInfo->linkLoadId[projects[theCurrentProject].patches[patch].objects[module].dsp]=index;
				loadInfo->linkMyObject[projects[theCurrentProject].patches[patch].objects[module].dsp]=module;
			}
		}
	}
	if (strcmp(loadInfo->headerTitle, "}}}MODULE")==0) {
		module=-1;
	}
	if (strcmp(loadInfo->headerTitle, "OFFSET")==0 && module>=0 && patch>=0) {
		int x,y;
		readInteger(loadInfo, &x, true);
		readInteger(loadInfo, &y, true);
		projects[theCurrentProject].patches[patch].objects[module].x=x-offsetX;
		projects[theCurrentProject].patches[patch].objects[module].y=y-offsetY;
		
	}
	if (strcmp(loadInfo->headerTitle, "POT") == 0 && module>=0 && patch>=0) {
		int n,v;
		readInteger(loadInfo, &n, true);
		readInteger(loadInfo, &v, true);
		if (projects[theCurrentProject].patches[patch].objects[module].dsp && n>=0 && n<maxObjectIo) {
			projects[theCurrentProject].dsp[projects[theCurrentProject].patches[patch].objects[module].dsp].io[n].knob=v;
		}
	}
	if (strcmp(loadInfo->headerTitle, "INPUT") == 0 && module>=0 && patch>=0) {
		int i=-1,n,o;
		readInteger(loadInfo, &i, true);
		readInteger(loadInfo, &n, true);
		readInteger(loadInfo, &o, true);
		if (i>=0 && i<maxObjectIo) {
			loadInfo->linkCable[projects[theCurrentProject].patches[patch].objects[module].dsp][i]=n;
			projects[theCurrentProject].patches[patch].objects[module].connections[i].fromOutput=o+1;
			projects[theCurrentProject].patches[patch].objects[module].connections[i].cableColor=0;
		}
	}
	if (strcmp(loadInfo->headerTitle, "CABLECOLOR") == 0 && module>=0 && patch>=0) {
		int i,c;
		readInteger(loadInfo, &i, true);
		readInteger(loadInfo, &c, true);
		if (i>=0 && i<maxObjectIo) {
			projects[theCurrentProject].patches[patch].objects[module].connections[i].cableColor=c;
		}
	}
	if (strcmp(loadInfo->headerTitle, "MODULENAME") == 0 && module>=0 && patch>=0) {
		int l=0;
		int i;

		readInteger(loadInfo, &l, true);
		projects[theCurrentProject].patches[patch].objects[module].name=(char*)calloc(1,l+1);
		for(i=0;i<l;i++) {
			int c;
			readChar(loadInfo, &c);
			projects[theCurrentProject].patches[patch].objects[module].name[i]=(char)c;
		}
	}
	if (strcmp(loadInfo->headerTitle, "SCRIPT") == 0  && module>=0 && patch>=0 && projects[theCurrentProject].patches[patch].objects[module].objectType==OBJ_NOTESCRIPT) {
		int l=0;
		int i;

		lockMutex(theAudioMutex);
		readInteger(loadInfo, &l, true);
		projects[theCurrentProject].dsp[projects[theCurrentProject].patches[patch].objects[module].dsp].scriptInfo->script=(char*)calloc(1,l+1);
		for(i=0;i<l;i++) {
			int c;
			readChar(loadInfo, &c);
			projects[theCurrentProject].dsp[projects[theCurrentProject].patches[patch].objects[module].dsp].scriptInfo->script[i]=(char)c;
		}
		unlockMutex(theAudioMutex);
	}
	if (strcmp(loadInfo->headerTitle, "SIZE") == 0 && module>=0 && patch>=0) {
		int xs,ys;
		readInteger(loadInfo, &xs, true);
		readInteger(loadInfo, &ys, true);
		projects[theCurrentProject].patches[patch].objects[module].xs=xs;
		projects[theCurrentProject].patches[patch].objects[module].ys=ys;
	}
	if (strcmp(loadInfo->headerTitle, "COLOR") == 0 && module>=0 && patch>=0) {
		int m,b;
		readInteger(loadInfo, &m, true);
		readInteger(loadInfo, &b, true);

		if (m>0 && m<8) { m=cablePenColors[m]; } else if (m>=8 && m<16) { m=cableFillColors[m-8]; } else m=0;
		if (b>0 && b<8) { b=cablePenColors[b]; } else if (b>=8 && b<16) { b=cableFillColors[b-8]; } else b=0;
		projects[theCurrentProject].patches[patch].objects[module].mainColor=m;
		projects[theCurrentProject].patches[patch].objects[module].borderColor=b;
	}
	if (strcmp(loadInfo->headerTitle, "DRUM") == 0 && module>=0 && patch>=0) {
		int l;
		readInteger(loadInfo, &l, true);
		if ((projects[theCurrentProject].patches[patch].objects[module].objectType==OBJ_DRM4 && l==64) ||
				(projects[theCurrentProject].patches[patch].objects[module].objectType==OBJ_DRM8 && l==128)) {
			int i;

			for(i=0;i<l;i++) {
				int f;
				readChar(loadInfo, &f);
				((unsigned char *)projects[theCurrentProject].dsp[projects[theCurrentProject].patches[patch].objects[module].dsp].buffer)[i]=(unsigned char)((f=='1')?1:0);
			}
		} else {
			logprintf("Can't process DRUM %d command\\n", l);
		}
	}
	if (strcmp(loadInfo->headerTitle, "MODE") == 0 && module>=0 && patch>=0) {
		int mode;

		readInteger(loadInfo, &mode, false);
		setMode(theCurrentProject, patch, module, mode);
//    projects[theCurrentProject].patches[patch].objects[module].mode=mode;
	}

	~cc~synLoad~
	}
}


//
// Read buffer and optional compare header of this buffer to template.
//
bool readAndCompare(LoadInfo *loadInfo, void *buf, int len, const void *cmpstr, int cmplen) {
	readBuffer(loadInfo, (char *)buf, len);
	// compare data
	if (cmplen && (memcmp(buf, cmpstr, cmplen)!=0)) {
		return false;
	}
	return true;
}

static const unsigned char riffId[4]={'R','I','F','F'};
static const unsigned char waveId[4]={'W','A','V','E'};
static void loadWav(LoadInfo *loadInfo) {
	unsigned char tmp[256];
	int fileSize=0;

	if (!readAndCompare(loadInfo, tmp, 4, riffId, 4)) {
		logprintf("Not a WAV file. The RIFF header is missing.\\n");
		return;
	}
	if (readLittleEndianLong(loadInfo, &fileSize)) {
		logprintf("Data size specified in RIFF header %d (8 less as filesize in bytes).\\n", fileSize);
	} else {
		logprintf("Not a WAV file. The RIFF header length field is missing.\\n");
		return;
	}
	if (!readAndCompare(loadInfo, tmp, 4, waveId, 4)) {
		logprintf("Not a WAV file. The header was found, but is not of expected type WAVE.\\n");
		return;
	}
}
EOF

# executeFile('studiofactory_files_mid.pl');

rc(<<EOF);
static void loadTry(LoadInfo *loadInfo) {
	if (LOADERROR_OTHERFORMAT == loadInfo->loadError) {
		loadReset(loadInfo);
		loadSyn(loadInfo);
	}
	if (LOADERROR_OTHERFORMAT == loadInfo->loadError) {
		loadReset(loadInfo);
		loadStf(loadInfo);
	}
//	if (LOADERROR_OTHERFORMAT == loadInfo->loadError) {
//		loadReset(loadInfo);
//		loadMid(loadInfo);
//	}
	if (LOADERROR_OTHERFORMAT == loadInfo->loadError) {
		loadReset(loadInfo);
		loadWav(loadInfo);
	}
	switch(loadInfo->loadError) {
	case LOADERROR_OK:
		break;
	case LOADERROR_OTHERFORMAT:
		logprintf("loaderror: Unknown or unsupported file format.\\n");
		break;
	case LOADERROR_UNEXPECTED_EOF:
		logprintf("loaderror: Unexpected end of file.\\n");
		break;
	default:
		logprintf("loaderror: Unknown error.\\n");
		break;
	}
}

static LoadError loadFromFilename(ContextPtr aContext, const char *aFilename, bool merge) {
	LoadInfo loadInfo;

	memset(&loadInfo, 0, sizeof(LoadInfo));
	loadInfo.loadError=LOADERROR_FILENOTOPEN;
	loadInfo.file=fopen(aFilename, "rb");
	logprintf("loadFromFilename '%s'\\n", aFilename);
	if (loadInfo.file) {
		logprintf("File opened ok.\\n");
		loadInfo.loadError=LOADERROR_OTHERFORMAT;
		loadInfo.filename=aFilename;
		loadInfo.copyPasteFlag=false;
		loadInfo.mergeFlag=merge;
		loadTry(&loadInfo);
		fclose(loadInfo.file);
	} else {
		logprintf("Error opening file.\\n");
	}

	return loadInfo.loadError;
}

static LoadError loadFromBuffer(const unsigned char *buffer, size_t bufferSize) {
	LoadInfo loadInfo;

	memset(&loadInfo, 0, sizeof(LoadInfo));
	loadInfo.loadError=LOADERROR_OTHERFORMAT;
	loadInfo.buffer=(unsigned char *)buffer; // I promise no writing, but try telling the compiler that :-)
	loadInfo.bufferSize=bufferSize;
	loadInfo.copyPasteFlag=true;
	loadInfo.mergeFlag=false;
	loadTry(&loadInfo);

	return loadInfo.loadError;
}

static void saveStfSynFactoryPatch(LoadInfo *loadInfo, int patch) {
	bool save=true;
	bool saveAll=true;
	int i;
	
	if (loadInfo->saveSelectedOnly) {
		int i;

		save=false;
		saveAll=false;
		if (projects[loadInfo->project].patches[patch].selected) saveAll=true;
		for(i=0;i<projects[loadInfo->project].patches[patch].maxObjects;i++) {
			if (projects[loadInfo->project].patches[patch].objects[i].selected) {
				save=true;
				break;
			}
		}
	}
	if (save || saveAll) {
		if (loadInfo->generatePatches) {
			storeStfBegin(loadInfo, "PatchBegin");
			storeStfEnd(loadInfo);
			storeStfBegin(loadInfo, "PatchMidi");
			storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].midiPort);
			storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].midiChannel);
			storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].sustainCC);
			storeStfEnd(loadInfo);
		}
		for(i=0;i<projects[loadInfo->project].patches[patch].maxObjects;i++) {
			if (saveAll || projects[loadInfo->project].patches[patch].objects[i].selected) {
				int j;
				storeStfBegin(loadInfo, "ModuleBegin");
				storeStfInteger(loadInfo, i+1);
				storeStfString(loadInfo, objectLoadNames[projects[loadInfo->project].patches[patch].objects[i].objectType]);
				storeStfEnd(loadInfo);
				storeStfBegin(loadInfo, "ModulePosition");
				// Give +16,+16 offset when copy/pasting so cloning and copy/paste in same patch will not exactly position the modules on top of the existing ones.
				storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].x+(loadInfo->generatePatches?0:16)-projects[loadInfo->project].patches[patch].offsX);
				storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].y+(loadInfo->generatePatches?0:16)-projects[loadInfo->project].patches[patch].offsY);
				storeStfEnd(loadInfo);
				if (projects[loadInfo->project].patches[patch].objects[i].showLabel) {
					storeStfBegin(loadInfo, "ModuleLabel");
					storeStfInteger(loadInfo, 1);
					storeStfEnd(loadInfo);          
				}
				for(j=0;j<maxObjectIo;j++) {
					if (projects[loadInfo->project].patches[patch].objects[i].connections[j].fromObject) {
						storeStfBegin(loadInfo, "ModuleCable");
						storeStfInteger(loadInfo, j);
						storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].connections[j].fromObject);
						storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].connections[j].fromOutput);
						storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].connections[j].cableColor);
						storeStfEnd(loadInfo);
					}
				}
				for(j=0;j<maxObjectIo;j++) {
					if (knobName(projects[loadInfo->project].patches[patch].objects[i].objectType, j)) {
						storeStfBegin(loadInfo, "ModuleKnob");
						storeStfInteger(loadInfo, j);
						storeStfInteger(loadInfo, projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].io[j].knob);
						storeStfEnd(loadInfo);
					}
				}
				if (projects[loadInfo->project].patches[patch].objects[i].name) {
					storeStfBegin(loadInfo, "ModuleName");
					storeStfString(loadInfo, projects[loadInfo->project].patches[patch].objects[i].name);
					storeStfEnd(loadInfo);
				}
				if (projects[loadInfo->project].patches[patch].objects[i].objectType==OBJ_NOTESCRIPT &&
						projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].scriptInfo->script) {
					storeStfBegin(loadInfo, "ModuleScript");
					storeStfString(loadInfo, projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].scriptInfo->script);
					storeStfEnd(loadInfo);
				} else {
					if (objectBufferKeep(projects[loadInfo->project].patches[patch].objects[i].objectType)) {
						storeStfBegin(loadInfo, "ModuleBuffer");
						storeStfBuffer(loadInfo, (const char *)projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].buffer, objectBufferSize(projects[loadInfo->project].patches[patch].objects[i].objectType));
						storeStfEnd(loadInfo);
					}
				}
				if (projects[loadInfo->project].patches[patch].objects[i].dsp==0) {
					storeStfBegin(loadInfo, "ModuleSize");
					storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].xs);
					storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].ys);
					storeStfEnd(loadInfo);
					storeStfBegin(loadInfo, "ModuleColor");
					storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].mainColor);
					storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].borderColor);
					storeStfEnd(loadInfo);
				}
				if (projects[loadInfo->project].patches[patch].objects[i].mode) {
					storeStfBegin(loadInfo, "ModuleMode");
					storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].mode);
					storeStfEnd(loadInfo);
				}
				storeStfBegin(loadInfo, "ModuleEnd");
				storeStfEnd(loadInfo);
			}
		}
		if (loadInfo->generatePatches) {
			storeStfBegin(loadInfo, "PatchEnd");
			storeStfEnd(loadInfo);
		}
	}
}

static void saveStfProject(LoadInfo *loadInfo, bool savePath) {
	int i;

	if (!projects[loadInfo->project].selected && loadInfo->saveSelectedOnly) {
		int currentPatch=-1;

		// If selection is only in one patch or track, only the modules and objects are saved.
		// In all other cases everything below the project level is generated (but only for the ones necessary to contain selected items)
		for(i=0;i<projects[loadInfo->project].maxPatches && (!loadInfo->generatePatches);i++) {
			int j;
			if (projects[loadInfo->project].patches[i].selected) {
				loadInfo->generatePatches=true;
			} else {
				for(j=0;j<projects[loadInfo->project].patches[i].maxObjects;j++) {
					if (projects[loadInfo->project].patches[i].objects[j].selected) {
						if (currentPatch<0) {
							currentPatch=i;
						} else if (currentPatch!=i) {
							loadInfo->generatePatches=true;
							break;
						}
					}
				}
			}
		}
	} else {
		loadInfo->generatePatches=true;
		storeStfBegin(loadInfo, "ProjectBegin");
		storeStfString(loadInfo, projects[loadInfo->project].projectName);
		storeStfEnd(loadInfo);
	}
	if (savePath && projects[loadInfo->project].projectPath) {
		storeStfBegin(loadInfo, "ProjectPath");
		storeStfString(loadInfo, projects[loadInfo->project].projectPath);
		storeStfEnd(loadInfo);
	}
	for(i=0;i<projects[loadInfo->project].maxPatches;i++) {
		saveStfSynFactoryPatch(loadInfo, i);
	}
	if (!loadInfo->saveSelectedOnly) {
		storeStfBegin(loadInfo, "ProjectEnd");
		storeStfEnd(loadInfo);
	}
}

static const int saveNoProjects=-3;
static const int saveSelectedOnly=-2;
static const int saveAllProjects=-1;
static void saveStfData(LoadInfo *loadInfo, bool saveSettings) {
	storeStfBegin(loadInfo, "STUDIOFACTORY100");
	storeStfEnd(loadInfo);
	if (saveSettings) {
		~cc~stfSave~
	}
	if (loadInfo->project>=0) {
		if (loadInfo->project<maxProjects) {
			saveStfProject(loadInfo, false);
			if (loadInfo->filename && (loadInfo->filename!=projects[loadInfo->project].projectPath)) {
				setProjectPath(loadInfo->project, loadInfo->filename);
			}
		} else {
			logprintf("Error saveStfData project %d is invalid.\\n", loadInfo->project);
		}
	} else if (loadInfo->project==saveAllProjects) {
		int i;

		for(i=0;i<maxProjects;i++) {
			loadInfo->project=i;
			saveStfProject(loadInfo, saveSettings);
		}
	}
	storeStfBegin(loadInfo, "STUDIOFACTORYEND");
	storeStfEnd(loadInfo);
}

static LoadError saveStfToFilename(ContextPtr aContext, const char *aFilename, bool saveSettings, int aProject) {
	LoadInfo loadInfo;

	memset(&loadInfo, 0, sizeof(LoadInfo));
	loadInfo.loadError=LOADERROR_FILENOTOPEN;
	loadInfo.file=fopen(aFilename, "wb");
	if (loadInfo.file) {
		loadInfo.loadError=LOADERROR_OK;
		loadInfo.filename=aFilename;
		loadInfo.project=aProject;
		saveStfData(&loadInfo, saveSettings);
		fclose(loadInfo.file);
		if (0 <= aProject) {
			setProjectModifiedFlag(aProject, false);
		}
	}

	return loadInfo.loadError;
}
EOF

rv('LoadError loadFromClipboard(void);'); # !!! Forward declaration
rv('void saveStfToClipboard(void);');     # !!! Forward declaration
rv('void cloneSelected(void);');          # !!! Forward declaration

rc(<<EOF);
static LoadError loadFromClipboard(void) {
	LoadError error=LOADERROR_OTHERFORMAT;
	if (OpenClipboard(mainWindow)) {
		HANDLE xhandle=GetClipboardData(CF_PRIVATEFIRST);
		if (xhandle) {
			const unsigned char *buffer;
			logprintf("pasteFromClipboard binary format\\n");
			buffer=(const unsigned char *)GlobalLock(xhandle);
			error=loadFromBuffer(buffer, GlobalSize(xhandle));
			GlobalUnlock(xhandle);
		} else {
			xhandle=GetClipboardData(CF_TEXT);
			if (xhandle) {
				const unsigned char *buffer;
				logprintf("pasteFromClipboard text format\\n");
				buffer=(const unsigned char *)GlobalLock(xhandle);
				error=loadFromBuffer(buffer, GlobalSize(xhandle));
				GlobalUnlock(xhandle);
			}
		}
		CloseClipboard();
	}
	return error;
}

static void saveStfToClipboard(void) {
	LoadInfo loadInfo;

	if (OpenClipboard(mainWindow)) {
		EmptyClipboard();

		memset(&loadInfo, 0, sizeof(LoadInfo));
		loadInfo.saveSelectedOnly=true;
		loadInfo.project=theCurrentProject;
		saveStfData(&loadInfo, false);

		// Reserve buffer of required length
		clipboardHandle=GlobalAlloc(GHND, loadInfo.bufferPos);
		logprintf("copySelectedToClipboard: buflen=%d xhandle=%d\\n", loadInfo.bufferPos, (int)(void*)clipboardHandle);
		if (clipboardHandle) {
			void *dest=GlobalLock(clipboardHandle);

			loadInfo.bufferPos=0;
			loadInfo.buffer=(unsigned char *)dest;
			saveStfData(&loadInfo, false);
			GlobalUnlock(clipboardHandle);
			SetClipboardData(CF_PRIVATEFIRST, clipboardHandle);
			CloseClipboard();
		}
	}
}

static void cloneSelected(void) {
	LoadInfo loadInfo;

	memset(&loadInfo, 0, sizeof(LoadInfo));
	loadInfo.saveSelectedOnly=true;
	loadInfo.project=theCurrentProject;
	saveStfData(&loadInfo, false);

	loadInfo.buffer=(unsigned char *)malloc(loadInfo.bufferPos);
	loadInfo.bufferPos=0;
	saveStfData(&loadInfo, false);
	loadInfo.bufferSize=loadInfo.bufferPos;
	loadInfo.bufferPos=0;
	loadStf(&loadInfo);
	free(loadInfo.buffer);
}

static const char * const openFileFilter=
	"All supported types\\0*.669;*.di;*.dmf;*.mid;*.mod;*.mtm;*.okt;*.s3m;*.stf;*.stm;*.syf;*.syn;*.wav\\0"
	"669 - 669 modules\\0*.669\\0"
	"DI - Digital Illusion encoded mod-modules\\0*.di\\0"
	"DMF - DMF modules\\0*.dmf\\0"
	"MID - Standard midi files\\0*.mid\\0"
	"MOD - ProTracker/NoiseTracker modules\\0*.mod\\0"
	"MTM - MultiTracker Modules\\0*.mtm\\0"
	"OKT - Oktalyzer modules\\0*.okt\\0"
	"S3M - ScreamTracker 3.x modules\\0*.s3m\\0"
	"STF - StudioFactory Files\\0*.stf\\0"
	"STM - ScreamTracker 1.0 modules\\0*.stm\\0"
	"SYN - SynFactory 1.x Files\\0*.syf;*.syn\\0"
	"WAV - Wavefiles\\0*.wav\\0"
	"All files\\0*.*\\0"
	"\\0";
static const char * const SaveFileFilter=
	"STF - StudioFactory File\\0*.stf\\0"
	"\\0";

//  "SYN - SynFactory 1.x File\\0*.syn\\0"

static const char * const captureFileFilter=
	"WAV - Wavefiles\\0*.wav\\0"
	"MP3 - MP3 files\\0*.mp3\\0"
	"\\0";

#define MAXFILEPATH               1024
#define MAXFILETITLE               512
static char openFilePath[MAXFILEPATH];
static char openFileTitle[MAXFILETITLE];
static int lastOpenFormat=0;
static void openFileSelect(ContextPtr aContext, bool merge) {
	OPENFILENAME ofn;

	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = mainWindow;
	ofn.hInstance = NULL;
	ofn.lpstrFilter = openFileFilter;
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter = 0;
	ofn.nFilterIndex = lastOpenFormat;
	ofn.lpstrFile = openFilePath;
	ofn.nMaxFile = MAXFILEPATH;
	ofn.lpstrFileTitle = openFileTitle;
	ofn.nMaxFileTitle = MAXFILETITLE;
	ofn.lpstrInitialDir = NULL;
	ofn.lpstrTitle = NULL;
	ofn.Flags = OFN_NOCHANGEDIR | OFN_EXPLORER | OFN_HIDEREADONLY;
	ofn.nFileOffset = 0;
	ofn.nFileExtension = 0;
	ofn.lpstrDefExt = "syn";
	ofn.lCustData = NULL;
	ofn.lpfnHook = NULL;
	ofn.lpTemplateName = NULL;
	if (GetOpenFileName(&ofn)) {
		lastOpenFormat=ofn.nFilterIndex;
		loadFromFilename(aContext, openFilePath, merge);
	}
}

static void saveFileSelect(ContextPtr aContext, bool saveSettingsOnly) {
	static char saveFilePath[MAXFILEPATH];
	if ((!saveSettingsOnly) && theCurrentProject>=0 && projects[theCurrentProject].projectPath) {
		strncpy(saveFilePath, projects[theCurrentProject].projectPath, MAXFILEPATH);
		saveFilePath[MAXFILEPATH-1]='\\0';
	} else {
		saveFilePath[0]='\\0';
	}
	OPENFILENAME ofn;

	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = mainWindow;
	ofn.hInstance = NULL;
	ofn.lpstrFilter = SaveFileFilter;
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter = 0;
	ofn.nFilterIndex = 0;
	ofn.lpstrFile = saveFilePath;
	ofn.nMaxFile = MAXFILEPATH;
	ofn.lpstrFileTitle = NULL;
	ofn.nMaxFileTitle = 0;
	ofn.lpstrInitialDir = NULL;
	ofn.lpstrTitle = NULL;
	ofn.Flags = OFN_OVERWRITEPROMPT | OFN_NOCHANGEDIR | OFN_HIDEREADONLY;
	ofn.nFileOffset = 0;
	ofn.nFileExtension = 0;
	ofn.lpstrDefExt = "stf";
	ofn.lCustData = NULL;
	ofn.lpfnHook = NULL;
	ofn.lpTemplateName = NULL;
	if (GetSaveFileName(&ofn)) {
		saveStfToFilename(aContext, saveFilePath, saveSettingsOnly, (saveSettingsOnly)?saveNoProjects:theCurrentProject);
	}
}

static char currentCaptureFilePath[MAXFILEPATH];
static char currentCaptureFileTitle[MAXFILETITLE];
static int lastCaptureFormat=1;
static bool captureFileSelect(void) {
	OPENFILENAME ofn;

	if (NULL==captureSample.file) {
		ofn.lStructSize = sizeof(OPENFILENAME);
		ofn.hwndOwner = mainWindow;
		ofn.hInstance = NULL;
		ofn.lpstrFilter = captureFileFilter;
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter = 0;
		ofn.nFilterIndex = lastCaptureFormat;
		ofn.lpstrFile = currentCaptureFilePath;
		ofn.nMaxFile = MAXFILEPATH;
		ofn.lpstrFileTitle = currentCaptureFileTitle;
		ofn.nMaxFileTitle = MAXFILETITLE;
		ofn.lpstrInitialDir = NULL;
		ofn.lpstrTitle = NULL;
		ofn.Flags = OFN_OVERWRITEPROMPT | OFN_NOCHANGEDIR | OFN_HIDEREADONLY;
		ofn.nFileOffset = 0;
		ofn.nFileExtension = 0;
		ofn.lpstrDefExt = "wav";
		ofn.lCustData = NULL;
		ofn.lpfnHook = NULL;
		ofn.lpTemplateName = NULL;

		if (GetSaveFileName(&ofn)) {
			lastCaptureFormat=ofn.nFilterIndex;
			switch(ofn.nFilterIndex) {
			case 1:
				captureSample.encoding=SAMPLE_ENCODING_WAV;
				break;
			case 2:
				captureSample.encoding=SAMPLE_ENCODING_MP3;
				break;
			default:
				captureSample.encoding=SAMPLE_ENCODING_WAV;
				break;
			}
			captureSample.sampleRate=44100;
			captureSample.noofChannels=2;
			captureSample.noofBits=16;
			createSampleFile(&captureSample, currentCaptureFilePath);
		}
	}
	return (captureSample.file!=NULL);
}
EOF

	rd('static bool captureFileSelect(void);'); # !!! forward decl
	rv('HANDLE clipboardHandle=NULL;');
}

sub fixCC {
	my $buffer = shift;
	my $count;

	do {
		$count = 0;
		$buffer =~ s/(\s+)~cc~(.+?)~/my$t=$1;my$cc=$cc{$2};$t=~s|\n\n|\n|g;$cc.='';$cc=~s|\n(.+)|$count++;$t.$1|ge;$cc/ge;
		print "fixCC $count strings replaced\n";
	} while ($count > 0);
	return $buffer;
}

##########################################################################
#
# Sub-components
#
##########################################################################
# genSynTextEditor();

useModule('maker_support');
useModule('studiofactory_ossupport');
useModule('support_expand_string');
useModule('support_logfile');
# push(@theTargetModules, 'studiofactory_stubs');
useModule('studiofactory_fonts');
useModule('studiofactory_gui');
useModule('studiofactory_languages');
useModule('studiofactory_menu');
useModule('studiofactory_dsp');
useModule('studiofactory_dspopt');
useModule('studiofactory_project');
useModule('studiofactory_mainwindow');
useModule('studiofactory_widgets');
useModule('studiofactory_4chscope');
useModule('studiofactory_joystick');
useModule('studiofactory_html_viewer');
useModule('studiofactory_audio');
useModule('studiofactory_midi');
useModule('studiofactory_config_panel');
useModule('studiofactory_helpwindow');
useModule('studiofactory_colorselector');
useModule('studiofactory_transport');
useModule('studiofactory_outputscope');
useModule('studiofactory_mixer');
useModule('studiofactory_midi_activity');
useModule('studiofactory_midi_scope');
useModule('studiofactory_virtualkeyboard');
useModule('studiofactory_sequencer');
useModule('studiofactory_synfactory_patches');
useModule('studiofactory_files_mp3');
useModule('studiofactory_files_ogg');
useModule('studiofactory_files_syn');
useModule('studiofactory_files_stf');
useModule('studiofactory_files');
useModule('studiofactory_samples');
useModule('studiofactory_project_browser');
useModule('studiofactory_context');
useModule('studiofactory_mainwindowproc');
useModule('studiofactory_main');

useModule('studiofactory_executable');

# old stuff

##########################################################################
#
# Build system
#
##########################################################################

#my $commandlineOs=shift;
#my $commandlineGraphics=shift;
#my $commandlineAudio=shift;

#if (!defined($commandlineOs) || (lc $commandlineOs) eq 'mingw') {
	# !!!TODO: remove
#	$platformOs = 'Win32';
#	$platformGraphics = 'GDI';

#	genAll('studiofactory_mingw32_gdi.cpp', 'win32', 'gdi', 'x86');
#	system('windres -o studiofactoryrc.o studiofactory.rc');
#	system('gcc -Wall -Wwrite-strings -O3 -march=i586 -mwindows -mno-cygwin -fno-rtti -fno-exceptions -o studiofactory_mingw32_gdi.exe studiofactoryrc.o studiofactory_mingw32_gdi.cpp -lwinmm -lwsock32');
#	system('strip studiofactory_mingw32_gdi.exe');
#} 

#if (defined($commandlineOs) && (lc $commandlineOs) eq 'linux') {
#	# !!!TODO: remove
#	$platformOs = 'Linux';
#	$platformGraphics = 'Qt';


#	genAll('studiofactory_linux_qt.cpp');
#	genLinuxTemp('studiofactory_linux_qt.cpp', 'linux', 'qt', 'alsa');
#	system("gcc -pipe -O3 -march=i686 -fno-rtti -fno-exceptions -Wall -Wextra -Wwrite-strings -I$ENV{QTDIR}/include -o studiofactory_linux_qt studiofactory_linux_qt.cpp $ENV{QTDIR}/lib/libqt.so");

#	system("gcc -pipe -g -march=i686 -fno-rtti -fno-exceptions -Wall -Wextra -Wwrite-strings -I$ENV{QTDIR}/include -o studiofactory_linux_qt studiofactory_linux_qt.cpp $ENV{QTDIR}/lib/libqt.so");

#	$platformGraphics='Gtk';
#	genAll('studiofactory_linux_gtk.cpp');
#	system("gcc -Wall -Wwrite-strings -O3 -march=i686 -fno-rtti -fno-exceptions -I$ENV{QTDIR}/include -o studiofactory_linux_qt studiofactory_linux_qt.cpp $ENV{QTDIR}/lib/libqt.so");
#}

