##########################################################################
#
# StudioFactory
#
# The desktop Audio/Video studio.
# Copyright (C) 2002-2004  Peter Wendrich (pwsoft@syntiac.com)
# Homepage: http://www.syntiac.com/studiofactory.html
#
##########################################################################
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
##########################################################################

use strict;
use warnings;
use Fcntl;


my $studioFactoryVersion = '0.30';


my $platformOS;          # Win32, Unix
my $platformGraphics;    # GDI, Gtk, Xt, X, SDL, LinuxFB ...
my $platformVideo;       # Win32
my $platformAudio;       # Win32, Alsa
my $ri='';               # includes
my $rd='';               # definitions
my $rv='';               # variables
my $rs='';               # support code
my $rc='';               # code
my @refreshFunctions=(); # window refresh
my @initFunctions=();    # startup code
my @termFunctions=();    # shutdown code
my @initWindows=();      # windows related startup code
my @termWindows=();      # windows related shutdown code
my @synLoad=();
my @synSave=();
my @stfLoad=();
my @stfSave=();

sub genTable {
  my $type=shift;
  my $name=shift;
  my $list=shift;
  my $name2=shift;
  my $i;

  if (!defined($name2)) {
    $name2='';
  }

  $rv .= "static $type $name\[" . scalar(@$list) . "]$name2 = {\n";
  for($i=0;$i<scalar(@$list);$i++) {
    if ($i>0) {
      $rv .= ",\n";
    }
    $rv .= "  " . $$list[$i];

  }
  $rv .= "\n};\n";
}

sub genEnumStruct {
  my $type=shift;
  my $name=shift;
  my $list=shift;
  my $seperator=shift;
  my $i;

  if (!defined($seperator)) {
    $seperator='';
  }

  $rd .= "typedef $type _$name {\n";
  for($i=0;$i<scalar(@$list);$i++) {
    if ($i>0) {
      $rd .= "$seperator\n";
    }
    $rd .= "  " . $$list[$i];

  }
  $rd .= "\n} $name, *${name}Ptr;\n";
}

sub genStringFromFile {
  my $stringName = shift;
  my $fileName = shift;

  $rv .= "static const char $stringName\[\]=";
  open(FILE, $fileName) || die "Can't open $fileName";
  while(<FILE>) {
    s/\r//;
    s/\n//;
    if (length($_)>0) {
      $rv .= "\n\"$_\"";
    }
  }
  close(FILE);
  $rv .= ";\n";
}

#
# Read a file from disk and return it as a string.
# input
#   filename - name of file to read
# output
#   returns string with file contents.
sub readCompleteFile {
  my $filename = shift; 
  my $r;
  local $/;

  open(FILE, $filename) || die "can't open $filename";
  $r=<FILE>;
  close(FILE);

  return $r;
}

#
# Read a file from disk and execute as perl script.
# input
#   filename - name of file to read
# output
#   none
sub executeFile {
  my $code=readCompleteFile(shift);
  eval $code;
  die $@ if $@;
}

sub include {
  $ri .= "#include " . (shift) . "\n";
}

sub rd {
  $rd .= (shift) . "\n";
}

sub rv {
  $rv .= "static " . (shift) . "\n";
}

sub rc {
  $rc .= (shift) . "\n";
}

sub mutex {
  my $mutexName=shift;

  if ($platformOS eq 'Win32') {
    rv("CRITICAL_SECTION $mutexName;");
    push(@initFunctions, "InitializeCriticalSection(&$mutexName);");
    push(@termFunctions, "DeleteCriticalSection(&$mutexName);");
  }
  if ($platformOS eq 'Unix') {
    rv("pthread_mutex_t $mutexName;");
    push(@initFunctions, "pthread_mutex_init(&$mutexName, NULL);");
    push(@termFunctions, "pthread_mutex_destroy(&$mutexName);");
  }
}

sub genSection {
  rc('');
  rc('// ///////////////////////////////////////////////////////////////////////');
  rc('//');
  rc("// " . (shift));
  rc('//');
  rc('// ///////////////////////////////////////////////////////////////////////');
}

my @languageCodes=();
my $noofLanguages=1;
sub genLanguages {
  my @languageEnum=();
  my @languageCodesStr=();

  @languageCodes=();

  open(LANGUAGES, 'languages.txt') || die "Can't open languages.txt";
  while(<LANGUAGES>) {
    if (m/(.+?)\s+([a-zA-Z]+)/) {
      push(@languageEnum, $1);
      push(@languageCodes, $2);
      push(@languageCodesStr, "\"$2\"");
    }
  }
  close(LANGUAGES);

  $noofLanguages=scalar(@languageEnum);
#  genEnumStruct("enum", "Language", \@languageEnum, ',');
  push(@languageCodesStr, 'NULL');
  genTable("const char * const", "languageCodes", \@languageCodesStr);
  $rd .= "static const int noofLanguages=$noofLanguages;\n";
  
  $rc .= <<EOF;
static void setLanguageIndex(int languageIndex) {
  currentLanguage=languageIndex;
  SetMenu(mainWindow, mainMenu[currentLanguage]);
  configWindowRefresh=true;
}
static void setLanguageCode(const char *languageCode) {  
  int i;

  currentLanguage=0;
  for(i=0;languageCodes[i];i++) {
    if (strcmp(languageCode, languageCodes[i])==0) {
      setLanguageIndex(i);
      break;
    }    
  }
}
EOF

  rv('int currentLanguage;');
  push(@stfLoad, 'if (strcmp("ProgramLanguage", loadInfo->headerTitle)==0) { char languageCode[4]; readStfString(loadInfo, languageCode, 4); setLanguageCode(languageCode); }');
  push(@stfSave, 'storeStfBegin(loadInfo, "ProgramLanguage"); storeStfString(loadInfo, languageCodes[currentLanguage]); storeStfEnd(loadInfo);');

}

sub genGuiMessages {
  my @enumdef=qw(
    GUIEVENT_NONE GUIEVENT_REFRESH GUIEVENT_RESIZE GUIEVENT_GETFOCUS GUIEVENT_LOSTFOCUS
    GUIEVENT_SHOWWINDOW GUIEVENT_HIDEWINDOW GUIEVENT_CLOSE GUIEVENT_DESTROY
    GUIEVENT_TIMERTICK GUIEVENT_MENU
    GUIEVENT_MOUSEBUTTON1DOWN GUIEVENT_MOUSEBUTTON1UP GUIEVENT_MOUSEBUTTON1DCLK
    GUIEVENT_MOUSEBUTTON2DOWN GUIEVENT_MOUSEBUTTON2UP GUIEVENT_MOUSEBUTTON2DCLK
    GUIEVENT_KEYDOWN GUIEVENT_KEYUP GUIEVENT_CHAR );
  genEnumStruct("enum", "GuiEvent", \@enumdef, ',');
}

sub genContext {
  my @structdef=();
  if ($platformOS eq 'Win32') {
    $rs .= <<EOF;
#define lockMutex(object) EnterCriticalSection(&object)
#define unlockMutex(object) LeaveCriticalSection(&object)
EOF
  }
  if ($platformOS eq 'Unix') {
    $rs .= <<EOF;
#define lockMutex(object) pthread_mutex_lock(&object)
#define unlockMutex(object) pthread_mutex_unlock(&object)
EOF
  }
  if ($platformGraphics eq 'GDI') {
    $rd .= "typedef RECT GuiRect;\n";
    $rd .= "typedef HWND GuiWindow;\n";
    push(@structdef, 'GuiWindow currentWindow;');
    push(@structdef, 'HDC hdc;');
  }
  if ($platformGraphics eq 'Xt') {
    $rd .= "typedef struct { int left; int top; int right; int bottom; } GuiRect;";
    push(@structdef, 'Window currentWindow;');
  }
  push(@structdef, 'GuiRect guiClientRect;');
  push(@structdef, 'GuiEvent guiEvent;');
  push(@structdef, 'int mouseClientX;');
  push(@structdef, 'int mouseClientY;');
  push(@structdef, 'int mouseX;');
  push(@structdef, 'int mouseY;');
  push(@structdef, 'int id;');
  push(@structdef, 'bool keyRepeatFlag;');
  push(@structdef, 'void *userData;');
  genEnumStruct("struct", "Context", \@structdef);
  $rd .= "typedef void (*EventFunc)(ContextPtr contextPtr);\n";
}

##########################################################################
#
# Menu
#
##########################################################################
my @menuEnum;
sub genMenuDef {
  my $menuname = shift;
  my $filename = shift;
  my @menuStruct = ();

  open(MENU, $filename) || die "Can't open $filename";
  while(<MENU>) {
    s/\r//;
    s/\n//;
    if (!m/^\#/) {
      if (m/^\+\s+(.+)/) {
        push(@menuStruct, "-1, {$1}");
      } elsif (m/^\-/) {
        push(@menuStruct, "-2, {NULL}");
      } elsif (m/^\./) {
        push(@menuStruct, "0, {NULL}");
      } elsif (m/^(.+?)\s+(.+)/) {
        push(@menuEnum, $1);
        push(@menuStruct, "$1, {$2}");
      }
    }
  }
  close(MENU);

  genTable("MenuDef", $menuname, \@menuStruct);
}

sub genMenus {
  my @structdef=('int menuItemType;','const char *menuItemName[noofLanguages];');
  genEnumStruct("struct", "MenuDef", \@structdef);

  @menuEnum=('MENU_NONE');
  my @empty=();
  genMenuDef('mainMenuDef', 'studiofactory_mainmenu.txt');
  genEnumStruct("enum", "MenuEnum", \@menuEnum, ',');

  $rc .= <<EOF;
static HMENU makeMenuHelper(MenuDefPtr *menuDef1, MenuDefPtr *menuDef2, int language, bool popupMenu) {
  HMENU myMenu;
  
  if (popupMenu) {
    myMenu=CreatePopupMenu();
  } else {
    myMenu=CreateMenu();
  }

  while (menuDef1) {
    while((*menuDef1)->menuItemType!=0) {
      switch((*menuDef1)->menuItemType) {
      case -2:
        AppendMenu(myMenu, MF_SEPARATOR, 0, NULL);
        (*menuDef1)++;
        break;
      case -1: {
          HMENU subMenu;
          const char *subMenuName=(*menuDef1)->menuItemName[language];

          if (NULL==subMenuName) subMenuName=(*menuDef1)->menuItemName[0];
          (*menuDef1)++;
          subMenu=makeMenuHelper(menuDef1, NULL, language, false);
          AppendMenu(myMenu, MF_POPUP, (UINT)subMenu, subMenuName);
        } break;
      default:
        const char *subMenuName=(*menuDef1)->menuItemName[language];

        if (NULL==subMenuName) subMenuName=(*menuDef1)->menuItemName[0];
        AppendMenu(myMenu, MF_STRING, (MenuEnum)((*menuDef1)->menuItemType), subMenuName);
        (*menuDef1)++;
        break;
      }
    }
    (*menuDef1)++;
    menuDef1=menuDef2;
    menuDef2=NULL;
  }

  return myMenu;
}

static HMENU makeMenu(MenuDefPtr menuDef1, MenuDefPtr menuDef2, int language, bool popupMenu) {
  MenuDefPtr myPtr1=menuDef1;
  MenuDefPtr myPtr2=menuDef2;

  if (myPtr2) {
    return makeMenuHelper(&myPtr1, &myPtr2, language, popupMenu);
  } else {
    return makeMenuHelper(&myPtr1, NULL, language, popupMenu);
  }
}

static void markMenuItem(HMENU *menu, int id, bool checkmark) {
  int i;

  for(i=0;i<noofLanguages;i++) {
    CheckMenuItem(menu[i], id, MF_BYCOMMAND | ((checkmark)?MF_CHECKED:MF_UNCHECKED));
  }
}

EOF

  rv('HMENU mainMenu[noofLanguages];');
  push(@initFunctions, "for(i=0;languageCodes[i];i++) { mainMenu[i] = makeMenu(mainMenuDef, NULL, i, false); }");
  push(@termFunctions, "for(i=0;languageCodes[i];i++) { DestroyMenu(mainMenu[i]); }");
}



sub genSystemColorsGdi {
#  my $nrColorsPlus = $nrColors+1;
  my $colorstr;
  my @colors=();

  rv('const int colorSteps=16;');
  rv('const int noofSystemColors=colorSteps*colorSteps*colorSteps;');

# original synfactory colors
#  rv('int defcolor_patchBackground=0333;');
#  rv('int defcolor_patchModules=0555;');
#  rv('int defcolor_patchKnobs=0030;');
#  rv('int defcolor_patchHighlight=0777;');
#  rv('int defcolor_patchShadow=0000;');

# studiofactory colors (Alice)
  rv('int defcolor_patchBackground=0x668888;');
  rv('int defcolor_patchModules   =0xAACCCC;');
  rv('int defcolor_patchKnobs     =0x66AAAA;');
  rv('int defcolor_patchHighlight =0xCCEEEE;');
  rv('int defcolor_patchShadow    =0x000000;');

  rv('HPEN currentPen=NULL;');
  rv('HBRUSH currentBrush=NULL;');


$rs .= <<EOF;

static void guiSelectPen1Color(ContextPtr aContext, int color) {
  HPEN newPen=NULL;
  if (color>=0 && color<0x1000000) {
    newPen=CreatePen(PS_SOLID, 1, 0x02000000 | color);
    SelectObject(aContext->hdc, newPen);
    SetTextColor(aContext->hdc, 0x02000000 | color);
  } else {
    SelectObject(aContext->hdc, GetStockObject(NULL_PEN));
  }
  if (currentPen) DeleteObject(currentPen);
  currentPen=newPen;
}

static void guiSelectPen3Color(ContextPtr aContext, int color) {
  HPEN newPen=NULL;
  if (color>=0 && color<0x1000000) {
    newPen=CreatePen(PS_SOLID, 3, 0x02000000 | color);
    SelectObject(aContext->hdc, newPen);
    SetTextColor(aContext->hdc, 0x02000000 | color);
  } else {
    SelectObject(aContext->hdc, GetStockObject(NULL_PEN));
  }
  if (currentPen) DeleteObject(currentPen);
  currentPen=newPen;
}

static void guiSelectFillColor(ContextPtr aContext, int color) {
  HBRUSH newBrush=NULL;
  if (color>=0 && color<0x1000000) {
    newBrush=CreateSolidBrush(0x02000000 | color);
    SelectObject(aContext->hdc, newBrush);
    SetBkColor(aContext->hdc, 0x02000000 | color);
  } else {
    SelectObject(aContext->hdc, GetStockObject(NULL_BRUSH));
  }
  if (currentBrush) DeleteObject(currentBrush);
  currentBrush=newBrush;
}
EOF

#  push(@initFunctions, "initColors();");

  push(@stfLoad, 'if (strcmp("DefColorPatch", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) defcolor_patchBackground=c; }');
  push(@stfSave, 'storeStfBegin(loadInfo, "DefColorPatch"); storeStfInteger(loadInfo, defcolor_patchBackground); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp("DefColorPatchModules", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) defcolor_patchModules=c; }');
  push(@stfSave, 'storeStfBegin(loadInfo, "DefColorPatchModules"); storeStfInteger(loadInfo, defcolor_patchModules); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp("DefColorPatchKnobs", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) defcolor_patchKnobs=c; }');
  push(@stfSave, 'storeStfBegin(loadInfo, "DefColorPatchKnobs"); storeStfInteger(loadInfo, defcolor_patchKnobs); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp("DefColorPatchHighlight", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) defcolor_patchHighlight=c; }');
  push(@stfSave, 'storeStfBegin(loadInfo, "DefColorPatchHighlight"); storeStfInteger(loadInfo, defcolor_patchHighlight); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp("DefColorPatchShadow", loadInfo->headerTitle)==0) { int c; readStfInteger(loadInfo, &c); if (c>=0 && c<0x1000000) defcolor_patchShadow=c; }');
  push(@stfSave, 'storeStfBegin(loadInfo, "DefColorPatchShadow"); storeStfInteger(loadInfo, defcolor_patchShadow); storeStfEnd(loadInfo);');

  push(@termFunctions, "if (currentPen) DeleteObject(currentPen);");
  push(@termFunctions, "if (currentBrush) DeleteObject(currentBrush);");
}


sub genDrawGdi {
$rs .= <<EOF;
static void guiDrawLine(ContextPtr context, GuiRect *rect) {
  (void)MoveToEx(context->hdc, rect->left, rect->top, NULL);
  (void)LineTo(context->hdc, rect->right, rect->bottom);
}

static void guiDrawRect(ContextPtr context, int left, int top, int right, int bottom) {
  (void)Rectangle(context->hdc, left, top, right+1, bottom+1);
}

static void guiDrawRoundRect(ContextPtr context, GuiRect *rect, int width, int height) {
  (void)RoundRect(context->hdc, rect->left, rect->top, rect->right, rect->bottom, width, height);
}

static void guiDrawRoundRect(ContextPtr context, int left, int top, int right, int bottom, int width, int height) {
  (void)RoundRect(context->hdc, left, top, right, bottom, width, height);
}

static void guiDrawEllipse(ContextPtr context, GuiRect *rect) {
  (void)Ellipse(context->hdc, rect->left, rect->top, rect->right, rect->bottom);
}

static void guiDrawText(ContextPtr context, int left, int top, int right, int bottom, const char *text, int count) {
  RECT myRect={left, top, right, bottom};
  DrawText(context->hdc, text, count, &myRect, DT_NOPREFIX | DT_WORDBREAK);
}

static void guiDrawText(ContextPtr context, int left, int top, const char *text, int count) {
  RECT myRect={left, top, left, top};
  DrawText(context->hdc, text, count, &myRect, DT_NOPREFIX | DT_NOCLIP);
}

void guiDraw3DRect(ContextPtr aContext, GuiRect *rect, int edge) {
  if (currentBrush) {
    (void)FillRect(aContext->hdc, rect, currentBrush);
    DrawEdge(aContext->hdc, rect, edge, BF_RECT);
  } else {
    DrawEdge(aContext->hdc, rect, edge, BF_RECT | BF_MIDDLE);
  }
}

static void guiRefreshWindow(GuiWindow window, GuiRect *rect)  {
  (void)InvalidateRect(window, (RECT*)rect, FALSE);
}

static void guiShowWindow(GuiWindow window) {
  (void)ShowWindow(window, SW_SHOW);
  (void)SetFocus(window);
}

static void guiHideWindow(GuiWindow window) {
  (void)ShowWindow(window, SW_HIDE);
  guiShowWindow(mainWindow);
}

static void guiSetWindowTitle(GuiWindow window, const char *title) {
  SetWindowText(window, title);
}

static void guiVScrollWindowTo(GuiWindow window, int pos) {
  SCROLLINFO myScrollInfo;
  myScrollInfo.cbSize=sizeof(myScrollInfo);
  myScrollInfo.fMask=SIF_POS | SIF_TRACKPOS | SIF_PAGE | SIF_RANGE;
  GetScrollInfo(window, SB_VERT, &myScrollInfo);

  myScrollInfo.nPos=pos;

  if (myScrollInfo.nPos<myScrollInfo.nMin) myScrollInfo.nPos=myScrollInfo.nMin;
  if (myScrollInfo.nPos>myScrollInfo.nMax) myScrollInfo.nPos=myScrollInfo.nMax;
  myScrollInfo.cbSize=sizeof(myScrollInfo);
  myScrollInfo.fMask=SIF_POS;
  SetScrollInfo(window, SB_VERT, &myScrollInfo, TRUE);
  guiRefreshWindow(window, NULL);
}

static void guiVScrollRange(GuiWindow window, int min, int max, int page) {
  SCROLLINFO myScrollInfo;
  myScrollInfo.cbSize=sizeof(myScrollInfo);
  myScrollInfo.fMask=SIF_POS | SIF_PAGE | SIF_RANGE;
  GetScrollInfo(window, SB_VERT, &myScrollInfo);
  myScrollInfo.nMin=min;
  myScrollInfo.nMax=max;
  myScrollInfo.nPage=page;

  if (myScrollInfo.nPos<myScrollInfo.nMin) myScrollInfo.nPos=myScrollInfo.nMin;
  if (myScrollInfo.nPos>myScrollInfo.nMax) myScrollInfo.nPos=myScrollInfo.nMax;
  myScrollInfo.cbSize=sizeof(myScrollInfo);
  myScrollInfo.fMask=SIF_POS | SIF_PAGE | SIF_RANGE | SIF_DISABLENOSCROLL;
  SetScrollInfo(window, SB_VERT, &myScrollInfo, TRUE);
}

static int guiGetVScrollPos(GuiWindow window) {
  SCROLLINFO myScrollInfo;

  myScrollInfo.cbSize=sizeof(myScrollInfo);
  myScrollInfo.fMask=SIF_POS;
  GetScrollInfo(window, SB_VERT, &myScrollInfo);
  return myScrollInfo.nPos;
}

static void guiSetVScrollPos(GuiWindow window, int pos) {
  SCROLLINFO myScrollInfo;

  myScrollInfo.cbSize=sizeof(myScrollInfo);
  myScrollInfo.fMask=SIF_POS;
  myScrollInfo.nPos=pos;
  SetScrollInfo(window, SB_VERT, &myScrollInfo, TRUE);
  guiRefreshWindow(window, NULL);
}

static void guiClearAreaList(ContextPtr aContext) {
  if (GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALIST)) {
    free((void*)GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALIST));
    SetWindowLong(aContext->currentWindow, WNDEXTRA_AREALIST, 0);
    SetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTUSEDLEN, 0);
    SetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTMAXLEN, 0);
  }
}

static void guiAddArea(ContextPtr aContext, int left, int top, int right, int bottom) {
  GuiRect *ptr=(GuiRect *)GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALIST);
  int usedLen=GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTUSEDLEN);
  int maxLen=GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTMAXLEN);

  if (usedLen==maxLen) {
    maxLen+=64; // equals 1 kbyte
    ptr=(GuiRect *)realloc(ptr, sizeof(GuiRect)*maxLen);
    SetWindowLong(aContext->currentWindow, WNDEXTRA_AREALIST, (LONG)(void*)ptr);
    SetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTMAXLEN, maxLen);
  }
  ptr[usedLen].left=left;
  ptr[usedLen].top=top;
  ptr[usedLen].right=right;
  ptr[usedLen].bottom=bottom;
  SetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTUSEDLEN, usedLen+1);
}

static int guiGetAreaListLen(ContextPtr aContext) {
  return GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTUSEDLEN);
}

static bool ctrlPressed(void) {
  return (HIWORD(GetKeyState(VK_CONTROL))!=0);
}

static void guiHideMouse(ContextPtr aContext) {
  ShowCursor(FALSE);
}

static void guiShowMouse(ContextPtr aContext) {
  ShowCursor(TRUE);
}
EOF
}

sub genDrawX {
$rc .= <<EOF;
static void guiHideWindow(Window window) {
  XUnmapWindow(display, window);
}

static void guiShowWindow(Window window) {
  XMapWindow(display, window);
}

static void guiSetWindowTitle(Window window, const char *title) {
}

static void guiVScrollRange(Window window, int min, int max, int page) {
}

static int guiGetVScrollPos(Window window) {
  return 0;
}

EOF
}

sub genDrawXt {
$rc .= <<EOF;
static void guiRefreshWindow(Window window, GuiRect *rect) {
}

static void guiHideWindow(Window window) {
//  XUnmapWindow(display, window);
}

static void guiShowWindow(Window window) {
//  XMapWindow(display, window);
}

static void guiSetWindowTitle(Window window, const char *title) {
}

static void guiVScrollRange(Window window, int min, int max, int page) {
}

static int guiGetVScrollPos(Window window) {
  return 0;
}

EOF
}


##########################################################################
#
# Text Editor Object
#
##########################################################################
sub genTextEditorGdi {
  my @TextEditorStructDef=('GuiWindow window;', 'char *textBuffer;', 'int cursorPos;', 'int bufferSize;', 'int bufferMaxSize;', 'void (*updateFunc)(int updateUserId, const char *text);', 'int updateUserId;');
  genEnumStruct("struct", "TextEditor", \@TextEditorStructDef);

  $rc .= <<EOF;
static void initTextEditor(TextEditorPtr editor, const char *text) {
  if (editor->textBuffer) free(editor->textBuffer);
  if (text!=NULL) {
    editor->bufferSize=strlen(text);
    editor->bufferMaxSize=(editor->bufferSize+255)&(~255); // round up to nearest 256 marker
    editor->textBuffer=(char*)malloc(editor->bufferMaxSize);
    strcpy(editor->textBuffer, text);
  } else {
    editor->textBuffer=NULL;
    editor->bufferSize=0;
    editor->bufferMaxSize=0;
  }
  editor->cursorPos=0;
}

static void showTextEditor(ContextPtr aContext, TextEditorPtr editor, int left, int top, int right, int bottom) {
}

static void hideTextEditor(ContextPtr aContext, TextEditorPtr editor) {
  guiRefreshWindow(aContext->currentWindow, NULL);
}
EOF
}

#
# !!! Temp solution use SF2.0 text editor
#
sub genSynTextEditor {
  $rc .= readCompleteFile("editor_functions.txt");

  push(@initFunctions, "InitNoteScriptEditor();");
  push(@termFunctions, "DeInitNoteScriptEditor();");
}

sub genGui {
  if ($platformGraphics eq 'GDI') {
    genSystemColorsGdi();
    executeFile('studiofactory_fonts_gdi.pl');
    genDrawGdi();
    genSynTextEditor();
#    genTextEditorGdi();
  }
  if ($platformGraphics eq 'X') {
    $rv .= "Display *display;\n";
    genDrawX();
  }
  if ($platformGraphics eq 'Xt') {
    genDrawXt();
  }
}

##########################################################################
#
# Strings
#
##########################################################################
sub genStrings {
  my @stringsEnum=();
  my @stringsTable=();
  my $language;

  foreach $language (@languageCodes) {
    my $filename = lc "studiofactory_lang_${language}.txt";
    my $i=0;

    printf $filename . "\n";
    open(FILE,$filename) || die "can't open $filename";
    while (<FILE>) {
      if (/([a-zA-Z0-9_]+)\s+(.+)/) {
        if (!defined($stringsEnum[$i]) || ($stringsEnum[$i] eq $1)) {
          $stringsEnum[$i]=$1;
          if (defined($stringsTable[$i])) {
            $stringsTable[$i] .= ",";
          }
          $stringsTable[$i] .= "$2";
        } else {
          die "enum $stringsEnum[$i] and $1 mismatch while reading $filename";
        }
        $i++;
      }
    }
    close(FILE);
  }

  for(my $i=0;$i<scalar(@stringsTable);$i++) {
    $stringsTable[$i] = '{' . $stringsTable[$i] . '}';
  }

  genEnumStruct("enum", "StringIndex", \@stringsEnum, ',');
  genTable("const char * const", "strings", \@stringsTable, "[noofLanguages]");
}

##########################################################################
#
# DSP Engine
#
##########################################################################
sub genDsp {

  rd('static const int maxObjectIo=32;');
  rd('static const int maxDspObjects=1024;');
  rd('static const int scriptReturnStackSize=256;');
  my @ScriptInfoStructDef=('int rSP;', 'int dSP;', 'int workVars[26];', 'int returnStack[scriptReturnStackSize];', 'char *script;');
  genEnumStruct("struct", "ScriptInfo", \@ScriptInfoStructDef);
  my @DspObjectStructDef=('bool inUse;', 'void (*dspRoutine)(_DspObject *);', 'struct {', 'int *input;', 'int intern;', 'int output;', 'int knob;', '} io[maxObjectIo];', 'union { ScriptInfo *scriptInfo; signed short *buffer; };');
  genEnumStruct("struct", "DspObject", \@DspObjectStructDef);
  rd('typedef void (*DspRoutinePtr)(DspObject *);');


$rv .= <<EOF;
static short EXPTBL16[32768];
static short SINTBL[65536];
static int EXPTBL32[32768];
static int DSP_Left;
static int DSP_Right;
static int dspInputLeft;
static int dspInputRight;
static unsigned char PERLINTBL[256]={
   151,160,137,91,90,15,
   131,13,201,95,96,53,194,233,7,225,140,36,103,30,69,142,8,99,37,240,21,10,23,
   190, 6,148,247,120,234,75,0,26,197,62,94,252,219,203,117,35,11,32,57,177,33,
   88,237,149,56,87,174,20,125,136,171,168, 68,175,74,165,71,134,139,48,27,166,
   77,146,158,231,83,111,229,122,60,211,133,230,220,105,92,41,55,46,245,40,244,
   102,143,54, 65,25,63,161, 1,216,80,73,209,76,132,187,208, 89,18,169,200,196,
   135,130,116,188,159,86,164,100,109,198,173,186, 3,64,52,217,226,250,124,123,
   5,202,38,147,118,126,255,82,85,212,207,206,59,227,47,16,58,17,182,189,28,42,
   223,183,170,213,119,248,152, 2,44,154,163, 70,221,153,101,155,167, 43,172,9,
   129,22,39,253, 19,98,108,110,79,113,224,232,178,185, 112,104,218,246,97,228,
   251,34,242,193,238,210,144,12,191,179,162,241, 81,51,145,235,249,14,239,107,
   49,192,214, 31,181,199,106,157,184, 84,204,176,115,121,50,45,127, 4,150,254,
   138,236,205,93,222,114,67,29,24,72,243,141,128,195,78,66,215,61,156,180};

//
// Remap pots in correct order for Sequencer
//
static const int SeqPotRemap[16]={
  0,2,4,6,8,10,12,14,
  1,3,5,7,9,11,13,15
};

static int dspRnd1;
static int dspRnd2;

EOF

$rc .= <<EOF;
//
// Generate SIN and EXP tables
//
void initDspTables(void) {
  double pi = 3.141592653589793;
  int cnt;

  // Generate EXP table
  for(cnt=0;cnt<32768;cnt++) {
    EXPTBL16[cnt]=(short)(exp(cnt*log(32767.0)/32767.0));
    EXPTBL32[cnt]=  (int)((exp(cnt*log(32768.0)/32767.0)-1)*8192.0);
  }

  // Generate SIN table
  for(cnt=-32768;cnt<32768;cnt++) {
    SINTBL[cnt+32768]=(short)(32767.0*sin((double)cnt*(pi/32768.0)));
  }
}


// Find unused Dsp object. We start allocating from 1 until maxDspObjects-1, because 0 is placeholder for constants and last is used for loop termination.
static int nextFreeDsp(void) {
  int n=0;
  int i;

  for(i=1;i<maxDspObjects-1;i++) {
    if (!projects[currentProject].dsp[i].inUse) {
      n=i;
      memset(&(projects[currentProject].dsp[i]), 0, sizeof(DspObject));
      for(i=0;i<maxObjectIo;i++) {
        projects[currentProject].dsp[n].io[i].input=&(projects[currentProject].dsp[0].io[0].output);
      }
      break;
    }
  }
  return n;
}
EOF

  $rc .= readCompleteFile("dsp_functions.txt");


$rc .= <<EOF;

//
// Reset all object to default state
//
static void dspResetAll(DspObject *objectList) {
  DspObject *currentObject=objectList;

  for(currentObject=objectList;;currentObject++) {
    if (currentObject->dspRoutine) {
      int i;

      for(i=0;i<maxObjectIo;i++) {
        currentObject->io[i].output=0;
        currentObject->io[i].intern=0;
      }
    } else {
      break;
    }
  }

#if 0
  for(;;) {
    if (CurObj->dspRoutine==NULL) {
      // No more modules
      tempoCounter=999999; // Force first step directly after restart
      ticksCounter=0;
      curPlayTime=0;
      curTempo=882;
      curTicks=6;
      curSongPos=0;
      curRow=-1;
      return;
    }
    if (CurObj->dspRoutine==DSPRoutine_FLA) {
      memset(CurObj->Buffer, 0 , objectInfo[OBJ_FLA].BufferSize);
    }
    if (CurObj->dspRoutine==DSPRoutine_DLY) {
      memset(CurObj->Buffer, 0 , objectInfo[OBJ_DLY].BufferSize);
    }
    if (CurObj->dspRoutine==DSPRoutine_SDY) {
      memset(CurObj->Buffer, 0 , objectInfo[OBJ_SDY].BufferSize);
    }
    if (CurObj->dspRoutine==DSPRoutine_SCRIPT) {
      CurObj->ScriptInfo->RSP=0;
      memset(&(CurObj->ScriptInfo->ReturnStack), 0, NOTESCRIPT_RETURNSTACKSIZE*sizeof(int));
      memset(&(CurObj->ScriptInfo->workVars), 0, 26*sizeof(int));
    }
    for(cnt=0; cnt<MAX_IO; cnt++) {
      CurObj->Out[cnt]=0;
    }
    for (cnt=0; cnt<MAX_INTERN; cnt++) {
      CurObj->Intern[cnt] = 0;
    }

    CurObj++;
  }
#endif
}


//
// Calculate wave block for pause/stop mode. Only zero's are output to the
// soundcard.
//
void dspCalculateZeroBlock(signed short *MemoryBlock, signed short *EndAddr) {
  do {
    *MemoryBlock = 0; // Fill with zero
    MemoryBlock++;
  } while (MemoryBlock != EndAddr);
  if (audioInReady) {
    int noofBuffers=audioInputRingWritePos-audioInputRingReadPos;
    if (noofBuffers<0) noofBuffers+=audioInputRingSize;
    if (noofBuffers>5) {
      DSP_BlockUse[audioInputRing[audioInputRingReadPos++]]=AUDIOBLOCK_UNUSED;
      if (audioInputRingReadPos>=audioInputRingSize) audioInputRingReadPos=0;
    }
    if (noofBuffers>4) {
      DSP_BlockUse[audioInputRing[audioInputRingReadPos++]]=AUDIOBLOCK_UNUSED;
      if (audioInputRingReadPos>=audioInputRingSize) audioInputRingReadPos=0;
    }
  }
}


static void dspCalculateBlock(DspObject *objectList, signed short *MemoryBlock, signed short *EndAddr) {
  int max_left=0;
  int max_right=0;
  DspObject *currentObject;

  dspInputLeft=0;
  dspInputRight=0;
  do {
    DSP_Left=0;
    DSP_Right=0;

    if (audioInReady) {
      int noofBuffers=audioInputRingWritePos-audioInputRingReadPos;
      if (noofBuffers<0) noofBuffers+=audioInputRingSize;
      if (noofBuffers>0) {
        // The noofBuffers>5 test throws away input blocks if we are to slow to process (high CPU load then sometimes playback is slower as realtime)
        if (noofBuffers>5 || audioInputBufferPos*2>=DSP_BlockSize[audioInputRing[audioInputRingReadPos]]) {
          audioInputBufferPos=0;
          DSP_BlockUse[audioInputRing[audioInputRingReadPos++]]=AUDIOBLOCK_UNUSED;
          if (audioInputRingReadPos>=audioInputRingSize) audioInputRingReadPos=0;
        }
        dspInputLeft=((signed short *)(DSP_DriverBlock[audioInputRing[audioInputRingReadPos]]))[audioInputBufferPos*2];
        dspInputRight=((signed short *)(DSP_DriverBlock[audioInputRing[audioInputRingReadPos]]))[(audioInputBufferPos++)*2+1];
        
        // Audio input <-> output Sync trying to keep 3 buffers in the ring
        if ((audioInputBufferSubPos++)>1024) {
          audioInputBufferSubPos=0;
          if (noofBuffers<3) audioInputBufferPos--;
          if (noofBuffers>3) audioInputBufferPos++;
        }
      } else {
        dspInputLeft=0;
        dspInputRight=0;
      }
    }

    for(currentObject=objectList;;currentObject++) {
      if (currentObject->dspRoutine) {
        (currentObject->dspRoutine)(currentObject);
      } else {
        break;
      }
    }

    // Clip audio and put in driver buffer
    {
      int audio_sample_l=limit(DSP_Left);
      int audio_sample_r=limit(DSP_Right);
      max_left=max(max_left, abs(audio_sample_l));
      max_right=max(max_right, abs(audio_sample_r));
      *(MemoryBlock++) = (short)(audio_sample_l);
      *(MemoryBlock++) = (short)(audio_sample_r);
      if (DSP_ScopePosition<(4*SCOPEBUFSIZE)) {
        DSP_ScopeL[DSP_ScopePosition>>2]=(short)audio_sample_l;
        DSP_ScopeR[DSP_ScopePosition>>2]=(short)audio_sample_r;
        DSP_ScopePosition++;
      }
    }
  } while (MemoryBlock != EndAddr);
  VU_newLeft=max_left;
  VU_newRight=max_right;
}

EOF

  push(@initFunctions, "initDspTables();");

}


##########################################################################
#
# Audio router
#
##########################################################################
sub genAudioRouter {
  my @enumplaymode=qw( PLAYMODE_STOP PLAYMODE_PLAY );
  my @enumrecordmode=qw( RECORDMODE_STOP RECORDMODE_RECORD );
  genEnumStruct("enum", "PlayMode", \@enumplaymode, ',');
  genEnumStruct("enum", "RecordMode", \@enumrecordmode, ',');

  rv('PlayMode playMode = PLAYMODE_STOP;');
  rv('RecordMode recordMode = RECORDMODE_STOP;');

  rv('const char **audioInDeviceNames;');
  rv('int currentAudioInDevice;');
  rv('const char **audioOutDeviceNames;');
  rv('int currentAudioOutDevice;');

  mutex('audioMutex');

$rv .= <<EOF;
static int audioBufferLengthTable[]={
          256,  512,  768,
   1024, 1280, 1536, 1792,
   2048, 2304, 2560, 2816,
   3072, 3328, 3584, 3840,
   4096, 4608, 5120, 5632,
   6144, 6656, 7168, 8192,
//   9216,10240,11264,12288,13312,14336,15360,16384,
      0
};
EOF

$rc .= <<EOF;
static void updatePlayModeMenu(void) {
  markMenuItem(mainMenu, MENU_STOP, PLAYMODE_PLAY != playMode);
  markMenuItem(synFactoryMenu, MENU_STOP, PLAYMODE_PLAY != playMode);
  markMenuItem(mainMenu, MENU_PLAY, PLAYMODE_PLAY == playMode);
  markMenuItem(synFactoryMenu, MENU_PLAY, PLAYMODE_PLAY == playMode);
}

static void setPlayMode(PlayMode newPlayMode) {
  logprintf("setPlayMode %d (current is %d)\\n", (int)newPlayMode, (int)playMode);
  if ((playMode!=newPlayMode || currentPlayingProject!=currentProject) && currentProject<maxProjects) {
    lockMutex(audioMutex);
    lockMutex(midiMutex);
    switch(newPlayMode) {
    case PLAYMODE_PLAY:
      currentPlayingProject=currentProject;
      break;
    default:
      currentPlayingProject=-1;
      break;
    }
    playMode=newPlayMode;
    unlockMutex(midiMutex);
    unlockMutex(audioMutex);
    updatePlayModeMenu();
    transportWindowRefresh=true;
  }
}

static void setRecordMode(RecordMode newRecordMode) {
  if (newRecordMode!=recordMode) {
    switch(newRecordMode) {
    case RECORDMODE_STOP:
      if (captureSample.file) {
        closeSampleFile(&captureSample);
      }
      recordMode=RECORDMODE_STOP;
      markMenuItem(mainMenu, MENU_RECORD, false);
      markMenuItem(synFactoryMenu, MENU_RECORD, false);
      break;
    case RECORDMODE_RECORD:
      if (captureFileSelect()) {
        captureBufWritePos=0;
        captureBufReadPos=0;
        recordMode=RECORDMODE_RECORD;
        markMenuItem(mainMenu, MENU_RECORD, true);
        markMenuItem(synFactoryMenu, MENU_RECORD, true);
      }
      break;
    default:
      break;
    }
    transportWindowRefresh=true;
  }
}
EOF

$rd .= <<EOF;
typedef enum  _audioBlockUse {
  AUDIOBLOCK_UNAVAILABLE,
  AUDIOBLOCK_UNUSED,
  AUDIOBLOCK_CARDINPUT,
  AUDIOBLOCK_CARDOUTPUT,
  AUDIOBLOCK_ZEROS,
  AUDIOBLOCK_INPUTRING,
  AUDIOBLOCK_CAPTURE,
} audioBlockUse;
EOF


rv('const int waveBlockSize=8192;');
rv('const int noofWaveBlocks=32;');
rv('LPSTR         DSP_DriverBlock[noofWaveBlocks];');
rv('audioBlockUse DSP_BlockUse[noofWaveBlocks];');
rv('int           DSP_BlockSize[noofWaveBlocks];');

rv('volatile bool audioInReady=false;');
rv('volatile bool audioOutReady=false;');
rv('Sample captureSample;');

#
# Audio input ringbuffer
#
rv('const int audioInputRingSize=(noofWaveBlocks);');
rv('int audioInputRing[audioInputRingSize];');
rv('int audioInputRingWritePos;');
rv('int audioInputRingReadPos;');
rv('int audioInputBufferPos;');
rv('int captureBufWritePos;');
rv('int captureBufReadPos;');
rv('int audioInputBufferSubPos;');

rv('int nextAudioOutBufferSize=waveBlockSize;');
rv('int currentAudioOutBufferSize=waveBlockSize;');
rv('int nextAudioInBufferSize=waveBlockSize;');
rv('int currentAudioInBufferSize=waveBlockSize;');

$rc .= <<EOF;
static inline PlayMode getPlayMode(void) {
  return playMode;
}

static inline RecordMode getRecordMode(void) {
  return recordMode;
}

#define WM_DSPRESTART    WM_USER+4

//
// Audio output definition
//
#define PLAYBACKTHREAD_WIN32 1
#define PLAYBACKRATE 44100



static int           DSP_NewNoofBuffers = 5;
static int           DSP_CurNoofBuffers = DSP_NewNoofBuffers;
static volatile int  DSP_NrActive=0;  /* number of active blocks currently in playback queue */
static int           noofInputBuffersOnCard=0;

static HANDLE        audioInThreadHandle;
static HANDLE        audioOutThreadHandle;
static DWORD         audioInThreadID;
static DWORD         audioOutThreadID;
static HWAVEIN       waveInHandle;
static HWAVEOUT      waveOutHandle;
static WAVEHDR       DSP_DriverBlockHDR[noofWaveBlocks];

// DSound.DLL function pointers
#include "sf_dsound.h"
static HINSTANCE         dsoundDLL=NULL;
static bool              dsound_ready=false;
static getVersionFunc    dsound_getVersion;
static initDLLFunc       dsound_initDLL;
static setBufferSizeFunc dsound_setBufferSize;
static audioLoopFunc     dsound_audioLoop;
static openDSoundFunc    dsound_open;
static closeDSoundFunc   dsound_close;

//
// Audio capture of output
//
static const int caputureBufferSize=(noofWaveBlocks+2);
static int captureBufNrBuffer[caputureBufferSize]; /* This buffer keeps the audio buffers in correct order. */

static void resetPlayback(void) {
  lockMutex(audioMutex);
  if (currentPlayingProject>=0) {
    dspResetAll(&(projects[currentPlayingProject].dsp[1]));
  } else if (currentProject>=0) {
    dspResetAll(&(projects[currentProject].dsp[1]));
  }
  unlockMutex(audioMutex);
}

//
// Update capture file buffer
//
static void BufferToCaptureFile(int bufnr) {
  if (recordMode==RECORDMODE_RECORD && (DSP_BlockUse[bufnr]==AUDIOBLOCK_CARDOUTPUT)) {
    DSP_BlockUse[bufnr]=AUDIOBLOCK_CAPTURE;
    captureBufNrBuffer[captureBufWritePos++]=bufnr;
    if (captureBufWritePos>=caputureBufferSize) {
      // Wrap around at end of buffer
      captureBufWritePos=0;    
    }
  } else {
    DSP_BlockUse[bufnr]=AUDIOBLOCK_UNUSED;
  }
}



//
// Calculate and process single block of audio
// returns true if block has audio or false if block only contains zeros (playback stopped or paused)
// This flag is used to decide which blocks need to be written to disk during capturing mode.
//
static inline bool calculateBlock(signed short *MemoryBlock, signed short *EndMemoryBlock) {
  // If this flag is set, we have generated audio, this information is used when
  // allocating audio blocks
  bool hasaudio=false;

  switch(playMode) {
  case PLAYMODE_PLAY:
//    case CAPTURE_RECORD:
    // !!! hack
    dspCalculateBlock(&(projects[currentPlayingProject].dsp[1]), MemoryBlock, EndMemoryBlock);
    hasaudio=true;
    break;
  default:
    dspCalculateZeroBlock(MemoryBlock, EndMemoryBlock);
    VU_newLeft=0;
    VU_newRight=0;
    break;
  }
  return hasaudio;
}


static void WINAPI dsoundCalc(signed short *MemoryBlock, signed short *EndMemoryBlock) {
  lockMutex(audioMutex);
  switch(playMode) {
  case PLAYMODE_PLAY:
    if (recordMode == RECORDMODE_RECORD) {
      // find suitable block
      for (int i=0; i<noofWaveBlocks; i++) {
        if (DSP_BlockUse[i] == AUDIOBLOCK_UNUSED) {
          // !!! hack
          dspCalculateBlock(&(projects[currentPlayingProject].dsp[1]), (signed short *)DSP_DriverBlock[i], ((signed short *)DSP_DriverBlock[i])+(EndMemoryBlock-MemoryBlock));
//          dspCalculateZeroBlock(&OBJs[1], (signed short *)DSP_DriverBlock[i], ((signed short *)DSP_DriverBlock[i])+(EndMemoryBlock-MemoryBlock));
          DSP_BlockUse[i]=AUDIOBLOCK_CARDOUTPUT;
          DSP_BlockSize[i]=EndMemoryBlock-MemoryBlock;
          memcpy(MemoryBlock, DSP_DriverBlock[i], (EndMemoryBlock-MemoryBlock)*sizeof(signed short));
          BufferToCaptureFile(i);
          break;
        }      
      }
    } else {
      dspCalculateBlock(&(projects[currentPlayingProject].dsp[1]), MemoryBlock, EndMemoryBlock);
    }
//      hasaudio=true;
//      allclear=false;
    break;
  default:
    dspCalculateZeroBlock(MemoryBlock, EndMemoryBlock);
    VU_newLeft=0;
    VU_newRight=0;
    break;
  }
  unlockMutex(audioMutex);
}

//
// Send block to audio device.
//
static inline void DSP_SendBlock(int blocknr) {
  if (DSP_DriverBlock[blocknr]!=NULL) {
    bool hasaudio;
    currentAudioOutBufferSize = nextAudioOutBufferSize;
    hasaudio=calculateBlock((signed short *)DSP_DriverBlock[blocknr], ((signed short *)DSP_DriverBlock[blocknr])+currentAudioOutBufferSize);
    DSP_BlockUse[blocknr]=(hasaudio)?AUDIOBLOCK_CARDOUTPUT:AUDIOBLOCK_ZEROS;
    DSP_BlockSize[blocknr]=currentAudioOutBufferSize;

    DSP_DriverBlockHDR[blocknr].lpData = DSP_DriverBlock[blocknr];
    DSP_DriverBlockHDR[blocknr].dwBufferLength = currentAudioOutBufferSize*sizeof(signed short);
    DSP_DriverBlockHDR[blocknr].dwFlags = 0;
    DSP_DriverBlockHDR[blocknr].dwUser=blocknr;
    waveOutPrepareHeader(waveOutHandle, &DSP_DriverBlockHDR[blocknr], sizeof(WAVEHDR));
    waveOutWrite(waveOutHandle, &DSP_DriverBlockHDR[blocknr], sizeof(WAVEHDR));

#ifdef PLAYBACKTHREAD_ARTS
    arts_write(artsStream, DSP_DriverBlock[blocknr], currentAudioOutBufferSize*sizeof(signed short));
#endif
  }
}


//
// Flush cached audio data to disk.
//
void flushCaptureFile(void) {
  bool needrestart=false;

  while (captureBufReadPos!=captureBufWritePos) {
    int bufnr=captureBufNrBuffer[captureBufReadPos++];

    writeSampleData(&captureSample, (signed short *)(DSP_DriverBlock[bufnr]), DSP_BlockSize[bufnr]);
    DSP_BlockUse[bufnr]=AUDIOBLOCK_UNUSED;
    needrestart=true;

    if (captureBufReadPos>=caputureBufferSize) {
      captureBufReadPos=0;
    }
  }
  if (needrestart) {
    PostThreadMessage(audioOutThreadID, WM_DSPRESTART, 0, 0);
  }
}


static void setAudioOutBufferSize(int buffersize) {
  nextAudioOutBufferSize=buffersize;
  if (dsound_setBufferSize) {
    dsound_setBufferSize(buffersize);
  }
  configWindowRefresh=true;
}


//
// Fill PCMformat struct with one of two configurations depending on
// MONOSYNTH define.
//
static void FillPCMformatStruct(WAVEFORMATEX *myPCMformat) {
#ifdef MONOSYNTH
  myPCMformat->wFormatTag=WAVE_FORMAT_PCM;
  myPCMformat->nChannels=1;          // Mono (single channel)
  myPCMformat->nSamplesPerSec=PLAYBACKRATE;
  myPCMformat->nAvgBytesPerSec=PLAYBACKRATE*2;
  myPCMformat->nBlockAlign=2;
  myPCMformat->wBitsPerSample=16;    // 16 bits audio
  myPCMformat->cbSize=0;             // No extra info
#else
  myPCMformat->wFormatTag=WAVE_FORMAT_PCM;
  myPCMformat->nChannels=2;          // Stereo (dual channel)
  myPCMformat->nSamplesPerSec=PLAYBACKRATE;
  myPCMformat->nAvgBytesPerSec=PLAYBACKRATE*4;
  myPCMformat->nBlockAlign=4;
  myPCMformat->wBitsPerSample=16;    // 16 bits audio
  myPCMformat->cbSize=0;             // No extra info
#endif
}

static void setAudioInDevice(int audioInDevice) {
  if (audioInReady) {
    audioInReady=false;
    if (currentAudioInDevice>0) {
      while(noofInputBuffersOnCard>0) {
        Sleep(100);
      }
      waveInReset(waveInHandle);
      waveInClose(waveInHandle);
    }
  }

  logprintf("AudioInDevice set to %d ", audioInDevice);
  currentAudioInDevice=audioInDevice;

  if (currentAudioInDevice==0) {
    logprintf("(Disabled).\\n");
  }

  if (audioInDevice>0) {
    WAVEFORMATEX myPCMformat;
    WAVEINCAPS caps;
    MMRESULT error;
    FillPCMformatStruct(&myPCMformat);

    logprintf("(MME).\\n");

    waveInGetDevCaps(audioInDevice-1, &caps, sizeof(caps));
    logprintf("caps: 0x%x\\n", caps.dwFormats);
    error=waveInOpen(&waveInHandle, audioInDevice-1, &myPCMformat,(unsigned long)audioInThreadID, 0, CALLBACK_THREAD);
    if (error != MMSYSERR_NOERROR) {
      char tmp[MAXERRORLENGTH ];
      logprintf("ERROR waveInOpen failed, check WAVE-Input settings!\\n");
      waveInGetErrorText(error, tmp, MAXERRORLENGTH);
      logprintf("Error %d '%s'\\n", error, tmp);
      MessageBox(mainWindow, "waveInOpen failed, check WAVE-Input settings",NULL,MB_OK | MB_ICONEXCLAMATION);
    } else {
      logprintf("waveInOpen ok. Sending start message to audio thread.\\n");
      audioInReady=true;
      waveInStart(waveInHandle);
      while (PostThreadMessage(audioInThreadID, WM_DSPRESTART, 0, 0)==0) {
        logprintf("Post startmessage failed, repeating.\\n");
        Sleep(500);
      }
    }
  }
  configWindowRefresh=true;
}


//
// (Re)Open wave device
//
static void setAudioOutDevice(int audioOutDevice) {
  if (audioOutReady) {
    if (currentAudioOutDevice==1) {
      audioOutReady=false;
      dsound_close();
    }
    if (currentAudioOutDevice>1) {
      audioOutReady=false;
      while (DSP_NrActive>0) {
        Sleep(100);
      }
      waveOutClose(waveOutHandle);
    }
  }

  logprintf("AudioOutDevice set to %d ", audioOutDevice);
  currentAudioOutDevice=audioOutDevice;

  if (currentAudioOutDevice==0) {
    logprintf("(Disabled).\\n");
  }

  // Use sf_dsound.dll
  if (currentAudioOutDevice==1) {
    logprintf("(DirectSound).\\n");
    
    if (NULL==dsoundDLL) {
      logprintf("Loading 'sf_dsound.dll'.\\n");
      dsoundDLL=LoadLibrary("sf_dsound.dll");
      if (dsoundDLL) {
        dsound_getVersion=(getVersionFunc)GetProcAddress(dsoundDLL, "getVersion");
        dsound_initDLL=(initDLLFunc)GetProcAddress(dsoundDLL, "initDLL");
        dsound_setBufferSize=(setBufferSizeFunc)GetProcAddress(dsoundDLL, "setBufferSize");
        dsound_audioLoop=(audioLoopFunc)GetProcAddress(dsoundDLL, "audioLoop");
        dsound_open=(openDSoundFunc)GetProcAddress(dsoundDLL, "openDSound");
        dsound_close=(closeDSoundFunc)GetProcAddress(dsoundDLL, "closeDSound");
        if (dsound_getVersion &&
            dsound_initDLL &&
            dsound_setBufferSize &&
            dsound_open &&
            dsound_close) {
          logprintf("Direct sound driver loaded. DLL version '%s'.\\n", dsound_getVersion());
          if (strcmp("SF_DSOUND.DLL V1.16", dsound_getVersion())==0 ||
              strcmp("SF_DSOUND.DLL V2.00", dsound_getVersion())==0) {
            dsound_ready=true;
          } else {
            logprintf("Failed to initialise 'sf_dsound.dll', because it has the wrong version information.\\n");
          }
        }
      } else {
        logprintf("Failed to load 'sf_dsound.dll', check StudioFactory installation.\\n");
      }
    }
    if (dsound_ready) {
      (*dsound_initDLL)(mainWindow, logprintf, dsoundCalc);
      (*dsound_setBufferSize)(nextAudioOutBufferSize);
      (*dsound_open)();

      audioOutReady=true;
      PostThreadMessage(audioOutThreadID, WM_DSPRESTART, 0, 0);
    }
  }

  // Use MME drivers
  if (currentAudioOutDevice>1) {
    WAVEFORMATEX myPCMformat;
    FillPCMformatStruct(&myPCMformat);

    logprintf("(MME).\\n");

    if (waveOutOpen(&waveOutHandle, (currentAudioOutDevice==2)?WAVE_MAPPER:(currentAudioOutDevice-3), (LPWAVEFORMATEX)&myPCMformat,(unsigned long)audioOutThreadID,0,CALLBACK_THREAD) != MMSYSERR_NOERROR) {
      logprintf("ERROR waveOutOpen failed, check WAVE-Output settings!\\n");
      MessageBox(mainWindow, "waveOutOpen failed, check WAVE-Output settings",NULL,MB_OK | MB_ICONEXCLAMATION);
    } else {
      logprintf("waveOutOpen ok. Sending start message to audio thread.\\n");
      audioOutReady=true;
      // Mmm waveOutOpen should result in a MM_WOM_OPEN but never got it.
      // Following postmessage kicks it on anyway.
      while (PostThreadMessage(audioOutThreadID, WM_DSPRESTART, 0, 0)==0) {
        logprintf("Post startmessage failed, repeating.\\n");
        Sleep(500);
      }
    }
  }
  configWindowRefresh=true;
}


static DWORD audioInThread(LPDWORD lpdwParam) {
  MSG msg;
  int i;
  int inUse;

  // Run in loop processing wave data
  logprintf("Audio input thread %d started\\n", audioInThreadID);

  for(;;) {
    int getMessageResult=GetMessage(&msg, NULL, 0, 0);

    if (msg.message==MM_WIM_DATA) {
      int noofBuffers=audioInputRingWritePos-audioInputRingReadPos;
      if (noofBuffers<0) noofBuffers+=audioInputRingSize;

      waveInUnprepareHeader(waveInHandle, ((WAVEHDR *)msg.lParam), sizeof(WAVEHDR));
      if (noofBuffers>6) {
        DSP_BlockUse[((WAVEHDR *)msg.lParam)->dwUser]=AUDIOBLOCK_UNUSED;
      } else {
        DSP_BlockUse[((WAVEHDR *)msg.lParam)->dwUser]=AUDIOBLOCK_INPUTRING;
        audioInputRing[audioInputRingWritePos++]=((WAVEHDR *)msg.lParam)->dwUser;
        if (audioInputRingWritePos>=audioInputRingSize) audioInputRingWritePos=0;
      }
      noofInputBuffersOnCard--;
    }

    lockMutex(audioMutex);
    inUse=0;
    for (i=0;i<noofWaveBlocks && noofInputBuffersOnCard<8 && audioInReady;i++) {
      if (DSP_BlockUse[i]==AUDIOBLOCK_UNUSED) {
        DSP_BlockUse[i]=AUDIOBLOCK_CARDINPUT;
        DSP_BlockSize[i]=currentAudioOutBufferSize;
        DSP_DriverBlockHDR[i].lpData = DSP_DriverBlock[i];
        DSP_DriverBlockHDR[i].dwBufferLength = DSP_BlockSize[i]*sizeof(signed short);
        DSP_DriverBlockHDR[i].dwFlags = 0;
        DSP_DriverBlockHDR[i].dwUser=i;
        waveInPrepareHeader(waveInHandle, &DSP_DriverBlockHDR[i], sizeof(WAVEHDR));
        waveInAddBuffer(waveInHandle, &DSP_DriverBlockHDR[i], sizeof(WAVEHDR));
        noofInputBuffersOnCard++;
      }
    }
    unlockMutex(audioMutex);
  }
  return 0;
}

static DWORD audioOutThread(LPDWORD lpdwParam) {
  MSG msg;
  int cnt;

  // Run in loop processing wave data
  logprintf("Audio output thread %d started\\n", audioOutThreadID);
  for(;;) {
    int getMessageResult=GetMessage(&msg, NULL, 0, 0);
    
    if (getMessageResult==0 || getMessageResult==-1) {
      logprintf("Audio thread terminated %d\\n", getMessageResult);
    }

    if (msg.message==MM_WOM_DONE) {
      waveOutUnprepareHeader(waveOutHandle, ((WAVEHDR *)msg.lParam), sizeof(WAVEHDR));
      BufferToCaptureFile(((WAVEHDR *)msg.lParam)->dwUser);
      DSP_NrActive--;
//      logprintf("NrActive %d\\n", DSP_NrActive);
    }

    if (currentAudioOutDevice==1 && audioOutReady) {
      // Use sf_dsound.dll loop
      (*dsound_audioLoop)();
    }

    // find unused block(s) to send
    if (currentAudioOutDevice>1 && audioOutReady) {
      DSP_CurNoofBuffers=DSP_NewNoofBuffers;
      lockMutex(audioMutex);
      for (cnt=0; cnt<noofWaveBlocks && DSP_NrActive<DSP_CurNoofBuffers; cnt++) {
        if (DSP_BlockUse[cnt]==AUDIOBLOCK_UNUSED) {
//          logprintf("SendBlock %d\\n", cnt);
          DSP_SendBlock(cnt);
          DSP_NrActive++;
//          logprintf("NrActive %d\\n", DSP_NrActive);
        }      
      }
      unlockMutex(audioMutex);
    }
  }
  return 0;
} 

static int noofAudioInDevices;
static int noofAudioOutDevices;
static void openAudioDriver(void) {
  WAVEINCAPS audioInInfo;
  WAVEOUTCAPS audioOutInfo;
  int i;

#ifdef _beginthreadex
  audioInThreadHandle = _beginthreadex(NULL, 0, (LPTHREAD_START_ROUTINE)audioInThread, NULL, CREATE_SUSPENDED, &audioInThreadID);
  audioOutThreadHandle = _beginthreadex(NULL, 0, (LPTHREAD_START_ROUTINE)audioOutThread, NULL, CREATE_SUSPENDED, &audioOutThreadID);
#else
  DWORD dwThrdParam = 1;
  audioInThreadHandle = CreateThread( 
    NULL,                        // no security attributes 
    0,                           // use default stack size  
    (LPTHREAD_START_ROUTINE)audioInThread, // thread function 
    &dwThrdParam,                // argument to thread function 
    CREATE_SUSPENDED,            // use default creation flags 
    &audioInThreadID);             // returns the thread identifier
  audioOutThreadHandle = CreateThread( 
    NULL,                        // no security attributes 
    0,                           // use default stack size  
    (LPTHREAD_START_ROUTINE)audioOutThread, // thread function 
    &dwThrdParam,                // argument to thread function 
    CREATE_SUSPENDED,            // use default creation flags 
    &audioOutThreadID);             // returns the thread identifier
#endif
//  SetPriorityClass(GetCurrentProcess(), REALTIME_PRIORITY_CLASS);
  SetThreadPriority(audioInThreadHandle, THREAD_PRIORITY_TIME_CRITICAL);
  SetThreadPriority(audioOutThreadHandle, THREAD_PRIORITY_TIME_CRITICAL);
  ResumeThread(audioInThreadHandle);
  ResumeThread(audioOutThreadHandle);

#ifdef PLAYBACKTHREAD_ARTS
  arts_init();
  artsStream=arts_play_stream(44100, 16, 2, "artsc_test");
  pthread_create(&threadHandle, NULL, audioOutThread, NULL); 
#endif

  timeBeginPeriod(1); // Set timer resolution to 1 ms

  noofAudioInDevices=waveInGetNumDevs();
//  int noofAudioInDevices=0;
  audioInDeviceNames=(const char **)malloc((noofAudioInDevices+2)*sizeof(const char *));
  audioInDeviceNames[0]="(no audio input)";
  for(i=0;i<noofAudioInDevices;i++) {
    if (MMSYSERR_NOERROR==waveInGetDevCaps(i, &audioInInfo, (UINT)sizeof(WAVEINCAPS))) {
      audioInDeviceNames[i+1]=strdup(audioInInfo.szPname);
    } else {
      audioInDeviceNames[i+1]="(unknown)";
    }
  }
  audioInDeviceNames[noofAudioInDevices+1]=NULL;

  noofAudioOutDevices=waveOutGetNumDevs();
  audioOutDeviceNames=(const char **)malloc((noofAudioOutDevices+4)*sizeof(const char *));
  audioOutDeviceNames[0]="(no audio output)";
  audioOutDeviceNames[1]="DirectSound (sf_dsound.dll)";
  audioOutDeviceNames[2]="WaveMapper (default)";
  for(i=0;i<noofAudioOutDevices;i++) {
    if (MMSYSERR_NOERROR==waveOutGetDevCaps(i, &audioOutInfo, (UINT)sizeof(WAVEOUTCAPS))) {
      audioOutDeviceNames[i+3]=strdup(audioOutInfo.szPname);
    } else {
      audioOutDeviceNames[i+3]="(unknown)";
    }
  }
  audioOutDeviceNames[noofAudioOutDevices+3]=NULL;

}

static void closeAudioDriver(void) {
  int i;

  if (audioInDeviceNames) {
    for(i=0;i<noofAudioInDevices;i++) {
      if (audioInDeviceNames[i+1]) {
        free((void*)(audioInDeviceNames[i+1]));
      }
    }
    free(audioInDeviceNames);
    audioInDeviceNames=NULL;
  }
  if (audioOutDeviceNames) {
    for(i=0;i<noofAudioOutDevices;i++) {
      if (audioOutDeviceNames[i+3]) {
        free((void*)(audioOutDeviceNames[i+3]));
      }
    }
    free(audioOutDeviceNames);
    audioOutDeviceNames=NULL;
  }
}

//
//
//
static void allocateBuffers(void) {
  int i;

  for(i=0; i<noofWaveBlocks; i++) {
    if ((DSP_DriverBlock[i]=(LPSTR)GlobalAlloc(GMEM_FIXED | GMEM_SHARE, waveBlockSize*2))==NULL) {
      logprintf("Error allocating DSP MemoryBlock\\n");
//      MessageBox(hwnd, "Error allocating MemoryBlock", NULL, MB_OK | MB_ICONEXCLAMATION);
      DSP_BlockUse[i]=AUDIOBLOCK_UNAVAILABLE;
	  } else {
      logprintf("Allocated dsp block %d\\n", i);
      DSP_BlockUse[i]=AUDIOBLOCK_UNUSED;
    }
  }
}


//
//
//
static void deallocateBuffers(void) {
  int i;

  for(i=0; i<noofWaveBlocks; i++) {
    if (DSP_DriverBlock[i]) {
      logprintf("Freeing dsp block %d\\n", i);
      GlobalFree(DSP_DriverBlock[i]);
      DSP_DriverBlock[i] = NULL;
      DSP_BlockUse[i]=AUDIOBLOCK_UNAVAILABLE;
    }
  }
}


EOF
  push(@initFunctions, 'allocateBuffers();');
  push(@termFunctions, 'deallocateBuffers();');

  push(@initFunctions, 'updatePlayModeMenu();');
  push(@initFunctions, 'openAudioDriver();');
  push(@termFunctions, 'closeAudioDriver();');
#  push(@initFunctions, 'setAudioOutDevice(2);');

  push(@termFunctions, 'setAudioOutDevice(0);');
  push(@termFunctions, 'setAudioInDevice(0);');

  push(@termFunctions, 'setPlayMode(PLAYMODE_STOP);');
  push(@termFunctions, 'setRecordMode(RECORDMODE_STOP);');

  push(@stfSave, 'storeStfBegin(loadInfo, "AudioInputDevice"); storeStfInteger(loadInfo, 1); storeStfInteger(loadInfo, currentAudioInDevice); storeStfInteger(loadInfo, DSP_NewNoofBuffers); storeStfInteger(loadInfo, nextAudioOutBufferSize); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp(loadInfo->headerTitle, "AudioInputDevice")==0) { int i,d; readStfInteger(loadInfo, &i); if (i==1) { readStfInteger(loadInfo, &d); setAudioInDevice(d); } }');
  push(@stfSave, 'storeStfBegin(loadInfo, "AudioOutputDevice"); storeStfInteger(loadInfo, 1); storeStfInteger(loadInfo, currentAudioOutDevice); storeStfInteger(loadInfo, DSP_NewNoofBuffers); storeStfInteger(loadInfo, nextAudioOutBufferSize); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp(loadInfo->headerTitle, "AudioOutputDevice")==0) { int i,d,n,b; readStfInteger(loadInfo, &i); if (i==1) { readStfInteger(loadInfo, &d); readStfInteger(loadInfo, &n); readStfInteger(loadInfo, &b); setAudioOutDevice(d); setAudioOutBufferSize(b); DSP_NewNoofBuffers=n; } }');
}


##########################################################################
#
# Synfactory editor
#
##########################################################################
sub genSynfactoryPatchEditor {
  $rd .= "typedef const char *(*TranslationFunc)(int);\n";
  rv('const int outputXMargin=32;');
  rv('const int outputXMargin2=8;');
  rv('const int minXSize=64;');
  rv('const int knobXOffset=48;');
  rv('const int knobXSpacing=40;');
  rv('const int ioYSpacing=14;');
  rv('const int ioXSize=10;');
  rv('const int ioYSize=10;');
  rv('const int maxAutoSize=8;');

  rv('int currentCableType=2;');
  rv('int cableTypeList[4]={STRING_CABLE_TYPE_0, STRING_CABLE_TYPE_1, STRING_CABLE_TYPE_2, STRING_};');
  rv('bool showResizeMarkers;');

$rc .= <<EOF;
static char TranslationStr[128];
const char *translateFreq(int value) {
  double freq=exposc(value)/97392.0;
  if (freq>=1000) {
    sprintf(TranslationStr, "%dHz%01d", (int)(freq), (int)((freq-(int)(freq))*10));
    return TranslationStr;
  }
  if (freq>=100) {
    sprintf(TranslationStr, "%dHz%02d", (int)(freq), (int)((freq-(int)(freq))*100));
    return TranslationStr;
  }
  if (freq>=10) {
    sprintf(TranslationStr, "%dHz%03d", (int)(freq), (int)((freq-(int)(freq))*1000));
    return TranslationStr;
  }
  if (freq>=1) {
    sprintf(TranslationStr, "%dHz%04d", (int)(freq), (int)((freq-(int)(freq))*10000));
    return TranslationStr;
  }
  sprintf(TranslationStr, "Hz%05d", (int)(freq*100000));
  return TranslationStr;
}

const char *translateTrig(int value) {
  double time=(double)(exposc(value))/(double)(PLAYBACKRATE*32.0);
  if (time>=1000) {
    sprintf(TranslationStr, "%ds", (int)(time));
    return TranslationStr;
  }
  if (time>=100) {
    sprintf(TranslationStr, "%ds%01d", (int)(time), (int)((time-(int)(time))*10));
    return TranslationStr;
  }
  if (time>=10) {
    sprintf(TranslationStr, "%ds%02d", (int)(time), (int)((time-(int)(time))*100));
    return TranslationStr;
  }
  if (time>=1) {
    sprintf(TranslationStr, "%ds%03d", (int)(time), (int)((time-(int)(time))*1000));
    return TranslationStr;
  }
  sprintf(TranslationStr, "s%04d", (int)(time*10000));
  return TranslationStr;
}

const char *translateAHD(int value) {
  double time=(double)(0x7FFFFFFF/(double)(exposc(32768-value)))/(double)(PLAYBACKRATE);
  if (time>=1000) {
    sprintf(TranslationStr, "%ds", (int)(time));
    return TranslationStr;
  }
  if (time>=100) {
    sprintf(TranslationStr, "%ds%01d", (int)(time), (int)((time-(int)(time))*10));
    return TranslationStr;
  }
  if (time>=10) {
    sprintf(TranslationStr, "%ds%02d", (int)(time), (int)((time-(int)(time))*100));
    return TranslationStr;
  }
  if (time>=1) {
    sprintf(TranslationStr, "%ds%03d", (int)(time), (int)((time-(int)(time))*1000));
    return TranslationStr;
  }
  sprintf(TranslationStr, "s%04d", (int)(time*10000));
  return TranslationStr;
}
EOF


  my @objectTypeStructDef=();
  my @objectLoadNames=();
  my @objectTitles=();
  my @objectMenu=();
  my @objectKnobs;
  my @objectKnobsValue;
  my @objectKnobsTranslation;
  my @objectInputs;
  my @objectOutputs;
  my $objectAutoSize;
  my $objectRoutine='';
  my $objectHelpText;
  my $objectKeys='';
  my $objectBufferSize='';
  my $objectBufferKeep='';
  my $subMenuFlag=0;

  open FILE, "studiofactory_objects.txt" || die "couldn't open studiofactory_objects.txt";
  while(<FILE>) {
    s/\r//;
    s/\n//;
    if (!(/^\s*\#/)) {
      if (/^(.+?)\s+((\".+?\")|NULL)\s+((\".+?\")|NULL)\s+(.)\s+(.)\s+(.+?)\s+(.+?)\s+((\".+?\")|NULL)((\s+.*)?)/) {
        my $enumName=$1;
        my $names=$12;
        push(@objectTypeStructDef, $enumName);    
        push(@objectLoadNames, $2);
        push(@objectTitles, $4);
        if (!($6 eq '-')) {
          $objectKeys .= "        case '$6': n=addSynFactoryObject(currentPatch, $1, false); break;\n";
        }
        if ($7 eq 'A') {
          $objectAutoSize .= "  case $enumName: return true;\n";
        } else {
          if (!($8 eq 'NULL')) {
            $objectRoutine .= "\n  case $enumName: routine=dspRoutine_$8; break;";
          }
        }
        if (!($9 eq 'NULL')) {
          $objectHelpText .= "  case $enumName: return objectHelp_$9;\n";
        }
        if (!($10 eq 'NULL')) {
          push(@objectMenu, "$1+1024, {$10}");
        }
        while (defined($names) && $names =~ s/([KIO])(\d+)\:(\".*?\")(,(\d+))?\s*//) {
          if ($1 eq 'K') {
            $objectKnobs[$2] .= "    case $enumName: return $3; break;\n";
            if (defined $5) {
              $objectKnobsValue[$2] .= "    case $enumName: return $5; break;\n";
            }
          }        
          if ($1 eq 'I') {
            $objectInputs[$2] .= "    case $enumName: return $3; break;\n";
          }        
          if ($1 eq 'O') {
            $objectOutputs[$2] .= "    case $enumName: return $3; break;\n";
          }
        }
        while (defined($names) && $names =~ s/T(\d+)\:(\w+)//) {
          $objectKnobsTranslation[$1] .= "    case $enumName: return $2; break;\n";
        }
        while (defined($names) && $names =~ s/B\:(.+?);//) {
          $objectBufferSize .= "\n  case $enumName: return $1;";
        }
        while (defined($names) && $names =~ s/BS\:(.+?);//) {
          $objectBufferSize .= "\n  case $enumName: return $1;";
          $objectBufferKeep .= "\n  case $enumName: return true;";
        }
      }
      if (/^(\".*?\")$/) {
        if ($subMenuFlag) {
          push(@objectMenu, "0, {NULL}");
        }
        push(@objectMenu, "-1, {$1}");
        $subMenuFlag=1;
      }
      if (/^\-\-\-$/) {
        push(@objectMenu, "-2, {NULL}");
      }
    }
  }
  close FILE;
  push(@objectLoadNames, 'NULL'); # terminate list
  genEnumStruct("enum", "ObjectType", \@objectTypeStructDef, ',');
  genTable('const char * const', 'objectLoadNames', \@objectLoadNames);
  genTable('const char * const', 'objectTitles', \@objectTitles);
  if ($subMenuFlag) {
    push(@objectMenu, "0, {NULL}");
  }
  push(@objectMenu, "-2, {NULL}");
  push(@objectMenu, "0, {NULL}");
  genTable("MenuDef", 'synFactoryMenuDef', \@objectMenu);
#  genMenuDef('synFactoryMenuDef', 'studiofactory_mainmenu.txt');
  
  my @SynFactoryObjectStructDef=('bool selected;', 'ObjectType objectType;', 'int dsp;', 'bool showLabel;', 'int mode;', 'int x;', 'int y;', 'int xs;', 'int ys;', 'int labelXs;', 'int maxInput;', 'GuiWindow window;', 'char *name;', 'int mainColor;', 'int borderColor;', 'int currentOutput;', 'struct {', 'int fromObject;', 'int fromOutput;', 'int cableColor;', '} connections[maxObjectIo];');
  genEnumStruct("struct", "SynFactoryObject", \@SynFactoryObjectStructDef);
  my @SynFactoryPatchStructDef=('bool selected;', 'bool unfoldedPatchView;', 'char *patchName;', 'int offsX;', 'int offsY;', 'SynFactoryObjectPtr objects;', 'int maxObjects;', 'int midiClockPort;', 'int midiPort;', 'int midiChannel;');
  genEnumStruct("struct", "SynFactoryPatch", \@SynFactoryPatchStructDef);

  $rc .= "static const char * objectHelpText(ObjectType type) {\n  switch(type) {\n${objectHelpText}  default: return \"No help available for module\";\n  }\n}\n\n";
  $rc .= "static bool autoSize(ObjectType type) {\n  switch(type) {\n${objectAutoSize}  default: return false;\n  }\n}\n\n";
  $rc .= "static int objectBufferSize(ObjectType type) {\n  switch(type) {${objectBufferSize}\n  default: return 0;\n  }\n}\n\n";
  $rc .= "static bool objectBufferKeep(ObjectType type) {\n  switch(type) {${objectBufferKeep}\n  default: return false;\n  }\n}\n\n";

  $rc .= "static const char *knobName(ObjectType type, int nr) {\n";
  $rc .= "  switch(nr) {\n";
  for(my $i=0;$i<100;$i++) {
    if (defined($objectKnobs[$i])) {
      $rc .= "  case $i:\n";
      $rc .= "    switch(type) {\n";
      $rc .= $objectKnobs[$i];
      $rc .= "    default: break;\n    }\n";
      $rc .= "    break;\n";
    }
  }
  $rc .= "  }\n  return NULL;\n}\n\n";

  $rc .= "static int knobDefaultValue(ObjectType type, int nr) {\n";
  $rc .= "  switch(nr) {\n";
  for(my $i=0;$i<100;$i++) {
    if (defined($objectKnobsValue[$i])) {
      $rc .= "  case $i:\n";
      $rc .= "    switch(type) {\n";
      $rc .= $objectKnobsValue[$i];
      $rc .= "    default: break;\n    }\n";
      $rc .= "    break;\n";
    }
  }
  $rc .= "  default:\n    break;\n";
  $rc .= "  }\n  return 0;\n}\n\n";

  $rc .= "static TranslationFunc knobTranslateFunction(ObjectType type, int nr) {\n";
  $rc .= "  switch(nr) {\n";
  for(my $i=0;$i<100;$i++) {
    if (defined($objectKnobsTranslation[$i])) {
      $rc .= "  case $i:\n";
      $rc .= "    switch(type) {\n";
      $rc .= $objectKnobsTranslation[$i];
      $rc .= "    default: break;\n    }\n";
      $rc .= "    break;\n";
    }
  }
  $rc .= "  default:\n    break;\n";
  $rc .= "  }\n  return NULL;\n}\n\n";

  $rc .= "static const char *inputName(ObjectType type, int nr) {\n";
  $rc .= "  if (autoSize(type) && nr<maxAutoSize) return \"\";\n";
  $rc .= "  switch(nr) {\n";
  for(my $i=0;$i<100;$i++) {
    if (defined($objectInputs[$i])) {
      $rc .= "  case $i:\n";
      $rc .= "    switch(type) {\n";
      $rc .= $objectInputs[$i];
      $rc .= "    default: break;\n    }\n";
      $rc .= "    break;\n";
    }
  }
  $rc .= "  }\n  return NULL;\n}\n\n";

  $rc .= "static const char *outputName(ObjectType type, int nr) {\n";
  $rc .= "  switch(nr) {\n";
  for(my $i=0;$i<100;$i++) {
    if (defined($objectOutputs[$i])) {
      $rc .= "  case $i:\n";
      $rc .= "    switch(type) {\n";
      $rc .= $objectOutputs[$i];
#      $rc .= "    default: return NULL;\n    }\n";
      $rc .= "    default: break;\n    }\n";
      $rc .= "    break;\n";
    }
  }
  $rc .= "  }\n  return NULL;\n}\n\n";

  $rc .= <<EOF;
static void drawPot(ContextPtr aContext, int x, int y, const char *name, int value, int color, int highlight, int shadow, bool halfRangePot, TranslationFunc transfunc1, TranslationFunc transfunc2) {
  GuiRect myrect;
  char tmpstr[64];
  int insidex, insidey, outsidex, outsidey;

  guiSelectPotFont(aContext);
  SetBkMode(aContext->hdc, TRANSPARENT);
  SetTextAlign(aContext->hdc, TA_TOP | TA_CENTER);


  guiSelectPen1Color(aContext, shadow);
  guiSelectFillColor(aContext, shadow);
//  SelectObject(hdc, ColorPen1[0]);
//  SelectObject(hdc, ColorBrush[0]);
  myrect.left=x+2;
  myrect.right=x+27;
  myrect.top=y+2;
  myrect.bottom=y+27;
  guiDrawEllipse(aContext, &myrect);

//  Ellipse(aContext->hdc, x+2, y+2, x + 27, y + 27);

  guiSelectPen1Color(aContext, color);
//  SelectObject(hdc, ColorPen1[color&7]);
//  SelectObject(hdc, GetStockObject(NULL_PEN));

//  SelectObject(hdc, PotGreenBrush1);
//  Ellipse(hdc, x-1, y-1, x + 24, y + 24);

  guiSelectFillColor(aContext, color);
  myrect.left=x;
  myrect.right=x+25;
  myrect.top=y;
  myrect.bottom=y+25;
  guiDrawEllipse(aContext, &myrect);
//  Ellipse(aContext->hdc, x, y, x + 25, y + 25);

  if (halfRangePot) {
    insidex  = x+12+(int)(sin((value-16384)/7000.0)*  5.0);
    insidey  = y+12+(int)(cos((value-16384)/7000.0)*- 5.0);
    outsidex = x+12+(int)(sin((value-16384)/7000.0)* 12.0);
    outsidey = y+12+(int)(cos((value-16384)/7000.0)*-12.0);
  } else {
    insidex  = x+12+(int)(sin(value/14000.0)*  5.0);
    insidey  = y+12+(int)(cos(value/14000.0)*- 5.0);
    outsidex = x+12+(int)(sin(value/14000.0)* 12.0);
    outsidey = y+12+(int)(cos(value/14000.0)*-12.0);
  }
  guiSelectPen3Color(aContext, highlight);
//  SelectObject(hdc, ColorPen3[15]);
  MoveToEx(aContext->hdc, insidex, insidey, NULL);
  LineTo(aContext->hdc,outsidex,outsidey);
  
  guiSelectPen1Color(aContext, shadow);
//  SelectObject(hdc, GetStockObject(BLACK_PEN));

  // Pot name
  if (name) {
    TextOut(aContext->hdc, x + 15, y + 26, name, (int)strlen(name));
  }

  // Pot value
  sprintf(tmpstr, "%d", value);
  TextOut(aContext->hdc, x + 15, y + ((!!name)?38:26), tmpstr, (int)strlen(tmpstr));

  // Secondary (translated) pot value
  if (transfunc1) {
    const char *convertedstr=(*transfunc1)(value);
    TextOut(aContext->hdc, x + 15, y + ((!!name)?50:38), convertedstr, (int)strlen(convertedstr));
  }
  if (transfunc2) {
    const char *convertedstr=(*transfunc2)(value);
    TextOut(aContext->hdc, x + 15, y + ((!!name)?62:50), convertedstr, (int)strlen(convertedstr));
  }
}

static DspRoutinePtr addRoutines[maxAutoSize]={
  dspRoutine_ADD1,  dspRoutine_ADD2,  dspRoutine_ADD3,  dspRoutine_ADD4,  dspRoutine_ADD5,  dspRoutine_ADD6,  dspRoutine_ADD7, dspRoutine_ADD8,
//   dspRoutine_ADD9,  dspRoutine_ADD10, dspRoutine_ADD11, dspRoutine_ADD12, dspRoutine_ADD13, dspRoutine_ADD14, dspRoutine_ADD15,
//  dspRoutine_ADD16, dspRoutine_ADD17, dspRoutine_ADD18, dspRoutine_ADD19, dspRoutine_ADD20, dspRoutine_ADD21, dspRoutine_ADD22, dspRoutine_ADD23,
//  dspRoutine_ADD24, dspRoutine_ADD25, dspRoutine_ADD26, dspRoutine_ADD27, dspRoutine_ADD28, dspRoutine_ADD29, dspRoutine_ADD30, dspRoutine_ADD31,
};
static DspRoutinePtr andRoutines[maxAutoSize]={
  dspRoutine_AND1,  dspRoutine_AND2,  dspRoutine_AND3,  dspRoutine_AND4,  dspRoutine_AND5,  dspRoutine_AND6,  dspRoutine_AND7, dspRoutine_AND8, 
//   dspRoutine_AND9,  dspRoutine_AND10, dspRoutine_AND11, dspRoutine_AND12, dspRoutine_AND13, dspRoutine_AND14, dspRoutine_AND15,
//  dspRoutine_AND16, dspRoutine_AND17, dspRoutine_AND18, dspRoutine_AND19, dspRoutine_AND20, dspRoutine_AND21, dspRoutine_AND22, dspRoutine_AND23,
//  dspRoutine_AND24, dspRoutine_AND25, dspRoutine_AND26, dspRoutine_AND27, dspRoutine_AND28, dspRoutine_AND29, dspRoutine_AND30, dspRoutine_AND31,
};
static DspRoutinePtr orRoutines[maxAutoSize]={
  dspRoutine_OR1,  dspRoutine_OR2,  dspRoutine_OR3,  dspRoutine_OR4,  dspRoutine_OR5,  dspRoutine_OR6,  dspRoutine_OR7, dspRoutine_OR8, 
//    dspRoutine_OR9,  dspRoutine_OR10, dspRoutine_OR11, dspRoutine_OR12, dspRoutine_OR13, dspRoutine_OR14, dspRoutine_OR15,
//   dspRoutine_OR16, dspRoutine_OR17, dspRoutine_OR18, dspRoutine_OR19, dspRoutine_OR20, dspRoutine_OR21, dspRoutine_OR22, dspRoutine_OR23,
//   dspRoutine_OR24, dspRoutine_OR25, dspRoutine_OR26, dspRoutine_OR27, dspRoutine_OR28, dspRoutine_OR29, dspRoutine_OR30, dspRoutine_OR31,
};
static void setDspRoutine(SynFactoryPatchPtr patch, int obj) {
  DspRoutinePtr routine;
  int i;

  switch(patch->objects[obj].objectType) {$objectRoutine
  case OBJ_ADD:
    routine=dspRoutine_Dummy;
    for(i=maxAutoSize-1;i>=0;i--) {
      if (patch->objects[obj].connections[i].fromObject) {
        routine=addRoutines[i];
        break;
      }
    }
    break;
  case OBJ_AND:
    routine=dspRoutine_Dummy;
    for(i=maxAutoSize-1;i>=0;i--) {
      if (patch->objects[obj].connections[i].fromObject) {
        routine=andRoutines[i];
        break;
      }
    }
    break;
  case OBJ_OR:
    routine=dspRoutine_Dummy;
    for(i=maxAutoSize-1;i>=0;i--) {
      if (patch->objects[obj].connections[i].fromObject) {
        routine=orRoutines[i];
        break;
      }
    }
    break;
  default:
    routine=NULL;
    break;
  }
  if (routine==dspRoutine_Dummy) {
    // Clear output of autosize modules if not connected (ADD, AND, OR)
    projects[currentProject].dsp[patch->objects[obj].dsp].io[0].output=0;
  }
  projects[currentProject].dsp[patch->objects[obj].dsp].dspRoutine=routine;
}

static void setMode(SynFactoryPatchPtr patch, int obj, int mode) {
  DspRoutinePtr routine=projects[currentProject].dsp[patch->objects[obj].dsp].dspRoutine;

  switch(patch->objects[obj].objectType) {
  case OBJ_OSC:
    switch(mode) {
    case 0:  routine=dspRoutine_OSC_SIN; break;
    case 1:  routine=dspRoutine_OSC_TRI; break;
    case 2:  routine=dspRoutine_OSC_SAW; break;
    case 3:  routine=dspRoutine_OSC_SPS; break;
    case 4:  routine=dspRoutine_OSC_PLS; break;
    case 5:  routine=dspRoutine_OSC_RND; break;
    default: routine=dspRoutine_Dummy; break;
    }
    break;
  default:
    break;
  }

  projects[currentProject].dsp[patch->objects[obj].dsp].dspRoutine=routine;
  patch->objects[obj].mode=mode;
}

static int calcObjectXs(ObjectType type) {
  int i;
  int xs=knobXOffset;
  int margin=outputXMargin2;

  switch(type) {
  case OBJ_DRM4:
  case OBJ_DRM8:
    margin = outputXMargin;
    xs += (16*16);
    break;
  default:
    for(i=0;i<maxObjectIo;i++) {
      if (outputName(type, i) && strlen(outputName(type, i))>0) {
        margin = outputXMargin;
      }
      if (knobName(type, i)) {
        xs = knobXOffset + knobXSpacing * ((i>>1)+1);
      }
    }
    break;
  }
  xs += margin;
  if (xs < minXSize) {
    xs = minXSize;
  }

  return xs;
}

static int calcObjectYs(ObjectType type, int autoSizeOffset) {
  int i;
  int maxIo=autoSizeOffset;

  for(i=0;i<maxObjectIo;i++) {
    if (autoSizeOffset==0 && i>maxIo && (inputName(type, i) || outputName(type,i))) {
      maxIo=i;
    }
    if (maxIo<2 && (i&1)==0 && knobName(type, i)) {
      maxIo=2;
    }
    if (maxIo<6 && (i&1)==1 && knobName(type, i)) {
      maxIo=6;
    }
  }

  return (maxIo+3)*ioYSpacing+ioYSize;
}

static void adjustAutoSize(SynFactoryPatchPtr patch, int obj) {
  int max=1;
  int i;

  for(i=1;i<maxAutoSize-1;i++) {
    if (patch->objects[obj].connections[i].fromObject) {
      max=i+1;
    }
  }
  if (patch->objects[obj].connections[maxAutoSize-1].fromObject) {
    max=maxAutoSize-1;
  }
  
  patch->objects[obj].maxInput=max;
  patch->objects[obj].ys=calcObjectYs(patch->objects[obj].objectType, max);
  setDspRoutine(patch, obj);
}

static int addSynFactoryObject(int patch, ObjectType type, bool keepSelection) {
  int n=projects[currentProject].patches[patch].maxObjects;
  int dsp=0;

  logprintf("addSynFactoryObject type %d\\n", (int)type);

  if ((type != OBJ_BOX) && (type != OBJ_ROUNDBOX) && (type != OBJ_ELLIPSE) && (type != OBJ_TEXT) && (type != OBJ_HTML)) {
    dsp=nextFreeDsp();
    if (0 == dsp) {
      return -1;
    }
    if (objectBufferSize(type)) {
      projects[currentProject].dsp[dsp].buffer = (signed short*)calloc(1, objectBufferSize(type));
      if (NULL == projects[currentProject].dsp[dsp].buffer) {
        return -1;
      }
    }
    for(int i=0;i<maxObjectIo;i++) {
      projects[currentProject].dsp[dsp].io[i].knob=knobDefaultValue(type, i);      
    }

    projects[currentProject].dsp[dsp].inUse=true;
  }

  projects[currentProject].patches[patch].maxObjects++;
  projects[currentProject].patches[patch].objects=(SynFactoryObjectPtr)realloc(projects[currentProject].patches[patch].objects, projects[currentProject].patches[patch].maxObjects*sizeof(SynFactoryObject));
  memset(&(projects[currentProject].patches[patch].objects[n]), 0, sizeof(SynFactoryObject));
  projects[currentProject].patches[patch].objects[n].xs=calcObjectXs(type);
  projects[currentProject].patches[patch].objects[n].ys=calcObjectYs(type, (autoSize(type))?1:0);
  projects[currentProject].patches[patch].objects[n].objectType=type;
  projects[currentProject].patches[patch].objects[n].maxInput=1; /* only for autoSize objects ADD AND OR */
  projects[currentProject].patches[patch].objects[n].dsp=dsp;
  if (type==OBJ_NOTESCRIPT || type==OBJ_TEXT || type==OBJ_HTML) {
    projects[currentProject].patches[patch].objects[n].window=createNoteScriptEditorWindow(currentProject, patch, n, "edit");
  }
  if (type==OBJ_SCOPE) {
    projects[currentProject].patches[patch].objects[n].window=createMultiScopeWindow(&(projects[currentProject].dsp[projects[currentProject].patches[patch].objects[n].dsp]));
  }
  setDspRoutine(&(projects[currentProject].patches[patch]), n);
  projects[currentProject].modifiedFlag=true;
  projectBrowserWindowRefresh=true;
  patchConfigWindowRefresh=true;
  mainWindowRefresh=true;
  selectProjectBrowserItem(currentProject, &(projects[currentProject].patches[patch]), n, -1, keepSelection);

  logprintf("addSynFactoryObject type %d returns %d\\n", (int)type, n);
  return n;
}


static void disconnectCable(SynFactoryPatchPtr patch, int obj, int port) {
  logprintf("disconnectCable obj %d port %d\\n", obj, port);
  patch->objects[obj].connections[port].fromObject = 0;
  projects[currentProject].dsp[patch->objects[obj].dsp].io[port].input=&(projects[currentProject].dsp[0].io[0].output);

  if (autoSize(patch->objects[obj].objectType)) {
    adjustAutoSize(patch, obj);
  }
}

// Connect from currently active output to specified object and input.
static void connectCable(SynFactoryPatchPtr patch, int obj, int port) {
  int i;

  logprintf("connectCable obj %d port %d\\n", obj, port);
  for(i=0;i<patch->maxObjects;i++) {
    if (patch->objects[i].currentOutput) {
      logprintf("Found output %d, %d\\n", i, patch->objects[i].currentOutput);
      patch->objects[obj].connections[port].fromObject = i+1;
      patch->objects[obj].connections[port].fromOutput = patch->objects[i].currentOutput;
      patch->objects[obj].connections[port].cableColor = currentCableColorIndex;
      projects[currentProject].dsp[patch->objects[obj].dsp].io[port].input=&(projects[currentProject].dsp[patch->objects[i].dsp].io[patch->objects[i].currentOutput-1].output);
      if (autoSize(patch->objects[obj].objectType)) {
        adjustAutoSize(patch, obj);
      }
      break;
    }
  }
}

// Toggle visability of the extra label for all selected objects
static void toggleLabel(void) {
  if (currentProject>=0 && currentPatch>=0) {
    for(int i=0;i<projects[currentProject].patches[currentPatch].maxObjects;i++) {
      if (projects[currentProject].patches[currentPatch].objects[i].dsp && projects[currentProject].patches[currentPatch].objects[i].selected) {
        projects[currentProject].patches[currentPatch].objects[i].showLabel=!projects[currentProject].patches[currentPatch].objects[i].showLabel;
        mainWindowRefresh=true;
      }
    }
  }
}

// delete all selected objects
static void deleteSynFactoryObjects(int project, SynFactoryPatchPtr patch, bool deleteAll) {
  int i,j;
  int s=0;
  int d=0;
  
  while (s<patch->maxObjects) {
    if (patch->objects[s].selected || deleteAll) {
      logprintf("delete object %d (type %d)\\n", s, (int)patch->objects[s].objectType);
      // Remove all connections from this object output to others (his inputs are not relevant)
      for(i=0;i<patch->maxObjects;i++) {        
        for(j=0;j<maxObjectIo;j++) {
          if (patch->objects[i].connections[j].fromObject == s+1) {
            disconnectCable(patch, i, j);
          }
        }
      }
      if (patch->objects[s].window) {
        DestroyWindow(patch->objects[s].window);
      }
      if (patch->objects[s].name) {
        free(patch->objects[s].name);
      }
      if (patch->objects[s].dsp) {
        lockMutex(audioMutex);
        projects[project].dsp[patch->objects[s].dsp].dspRoutine=dspRoutine_Dummy;
        projects[project].dsp[patch->objects[s].dsp].inUse=false;
        if (patch->objects[s].objectType==OBJ_NOTESCRIPT && projects[project].dsp[patch->objects[s].dsp].scriptInfo->script) {
          free(projects[project].dsp[patch->objects[s].dsp].scriptInfo->script);
        }
        if (projects[project].dsp[patch->objects[s].dsp].buffer) {
          free(projects[project].dsp[patch->objects[s].dsp].buffer);
        }
        unlockMutex(audioMutex);
      }
    } else {
      if (d!=s) {
        patch->objects[d]=patch->objects[s];
        // Rewire all objects inputs to new index of object
        for(i=0;i<patch->maxObjects;i++) {
          for(j=0;j<maxObjectIo;j++) {
            if (patch->objects[i].connections[j].fromObject == s+1) {
              patch->objects[i].connections[j].fromObject = d+1;
            }
          }
        }
      }
      d++;
    }
    s++;
  }
  if (d != patch->maxObjects) {
    patch->maxObjects=d;
    patch->objects=(SynFactoryObjectPtr)realloc(patch->objects, patch->maxObjects*sizeof(SynFactoryObject));
    projectBrowserWindowRefresh=true;
    mainWindowRefresh=true;
  }
}

/*static void moveOneUpSynFactoryObject(SynFactoryPatch *patch, int objectIndex) {
  SynFactoryObject tmp;
  
  if (objectIndex>patch->maxObjects-1) {
    tmp=patch->objects[objectIndex];
    patch->objects[objectIndex]=patch->objects[objectIndex+1];
    patch->objects[objectIndex]=tmp;
  }  
}*/

static int createSynfactoryPatch(void) {
  char tmpName[256];
  int n=-1;

  logprintf("createSynfactoryPatch\\n");

  if (projects==NULL) {
    createProject(newProjectName);
  }

  if (currentPlayingProject==currentProject) {
    lockMutex(audioMutex);
    lockMutex(midiMutex);
  }

  n=projects[currentProject].maxPatches;
  projects[currentProject].maxPatches++;
  projects[currentProject].patches=(SynFactoryPatchPtr)realloc(projects[currentProject].patches, projects[currentProject].maxPatches*sizeof(SynFactoryPatch));
  memset(&(projects[currentProject].patches[n]), 0, sizeof(SynFactoryPatch));
  sprintf(tmpName, "patch %d", ++projects[currentProject].patchCount);
  projects[currentProject].patches[n].patchName=strdup(tmpName);
  projects[currentProject].modifiedFlag=true;
  if (currentPlayingProject==currentProject) {
    unlockMutex(audioMutex);
    unlockMutex(midiMutex);
  }
  currentPatch=n;
  projectBrowserWindowRefresh=true;
  patchConfigWindowRefresh=true;
  mainWindowRefresh=true;

  logprintf("createSynfactoryPatch returns %d\\n", n);
  return n;
}

typedef enum _MouseMode {
  MOUSE_NONE,
  MOUSE_DRAGSCREEN,          // Right mouse button with movement / Popup menu when just click
  MOUSE_DRAGSCREENZOOM,      // Drag screen and press left mouse button
  MOUSE_DRAGOBJECT,          // Left mouse button with movement.
  MOUSE_DRAGOBJECTANDSCREEN, // While drag of object press right mousebutton to drag screen
  MOUSE_DRAGCABLE,           // Left mouse button with drawing cable to other object.
  MOUSE_DRAGCABLEANDSCREEN,  // While drag of cable press right mousebutton to drag screen
  MOUSE_DRAGRECT,            // Paint selection rectangle
  MOUSE_DRAGRECTANDSCREEN,   // Paint selection rectangle and drag screen with right mouse button

  MOUSE_KNOB,
  MOUSE_KNOBSLOW,

  MOUSE_RESIZE_TOP_LEFT,
  MOUSE_RESIZE_TOP_RIGHT,
  MOUSE_RESIZE_BOTTOM_LEFT,
  MOUSE_RESIZE_BOTTOM_RIGHT,

  MOUSE_EDIT_LABEL,
//  MOUSE_
} MouseMode;
static MouseMode mouseMode = MOUSE_NONE;
static int startX; // first location of drawing operation (second location is current mouse pos)
static int startY; // first location of drawing operation (second location is current mouse pos)
static int oldMouseX;
static int oldMouseY;
static int oldMouseClientX;
static int oldMouseClientY;
static int mouseDownTicks=0;
static int redrawDelay;
static bool XorFlag;
static int editObject;
static int knobIndex;
static void storeMouse(ContextPtr aContext) {
  oldMouseX = aContext->mouseX;
  oldMouseY = aContext->mouseY;
  oldMouseClientX = aContext->mouseClientX;
  oldMouseClientY = aContext->mouseClientY;
}

static bool dragScreen(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int deltaX = aContext->mouseX - oldMouseX;
  int deltaY = aContext->mouseY - oldMouseY;

  if (mouseMode==MOUSE_DRAGSCREENZOOM) {
    patch->offsX -= deltaX*5;
    patch->offsY -= deltaY*5;
  } else {
    patch->offsX -= deltaX;
    patch->offsY -= deltaY;
  }

  storeMouse(aContext);
  return (0!=deltaX || 0!=deltaY);
}

static void updateDragScreen(ContextPtr aContext, SynFactoryPatchPtr patch) {
  mouseDownTicks++;
  if (dragScreen(aContext, patch)) {
    redrawDelay = 4;
    mainWindowRefresh=true;
  } else {
    // Repaint window with all details if mouse hasn't moved some time.
    if (redrawDelay > 0) {
      redrawDelay--;
      if (redrawDelay == 0) {
        mainWindowRefresh=true;
      }
    }
  }
}

static void selectOutput(SynFactoryPatchPtr patch, int object, int output) {
  int i;
  
  // reset all other outputs
  for(i=0;i<patch->maxObjects;i++) {
    patch->objects[i].currentOutput=0;
  }

  patch->objects[object].currentOutput=output;
  mainWindowRefresh=true;
}

static void selectObjectsInArea(SynFactoryPatchPtr patch, int x1, int y1, int x2, int y2) {
  int i;
  bool flag=ctrlPressed();

  for(i=patch->maxObjects-1;i>=0;i--) {
    SynFactoryObject *curObj=&(patch->objects[i]);
    if (x1<curObj->x && x2>curObj->x+curObj->xs &&
        y1<curObj->y && y2>curObj->y+curObj->ys) {
      selectProjectBrowserItem(currentProject, patch, i, -1, flag);
      flag=true;
    }
  }
}

static int findObject(SynFactoryPatchPtr patch, int x, int y, MouseMode *newMouseMode) {
  int i;

  for(i=patch->maxObjects-1;i>=0;i--) {
    SynFactoryObject *curObj=&(patch->objects[i]);
    if (curObj->dsp>0 || showResizeMarkers) {
      if (MOUSE_NONE==mouseMode && curObj->showLabel &&
          x>=curObj->x && x<curObj->x+curObj->labelXs &&
          y>=curObj->y-18 && y<curObj->y-2) {
          *newMouseMode=MOUSE_EDIT_LABEL;
          return i;
      }
      if (x>=curObj->x && x<curObj->x+curObj->xs &&
          y>=curObj->y && y<curObj->y+curObj->ys) {
        if (y < curObj->y+2*ioYSpacing) {
          *newMouseMode = MOUSE_DRAGOBJECT;
        } else {
          int j;

          for(j=0;j<maxObjectIo;j++) {
            // inputs
            if ((x >= curObj->x) &&
                (x <= curObj->x+ioXSize) &&
                (y >= curObj->y+(j+2)*ioYSpacing) &&
                (y <= curObj->y+(j+2)*ioYSpacing+ioYSize) &&
                inputName(curObj->objectType, j)) {
              if ((curObj->connections[j].fromObject==0)
                  || (MOUSE_DRAGCABLE==mouseMode)) {
                connectCable(patch, i, j);
              } else {
                disconnectCable(patch, i, j);
              }
            }
            // outputs
            if ((x >= curObj->x+curObj->xs-ioXSize) &&
                (x <= curObj->x+curObj->xs) &&
                (y >= curObj->y+(j+2)*ioYSpacing) &&
                (y <= curObj->y+(j+2)*ioYSpacing+ioYSize) &&
                outputName(curObj->objectType, j)) {
              selectOutput(patch, i, j+1);
              *newMouseMode = MOUSE_DRAGCABLE;
              break;
            }
            // knobs
            if ((x >= curObj->x + (j>>1)*knobXSpacing + knobXOffset) &&
                (x <= curObj->x + (j>>1)*knobXSpacing + knobXOffset +30) &&
                (y >= curObj->y + ioYSpacing*(2 + ((j&1)?4:0))) &&
                (y <= curObj->y + ioYSpacing*(2 + ((j&1)?4:0)) + 30) &&
                knobName(curObj->objectType, j)) {
              editObject=i;
              knobIndex=j;
              *newMouseMode = MOUSE_KNOB;
              break;
            }
          }
        }

        // OSC specific actions
        if (curObj->objectType == OBJ_OSC) {
          if ((x >= curObj->x +  90) &&
              (x <  curObj->x + 110) &&
              (y >= curObj->y + ioYSpacing*6 + 20) &&
              (y <  curObj->y + ioYSpacing*6 + 40)) {
            setMode(patch, i, 0);
          }
          if ((x >= curObj->x +  90) &&
              (x <  curObj->x + 110) &&
              (y >= curObj->y + ioYSpacing*6) &&
              (y <  curObj->y + ioYSpacing*6 + 20)) {
            setMode(patch, i, 1);
          }
          if ((x >= curObj->x + 110) &&
              (x <  curObj->x + 130) &&
              (y >= curObj->y + ioYSpacing*6) &&
              (y <  curObj->y + ioYSpacing*6 + 20)) {
            setMode(patch, i, 2);
          }
          if ((x >= curObj->x + 130) &&
              (x <  curObj->x + 150) &&
              (y >= curObj->y + ioYSpacing*6 + 20) &&
              (y <  curObj->y + ioYSpacing*6 + 40)) {
            setMode(patch, i, 3);
          }
          if ((x >= curObj->x + 130) &&
              (x <  curObj->x + 150) &&
              (y >= curObj->y + ioYSpacing*6) &&
              (y <  curObj->y + ioYSpacing*6 + 20)) {
            setMode(patch, i, 4);
          }
          if ((x >= curObj->x + 110) &&
              (x <  curObj->x + 130) &&
              (y >= curObj->y + ioYSpacing*6 + 20) &&
              (y <  curObj->y + ioYSpacing*6 + 40)) {
            setMode(patch, i, 5);
          }
        }

        // DRUM specific actions
        if (curObj->objectType == OBJ_DRM4 || curObj->objectType == OBJ_DRM8) {
          int col = (x + 10 - (curObj->x+knobXOffset)) / 16;
          int row = (y - (curObj->y+ioYSpacing*2)) / (ioYSpacing*2);

          if (col>=0 && col<16 && (y >= curObj->y+ioYSpacing*2) && row<((curObj->objectType == OBJ_DRM4)?4:8)) {
            ((unsigned char *)projects[currentProject].dsp[curObj->dsp].buffer)[row*16+col]^=1;
            mainWindowRefresh=true;
          }
        }

        // take first object found
        return i;
      } else  if (curObj->dsp==0) {
        // Resize icons
        editObject=i;
        if (x>curObj->x-6 && x<curObj->x && y>curObj->y-6 && y<curObj->y) {
          *newMouseMode = MOUSE_RESIZE_TOP_LEFT;
          return i;
        }
        if (x>curObj->x+curObj->xs && x<curObj->x+curObj->xs+6 && y>curObj->y-6 && y<curObj->y) {
          *newMouseMode = MOUSE_RESIZE_TOP_RIGHT;
          return i;
        }
        if (x>curObj->x-6 && x<curObj->x && y>curObj->y+curObj->ys && y<curObj->y+curObj->ys+6) {
          *newMouseMode = MOUSE_RESIZE_BOTTOM_LEFT;
          return i;
        }
        if (x>curObj->x+curObj->xs && x<curObj->x+curObj->xs+6 && y>curObj->y+curObj->ys && y<curObj->y+curObj->ys+6) {
          *newMouseMode = MOUSE_RESIZE_BOTTOM_RIGHT;
          return i;
        }
      }
    }
  }
  return -1;
}

static void synfactoryLeftClick(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int obj=findObject(patch, aContext->mouseClientX + patch->offsX, aContext->mouseClientY + patch->offsY, &mouseMode);
  if (obj<0) {
    // mouse pressed on background activate select rectangle
//    storeMouse(aContext);
//    SetCapture(aContext->currentWindow);
    startX=aContext->mouseClientX+patch->offsX;
    startY=aContext->mouseClientY+patch->offsY;
    mouseMode=MOUSE_DRAGRECT;
  } else {
    selectProjectBrowserItem(currentProject, patch, obj, -1, ctrlPressed());
  }

  if (MOUSE_EDIT_LABEL == mouseMode) {
     mouseMode = MOUSE_NONE;
  }
  if (MOUSE_NONE != mouseMode) {
    storeMouse(aContext);
    SetCapture(aContext->currentWindow);
    if (MOUSE_KNOB == mouseMode) {
      guiHideMouse(aContext);
//      oldMouseX=aContext->mouseX;
//      oldMouseY=aContext->mouseY;
      SetCursorPos(GetSystemMetrics(SM_CXFULLSCREEN)/2, GetSystemMetrics(SM_CYFULLSCREEN)/2);
    }
  }
}

static void processPatchDoubleClick(ContextPtr aContext, SynFactoryPatchPtr patch) {
  MouseMode dummyMouseMode=MOUSE_NONE;
  int obj=findObject(patch, aContext->mouseClientX + patch->offsX, aContext->mouseClientY + patch->offsY, &dummyMouseMode);
//  mouseMode=MOUSE_NONE;

  if (obj>=0) {
    if (patch->objects[obj].window) {
      guiShowWindow(patch->objects[obj].window);
    }
  }
}

static bool dragObjects(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int deltaX = aContext->mouseX - oldMouseX;
  int deltaY = aContext->mouseY - oldMouseY;

  if (0!=deltaX || 0!=deltaY) {
    int i;

    for(i=0;i<patch->maxObjects;i++) {
      if (patch->objects[i].selected) {
        patch->objects[i].x += deltaX;
        patch->objects[i].y += deltaY;
      }
    }
  }

  oldMouseX = aContext->mouseX;
  oldMouseY = aContext->mouseY;
  return (0!=deltaX || 0!=deltaY);
}

static void updateDragObjects(ContextPtr aContext, SynFactoryPatchPtr patch) {
  if (dragObjects(aContext, patch)) {
    mainWindowRefresh=true;
  }
}

static void finishDragObjects(ContextPtr aContext, SynFactoryPatchPtr patch) {
  if (dragObjects(aContext, patch)) {
    mainWindowRefresh=true;
  }
  ReleaseCapture();
  mouseMode = MOUSE_NONE;  
}

static void finishDragCable(ContextPtr aContext, SynFactoryPatchPtr patch) {
  ReleaseCapture();
  findObject(patch, aContext->mouseClientX + patch->offsX, aContext->mouseClientY + patch->offsY, &mouseMode);
  mouseMode = MOUSE_NONE;  
  mainWindowRefresh=true;
}

static void updateModuleLabel(SynFactoryPatchPtr patch, int obj, int chr) {
  if (chr==8) {
    if (patch->objects[obj].name) {
      char *oldName=patch->objects[obj].name;
      int len=strlen(oldName);
      if (len<2) {
        patch->objects[obj].name=NULL;
      } else {
        patch->objects[obj].name=(char *)malloc(len);
        memcpy(patch->objects[obj].name, oldName, len);
        patch->objects[obj].name[len-1]='\\0';
      }
      free(oldName);
    }
  }
  if (chr>=32) {
    if (patch->objects[obj].name) {
      char *oldName=patch->objects[obj].name;
      int len=strlen(oldName);
      patch->objects[obj].name=(char *)malloc(len+2);
      strcpy(patch->objects[obj].name, oldName);
      patch->objects[obj].name[len]=(char)chr;
      patch->objects[obj].name[len+1]='\\0';
      free(oldName);
    } else {
      patch->objects[obj].name=(char *)malloc(2);
      patch->objects[obj].name[0]=(char)chr;
      patch->objects[obj].name[1]='\\0';
    }
  }

  patch->objects[obj].labelXs=0;
  mainWindowRefresh=true;
}

static void updateKnob(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int incr;
  int temp;

  // Only redraw on vertical movement
  if (aContext->mouseY != GetSystemMetrics(SM_CYFULLSCREEN)/2) {
    incr = GetSystemMetrics(SM_CYFULLSCREEN)/2 - aContext->mouseY;
    if (MOUSE_KNOBSLOW == mouseMode) {
      incr = incr / 4;
    } else {
      incr = incr * 16;
    }

    temp = projects[currentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob + incr;
    if (temp > 32767) temp = 32767;
    if (temp < -32768) temp = -32768;
    projects[currentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob = temp;
    SetCursorPos(GetSystemMetrics(SM_CXFULLSCREEN)/2, GetSystemMetrics(SM_CYFULLSCREEN)/2);

    mainWindowRefresh=true;
  }
}

static void resizeObject(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int deltaX = aContext->mouseX - oldMouseX;
  int deltaY = aContext->mouseY - oldMouseY;

  if (0!=deltaX || 0!=deltaY) {
    switch(mouseMode) {
    case MOUSE_RESIZE_TOP_LEFT:
      patch->objects[editObject].x+=deltaX;
      patch->objects[editObject].xs-=deltaX;
      patch->objects[editObject].y+=deltaY;
      patch->objects[editObject].ys-=deltaY;
      break;
    case MOUSE_RESIZE_TOP_RIGHT:
      patch->objects[editObject].xs+=deltaX;
      patch->objects[editObject].y+=deltaY;
      patch->objects[editObject].ys-=deltaY;
      break;
    case MOUSE_RESIZE_BOTTOM_LEFT:
      patch->objects[editObject].x+=deltaX;
      patch->objects[editObject].xs-=deltaX;
      patch->objects[editObject].ys+=deltaY;
      break;
    case MOUSE_RESIZE_BOTTOM_RIGHT:
      patch->objects[editObject].xs+=deltaX;
      patch->objects[editObject].ys+=deltaY;
      break;
    default:
      break;
    }
    if (patch->objects[editObject].xs<0) patch->objects[editObject].xs=0;
    if (patch->objects[editObject].ys<0) patch->objects[editObject].ys=0;
    mainWindowRefresh=true;
  }

  oldMouseX = aContext->mouseX;
  oldMouseY = aContext->mouseY; 
}

static void drawObjectsZoomed(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int i,j;
  int offsX=(aContext->guiClientRect.right-aContext->guiClientRect.left)/5*2-patch->offsX/5;
  int offsY=(aContext->guiClientRect.bottom-aContext->guiClientRect.top)/5*2-patch->offsY/5;
  

  for(i=0;i<patch->maxObjects;i++) {
    SynFactoryObject *curObj=&(patch->objects[i]);        
    GuiRect rect2={(curObj->x)/5+offsX,(curObj->y)/5+offsY,(curObj->x+curObj->xs)/5+offsX,(curObj->y+curObj->ys)/5+offsY};
    guiSelectPen1Color(aContext, 0);
    guiSelectFillColor(aContext, defcolor_patchModules);
    guiDrawRoundRect(aContext, &rect2, 4, 4);
    guiSelectFillColor(aContext, defcolor_patchKnobs);
    for(j=0;j<maxObjectIo;j++) {
      if (knobName(curObj->objectType, j)) {
        rect2.left=(curObj->x+(j>>1)*knobXSpacing+knobXOffset)/5+offsX;
        rect2.top=(curObj->y+ioYSpacing*(2+(j&1)*4))/5+offsY;
        rect2.right=rect2.left+6;
        rect2.bottom=rect2.top+6;
        guiDrawEllipse(aContext, &rect2);
      }
    }
  }        

  guiSelectFillColor(aContext, -1);
  guiSelectPen1Color(aContext, 15);
  guiDrawRect(aContext,
    (aContext->guiClientRect.right-aContext->guiClientRect.left)/5*2,
    (aContext->guiClientRect.bottom-aContext->guiClientRect.top)/5*2,
    (aContext->guiClientRect.right-aContext->guiClientRect.left)/5*3,
    (aContext->guiClientRect.bottom-aContext->guiClientRect.top)/5*3);
}

static void drawResizeMarkers(ContextPtr aContext, int left, int top, int right, int bottom, bool selected) {
  guiSelectPen3Color(aContext, (selected)?defcolor_patchHighlight:defcolor_patchShadow);
  guiSelectFillColor(aContext, (selected)?defcolor_patchShadow:defcolor_patchModules);
  guiDrawRect(aContext, left-6, top-6, left, top);
  guiDrawRect(aContext, right, top-6, right+6, top);
  guiDrawRect(aContext, left-6, bottom, left, bottom+6);
  guiDrawRect(aContext, right, bottom, right+6, bottom+6);
}

static void drawObject(ContextPtr aContext, SynFactoryPatchPtr patch, SynFactoryObjectPtr object) {
  int i;

  // Check if object is completely off screen. Only draw modules partly or completely visible
  if (object->x+object->xs-patch->offsX<aContext->guiClientRect.left ||
      object->x-patch->offsX>aContext->guiClientRect.right ||
      object->y+object->ys-patch->offsY<aContext->guiClientRect.top ||
      object->y-patch->offsY>aContext->guiClientRect.bottom) {
    return;
  }

  if (object->dsp>0) {
    if (object->labelXs<48) {
      if (object->name) {
        RECT myRect={0,0,0,0};
        guiSelectPotFont(aContext);
        DrawText(aContext->hdc, object->name, -1, &myRect, DT_CALCRECT);
        object->labelXs=myRect.right+12;  
      }
      if (object->labelXs<48) {
        object->labelXs=48;
      }
    }

    guiSelectPen1Color(aContext, defcolor_patchShadow);
    if (0 == redrawDelay) {
      guiSelectFillColor(aContext, defcolor_patchShadow);
      guiDrawRoundRect(aContext, object->x+2-patch->offsX, object->y+2-patch->offsY, object->x+object->xs+2-patch->offsX, object->y+object->ys+2-patch->offsY, 12, 12);
      if (object->showLabel) {
        guiDrawRoundRect(aContext, object->x+2-patch->offsX, -18+object->y-patch->offsY, object->x+object->labelXs+2-patch->offsX, -2+object->y-patch->offsY, 12, 12);
      }
    }
  //        guiDraw3DRect(aContext, &rect, 0);
    GuiRect rect2={object->x-patch->offsX,object->y-patch->offsY,object->x+object->xs-patch->offsX,object->y+object->ys-patch->offsY};
    guiSelectFillColor(aContext, defcolor_patchModules);
    guiDrawRoundRect(aContext, &rect2, 12, 12);
    if (object->showLabel) {
      guiDrawRoundRect(aContext, object->x-patch->offsX, -20+object->y-patch->offsY, object->x+object->labelXs-patch->offsX, -4+object->y-patch->offsY, 12, 12);
      if (object->name) {
        SetTextAlign(aContext->hdc, TA_TOP | TA_LEFT);
        guiSelectPen1Color(aContext, defcolor_patchShadow);
        guiSelectPotFont(aContext);
        guiTextTransparent(aContext, true);
        guiDrawText(aContext, object->x+6-patch->offsX, -18+object->y-patch->offsY, object->name, strlen(object->name));
      }
    }
  //        guiDraw3DRect(aContext, &rect2, 0);

    if (objectTitles[object->objectType]) {
      if (object->selected) {
        guiSelectFillColor(aContext, defcolor_patchHighlight);
        rect2.bottom=(object->y-patch->offsY) + 2*ioYSpacing-ioYSize;
        guiDrawRoundRect(aContext, &rect2, 12, 12);
      }
      SetBkMode(aContext->hdc, TRANSPARENT);
      guiSelectTitleFont(aContext);
      SetTextAlign(aContext->hdc, TA_TOP | TA_LEFT);
      TextOut(aContext->hdc, (object->x-patch->offsX)+4, (object->y-patch->offsY)+2, objectTitles[object->objectType], strlen(objectTitles[object->objectType]));
    }

    // Inputs
    guiSelectIOFont(aContext);
    SetTextAlign(aContext->hdc, TA_BASELINE | TA_LEFT);
    for(i=0;i<maxObjectIo;i++) {
      if (inputName(object->objectType, i) && (autoSize(object->objectType)==false || object->maxInput>=i)) {
  //      myRect.left = (object->x-patch->offsX);
  //      myRect.top = (object->y-patch->offsY)+(i+2)*ioYSpacing;
  //      myRect.right = (object->x-patch->offsX)+ioXSize;
  //      myRect.bottom = (object->y-patch->offsY)+(i+2)*ioYSpacing+ioYSize;
        guiSelectFillColor(aContext, -1);
        guiSelectPen1Color(aContext, defcolor_patchShadow);
        guiDrawRect(aContext,
          (object->x-patch->offsX),
          (object->y-patch->offsY)+(i+2)*ioYSpacing,
          (object->x-patch->offsX)+ioXSize,
          (object->y-patch->offsY)+(i+2)*ioYSpacing+ioYSize);

        TextOut(aContext->hdc, (object->x-patch->offsX)+ioXSize+4, (object->y-patch->offsY)+(i+2)*ioYSpacing+ioYSize, inputName(object->objectType, i), strlen(inputName(object->objectType, i)));
      }
    }

    // Outputs
    SetTextAlign(aContext->hdc, TA_BASELINE | TA_RIGHT);
    for(i=0;i<maxObjectIo;i++) {
      if (outputName(object->objectType, i)) {
  //      myRect.left = (object->x-patch->offsX)+object->xs-ioXSize;
  //      myRect.top = (object->y-patch->offsY)+(i+2)*ioYSpacing;
  //      myRect.right = (object->x-patch->offsX)+object->xs;
  //      myRect.bottom = (object->y-patch->offsY)+(i+2)*ioYSpacing+ioYSize;
        if (object->currentOutput == i+1) {
          guiSelectFillColor(aContext, defcolor_patchShadow);
        } else {
          guiSelectFillColor(aContext, -1);
        }
        guiSelectPen1Color(aContext, defcolor_patchShadow);
        guiDrawRect(aContext,
          (object->x-patch->offsX)+object->xs-ioXSize,
          (object->y-patch->offsY)+(i+2)*ioYSpacing,
          (object->x-patch->offsX)+object->xs,
          (object->y-patch->offsY)+(i+2)*ioYSpacing+ioYSize);
        TextOut(aContext->hdc, (object->x+object->xs-patch->offsX)-(ioXSize+4), (object->y-patch->offsY)+(i+2)*ioYSpacing+ioYSize, outputName(object->objectType, i), strlen(outputName(object->objectType, i)));
      }
    }

    // Knobs
    for(i=0;i<maxObjectIo;i++) {
      if (knobName(object->objectType, i)) {
        drawPot(aContext, (object->x-patch->offsX)+(i>>1)*knobXSpacing+knobXOffset, (object->y-patch->offsY)+ioYSpacing*(2+(i&1)*4), knobName(object->objectType, i), projects[currentProject].dsp[object->dsp].io[i].knob, defcolor_patchKnobs, defcolor_patchHighlight, defcolor_patchShadow, false, knobTranslateFunction(object->objectType, i), NULL);
      }
    }
  }

  // Object specific drawing stuff
  switch(object->objectType) {
  case OBJ_OSC:
        DrawIcon(aContext->hdc, object->x-patch->offsX +  90, object->y-patch->offsY + ioYSpacing*6,    (HICON)((object->mode==1)?ICON_TRI1:ICON_TRI0));
        DrawIcon(aContext->hdc, object->x-patch->offsX +  90, object->y-patch->offsY + ioYSpacing*6+20, (HICON)((object->mode==0)?ICON_SIN1:ICON_SIN0));
        DrawIcon(aContext->hdc, object->x-patch->offsX + 110, object->y-patch->offsY + ioYSpacing*6,    (HICON)((object->mode==2)?ICON_SAW1:ICON_SAW0));
        DrawIcon(aContext->hdc, object->x-patch->offsX + 110, object->y-patch->offsY + ioYSpacing*6+20, (HICON)((object->mode==5)?ICON_RND1:ICON_RND0));
        DrawIcon(aContext->hdc, object->x-patch->offsX + 130, object->y-patch->offsY + ioYSpacing*6,    (HICON)((object->mode==4)?ICON_PLS1:ICON_PLS0));
        DrawIcon(aContext->hdc, object->x-patch->offsX + 130, object->y-patch->offsY + ioYSpacing*6+20, (HICON)((object->mode==3)?ICON_SPS1:ICON_SPS0));
    break;
  case OBJ_ADD:
    guiSelectTitleFont(aContext);
    guiSelectPen1Color(aContext, defcolor_patchShadow);
    SetTextAlign(aContext->hdc, TA_BOTTOM | TA_CENTER);
    guiDrawText(aContext, 40+object->x-patch->offsX, ioYSpacing*3+object->y-patch->offsY, "+", 1);
    break;
  case OBJ_MUL:
    guiSelectTitleFont(aContext);
    guiSelectPen1Color(aContext, defcolor_patchShadow);
    SetTextAlign(aContext->hdc, TA_BOTTOM | TA_CENTER);
    guiDrawText(aContext, 40+object->x-patch->offsX, ioYSpacing*3+object->y-patch->offsY, "X", 1);
    break;
  case OBJ_OR:
    guiSelectTitleFont(aContext);
    guiSelectPen1Color(aContext, defcolor_patchShadow);
    SetTextAlign(aContext->hdc, TA_BOTTOM | TA_CENTER);
    guiDrawText(aContext, 40+object->x-patch->offsX, ioYSpacing*3+object->y-patch->offsY, ">0", 2);
    break;
  case OBJ_AND:
    guiSelectTitleFont(aContext);
    guiSelectPen1Color(aContext, defcolor_patchShadow);
    SetTextAlign(aContext->hdc, TA_BOTTOM | TA_CENTER);
    guiDrawText(aContext, 40+object->x-patch->offsX, ioYSpacing*3+object->y-patch->offsY, "&", 1);
    break;
  case OBJ_DRM4:
  case OBJ_DRM8:
    if (0 == redrawDelay) {
      int col;
      int row;

      for(row=0; row<((object->objectType==OBJ_DRM4)?4:8); row++) {
        for(col=0; col<16; col++) {
          guiSelectFillColor(aContext, (((unsigned char *)projects[currentProject].dsp[object->dsp].buffer)[row * 16 + col])?defcolor_patchShadow:defcolor_patchModules);
          guiDrawRect(aContext,
            (object->x-patch->offsX)+knobXSpacing+col*16,
            (object->y-patch->offsY)+ioYSpacing*2*(row+1),
            (object->x-patch->offsX)+knobXSpacing+col*16+10,
            (object->y-patch->offsY)+ioYSpacing*2*(row+1)+12);
        }
      }
    }
    break;
  case OBJ_TEXT:
    if (object->name) {
      SetTextAlign(aContext->hdc, TA_TOP | TA_LEFT);
      guiSelectPotFont(aContext);
      guiSelectPen1Color(aContext, -1);
      guiSelectFillColor(aContext, object->mainColor);
      guiDrawRect(aContext, object->x-patch->offsX, object->y-patch->offsY, object->x+object->xs-patch->offsX, object->y+object->ys-patch->offsY);
      guiSelectPen1Color(aContext, object->borderColor);
      guiTextTransparent(aContext, false);
      guiDrawText(aContext, object->x-patch->offsX, object->y-patch->offsY, object->x+object->xs-patch->offsX, object->y+object->ys-patch->offsY, object->name, strlen(object->name));
//      TextOut(aContext->hdc, object->x-patch->offsX, object->y-patch->offsY, object->name, strlen(object->name));
    }
    if (showResizeMarkers) {
      drawResizeMarkers(aContext, object->x-patch->offsX, object->y-patch->offsY, object->x+object->xs-patch->offsX, object->y+object->ys-patch->offsY, object->selected);
    }
    break;
  case OBJ_BOX:
    if (showResizeMarkers) {
      drawResizeMarkers(aContext, object->x-patch->offsX, object->y-patch->offsY, object->x+object->xs-patch->offsX, object->y+object->ys-patch->offsY, object->selected);
    }
    break;
  case OBJ_ROUNDBOX:
    if (showResizeMarkers) {
      drawResizeMarkers(aContext, object->x-patch->offsX, object->y-patch->offsY, object->x+object->xs-patch->offsX, object->y+object->ys-patch->offsY, object->selected);
    }
    break;
  case OBJ_ELLIPSE:
    if (showResizeMarkers) {
      drawResizeMarkers(aContext, object->x-patch->offsX, object->y-patch->offsY, object->x+object->xs-patch->offsX, object->y+object->ys-patch->offsY, object->selected);
    }
    break;
  default:
    break;
  }
}

static void setCableType(int newCableType) {
  currentCableType=newCableType;
  mainWindowRefresh=true;
  configWindowRefresh=true;
}

static void drawCable(ContextPtr aContext, int x1, int y1, int x2, int y2, int color) {
  switch (currentCableType) {
  case 1: // Colored cables
    guiSelectPen3Color(aContext, cablePenColors[color]);
    MoveToEx(aContext->hdc, x1, y1, NULL);
    LineTo(aContext->hdc, x2, y2);
    guiSelectPen1Color(aContext, cableFillColors[color]);
    MoveToEx(aContext->hdc, x1, y1, NULL);
    LineTo(aContext->hdc, x2, y2);
    break;
  case 2: { // Hanging colored cables
      POINT curve[4];
      curve[0].x = x1;
      curve[0].y = y1;
      curve[3].x = x2;
      curve[3].y = y2;
      curve[1].x = (curve[0].x + curve[3].x) / 2;
      curve[1].y = max(curve[0].y, curve[3].y) + (abs(curve[0].x - curve[3].x) / 8) + (abs(curve[0].y - curve[3].y) / 16);
      curve[2].x = curve[1].x;
      curve[2].y = curve[1].y;

      guiSelectPen3Color(aContext, cablePenColors[color]);
      PolyBezier(aContext->hdc, curve, 4);
      guiSelectPen1Color(aContext, cableFillColors[color]);
      PolyBezier(aContext->hdc, curve, 4);
    }
    break;
  default: // Black cables
    guiSelectPen1Color(aContext, 0);
    MoveToEx(aContext->hdc, x1, y1, NULL);
    LineTo(aContext->hdc, x2, y2);
    break;
  }
}

static void drawCables(ContextPtr aContext, SynFactoryPatchPtr patch, SynFactoryObjectPtr object) {
  int i;

  for(i=0;i<maxObjectIo;i++) {
    if (object->connections[i].fromObject) {
      drawCable(aContext, -patch->offsX+patch->objects[object->connections[i].fromObject-1].x+patch->objects[object->connections[i].fromObject-1].xs-ioXSize/2, -patch->offsY+patch->objects[object->connections[i].fromObject-1].y+(object->connections[i].fromOutput+1)*ioYSpacing+ioYSize/2, -patch->offsX+object->x+ioXSize/2, -patch->offsY+object->y+(i+2)*ioYSpacing+ioYSize/2, object->connections[i].cableColor);
    }
  }
}

static void drawXorCable(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int i;
  int oldROP2=GetROP2(aContext->hdc);

  SetROP2(aContext->hdc, R2_XORPEN);
  for(i=0;i<patch->maxObjects;i++) {
    if (patch->objects[i].currentOutput) {
      if (XorFlag) {
        drawCable(aContext, -patch->offsX+patch->objects[i].x+patch->objects[i].xs-ioXSize/2, -patch->offsY+patch->objects[i].y+(patch->objects[i].currentOutput+1)*ioYSpacing+ioYSize/2, oldMouseClientX, oldMouseClientY, 8);
        storeMouse(aContext);
      }
      drawCable(aContext, -patch->offsX+patch->objects[i].x+patch->objects[i].xs-ioXSize/2, -patch->offsY+patch->objects[i].y+(patch->objects[i].currentOutput+1)*ioYSpacing+ioYSize/2, oldMouseClientX, oldMouseClientY, 8);
      XorFlag=true;
    }
  }
  SetROP2(aContext->hdc, oldROP2);
}

static void drawXorRect(ContextPtr aContext, SynFactoryPatchPtr patch) {
  int oldROP2=GetROP2(aContext->hdc);

  SetROP2(aContext->hdc, R2_XORPEN);
  guiSelectPen1Color(aContext, 0x888888);
  guiSelectFillColor(aContext, -1);
  if (XorFlag) {
    guiDrawRect(aContext,
      min(startX-patch->offsX,oldMouseClientX),
      min(startY-patch->offsY,oldMouseClientY),
      max(startX-patch->offsX,oldMouseClientX),
      max(startY-patch->offsY,oldMouseClientY));
  }
  storeMouse(aContext);
  guiDrawRect(aContext,
    min(startX-patch->offsX,oldMouseClientX),
    min(startY-patch->offsY,oldMouseClientY),
    max(startX-patch->offsX,oldMouseClientX),
    max(startY-patch->offsY,oldMouseClientY));
  XorFlag=true;
  SetROP2(aContext->hdc, oldROP2);
}

static void processSynFactory(ContextPtr aContext, SynFactoryPatchPtr patch) {
  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      guiSelectFillColor(aContext, defcolor_patchBackground);
      guiDraw3DRect(aContext, &(aContext->guiClientRect), 0);

      SelectObject(aContext->hdc, Fonts[0x18]);
      const char * const satisfactory="SynFactory";
      SetBkMode(aContext->hdc, TRANSPARENT);
      guiSelectPen1Color(aContext, defcolor_patchModules);
      TextOut(aContext->hdc, 16, 16, satisfactory,strlen(satisfactory));
//      guiSelectPen1Color(aContext, 0);
//      TextOut(aContext->hdc, 14, 14, satisfactory,strlen(satisfactory));


      int i;

      if (mouseMode==MOUSE_DRAGSCREENZOOM) {
        drawObjectsZoomed(aContext, patch);
      } else {
        for(i=0;i<patch->maxObjects;i++) {
          drawObject(aContext, patch, &(patch->objects[i]));
        }
        for(i=0;i<patch->maxObjects;i++) {
          drawCables(aContext, patch, &(patch->objects[i]));
        }
      }
      XorFlag=false;
      if (mouseMode==MOUSE_DRAGRECT || mouseMode==MOUSE_DRAGRECTANDSCREEN) {
        drawXorRect(aContext, patch);
      }
    } break;
  case GUIEVENT_MENU:
    if (aContext->id>1024) {
      // !!!  should use patch but is pointer so we take currentPatch (need to analyse how to solve)
      int n=addSynFactoryObject(currentPatch, (ObjectType)(aContext->id-1024), false);
      if (n>=0) {
        patch->objects[n].x=oldMouseClientX+patch->offsX;
        patch->objects[n].y=oldMouseClientY+patch->offsY;
      }
    } else {
      switch(aContext->id) {
      case MENU_SELECTALL: {
          int i;

          for(i=0;i<patch->maxObjects;i++) {
            selectProjectBrowserItem(currentProject, patch, i, -1, (i!=0) || ctrlPressed());
          }
        } break;
      case MENU_MODULEHELP: {
          int i;
          for(i=0;i<patch->maxObjects;i++) {
            if (patch->objects[i].selected) {
              displayHelp("Context help", objectHelpText(patch->objects[i].objectType));
              break;
            }
          }
        } break;
      default:
        break;
      }
    }
    break;
  case GUIEVENT_LOSTFOCUS:
    switch(mouseMode) {
    case MOUSE_DRAGOBJECT:
      finishDragObjects(aContext, patch);
      break;
    case MOUSE_DRAGCABLE:
    case MOUSE_DRAGCABLEANDSCREEN:
      finishDragCable(aContext, patch);
      break;
    case MOUSE_DRAGSCREEN:
    case MOUSE_DRAGSCREENZOOM:
    case MOUSE_DRAGRECT:
    case MOUSE_DRAGRECTANDSCREEN:
      ReleaseCapture();
      mouseMode=MOUSE_NONE;
      mainWindowRefresh=true;
      redrawDelay=0;
      break;
    case MOUSE_KNOB:
    case MOUSE_KNOBSLOW:
      SetCursorPos(oldMouseX, oldMouseY);
      guiShowMouse(aContext);
      ReleaseCapture();
      mouseMode=MOUSE_NONE;
      mainWindowRefresh=true;
      break;
    case MOUSE_RESIZE_TOP_LEFT:
    case MOUSE_RESIZE_TOP_RIGHT:
    case MOUSE_RESIZE_BOTTOM_LEFT:
    case MOUSE_RESIZE_BOTTOM_RIGHT:
      resizeObject(aContext, patch);
      ReleaseCapture();
      mouseMode=MOUSE_NONE;
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_MOUSEBUTTON1DOWN:
    switch(mouseMode) {
    case MOUSE_NONE:
      synfactoryLeftClick(aContext, patch);
      break;
    case MOUSE_DRAGSCREEN:
      mouseMode=MOUSE_DRAGSCREENZOOM;
      mainWindowRefresh=true;
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_MOUSEBUTTON1UP:
    switch(mouseMode) {
    case MOUSE_DRAGSCREENZOOM:
      mouseMode=MOUSE_DRAGSCREEN;
      mainWindowRefresh=true;
      break;
    case MOUSE_DRAGOBJECT:
      finishDragObjects(aContext, patch);
      break;
    case MOUSE_DRAGCABLE:
      finishDragCable(aContext, patch);
      break;
    case MOUSE_DRAGOBJECTANDSCREEN:
      mouseMode=MOUSE_DRAGSCREEN;
      mainWindowRefresh=true;
      break;
    case MOUSE_DRAGRECT:
      ReleaseCapture();
      selectObjectsInArea(patch, min(startX,oldMouseClientX+patch->offsX), min(startY,oldMouseClientY+patch->offsY), max(startX,oldMouseClientX+patch->offsX), max(startY,oldMouseClientY+patch->offsY));
      mouseMode=MOUSE_NONE;
      mainWindowRefresh=true;
      break;
    case MOUSE_DRAGRECTANDSCREEN:
      selectObjectsInArea(patch, min(startX,oldMouseClientX+patch->offsX), min(startY,oldMouseClientY+patch->offsY), max(startX,oldMouseClientX+patch->offsX), max(startY,oldMouseClientY+patch->offsY));
      mouseMode=MOUSE_DRAGSCREEN;
      mainWindowRefresh=true;
      break;
    case MOUSE_KNOB:
    case MOUSE_KNOBSLOW:
      SetCursorPos(oldMouseX, oldMouseY);
      guiShowMouse(aContext);
      ReleaseCapture();
      mouseMode=MOUSE_NONE;
      mainWindowRefresh=true;
      break;
    case MOUSE_RESIZE_TOP_LEFT:
    case MOUSE_RESIZE_TOP_RIGHT:
    case MOUSE_RESIZE_BOTTOM_LEFT:
    case MOUSE_RESIZE_BOTTOM_RIGHT:
      resizeObject(aContext, patch);
      ReleaseCapture();
      mouseMode=MOUSE_NONE;
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_MOUSEBUTTON2DOWN:
    mouseDownTicks = 0;
    switch(mouseMode) {
    case MOUSE_NONE:
      storeMouse(aContext);
      SetCapture(aContext->currentWindow);
      mouseMode = MOUSE_DRAGSCREEN;
      break;
    case MOUSE_DRAGOBJECT:
      storeMouse(aContext);
      mouseMode=MOUSE_DRAGOBJECTANDSCREEN;
      break;
    case MOUSE_DRAGCABLE:
      storeMouse(aContext);
      mouseMode=MOUSE_DRAGCABLEANDSCREEN;
      break;
    case MOUSE_DRAGRECT:
      storeMouse(aContext);
      mouseMode=MOUSE_DRAGRECTANDSCREEN;
      break;
    case MOUSE_KNOB:
      mouseMode=MOUSE_KNOBSLOW;
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_MOUSEBUTTON2UP:
    switch(mouseMode) {
    case MOUSE_DRAGSCREEN:
    case MOUSE_DRAGSCREENZOOM:
      ReleaseCapture();
      mouseMode=MOUSE_NONE;
      mainWindowRefresh=true;
      redrawDelay=0;
      if (mouseDownTicks<3) {
        TrackPopupMenu(synFactoryMenu[currentLanguage], TPM_RIGHTBUTTON, aContext->mouseX, aContext->mouseY, 0, mainWindow, NULL);
      }
      break;
    case MOUSE_DRAGOBJECTANDSCREEN:
      mouseMode=MOUSE_DRAGOBJECT;
      mainWindowRefresh=true;
      redrawDelay=0;
      break;
    case MOUSE_DRAGCABLEANDSCREEN:
      mouseMode=MOUSE_DRAGCABLE;
      mainWindowRefresh=true;
      redrawDelay=0;
      break;
    case MOUSE_DRAGRECTANDSCREEN:
      mouseMode=MOUSE_DRAGRECT;
      mainWindowRefresh=true;
      redrawDelay=0;
      break;
    case MOUSE_KNOBSLOW:
      mouseMode=MOUSE_KNOB;
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_MOUSEBUTTON1DCLK:
    switch(mouseMode) {
    case MOUSE_NONE:
      processPatchDoubleClick(aContext, patch);
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_MOUSEBUTTON2DCLK:
    switch(mouseMode) {
    case MOUSE_KNOB:
    case MOUSE_KNOBSLOW:
      projects[currentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob=0;
      mainWindowRefresh=true;
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_TIMERTICK:
    switch(mouseMode) {
    case MOUSE_DRAGOBJECT:
      updateDragObjects(aContext, patch);
      break;
    case MOUSE_DRAGCABLE:
      drawXorCable(aContext, patch);
      break;
    case MOUSE_DRAGSCREEN:
    case MOUSE_DRAGSCREENZOOM:
    case MOUSE_DRAGOBJECTANDSCREEN:
      updateDragScreen(aContext, patch);
      break;
    case MOUSE_DRAGCABLEANDSCREEN:
      updateDragScreen(aContext, patch);
      drawXorCable(aContext, patch);
      break;
    case MOUSE_DRAGRECT:
      drawXorRect(aContext, patch);
      break;
    case MOUSE_DRAGRECTANDSCREEN:
      updateDragScreen(aContext, patch);
      drawXorRect(aContext, patch);
      break;
    case MOUSE_KNOB:
    case MOUSE_KNOBSLOW:
      updateKnob(aContext, patch);
      break;
    case MOUSE_RESIZE_TOP_LEFT:
    case MOUSE_RESIZE_TOP_RIGHT:
    case MOUSE_RESIZE_BOTTOM_LEFT:
    case MOUSE_RESIZE_BOTTOM_RIGHT:
      resizeObject(aContext, patch);
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_KEYDOWN: {
      switch(aContext->id) {
      case VK_F1: {
          int i;
          for(i=0;i<patch->maxObjects;i++) {
            if (patch->objects[i].selected) {
              displayHelp("Context help", objectHelpText(patch->objects[i].objectType));
              break;
            }
          }
        } break;
      case VK_RETURN:
        if (MOUSE_NONE == mouseMode) {
          storeMouse(aContext);
          TrackPopupMenu(synFactoryMenu[currentLanguage], TPM_RIGHTBUTTON, aContext->mouseX, aContext->mouseY, 0, mainWindow, NULL);
        }
        break;
      case VK_DELETE:
        deleteSynFactoryObjects(currentProject, patch, false);
        break;
      default:
        break;
      }
    } break;
  case GUIEVENT_CHAR: {
      MouseMode tempMouseMode=MOUSE_NONE;
      int n=-1;

      int obj=findObject(patch, aContext->mouseClientX + patch->offsX, aContext->mouseClientY + patch->offsY, &tempMouseMode);

      if (MOUSE_NONE==tempMouseMode && !aContext->keyRepeatFlag) {
        switch(aContext->id) {
$objectKeys
        default:
          break;
        }
        if (n>=0) {
          patch->objects[n].x=aContext->mouseClientX+patch->offsX;
          patch->objects[n].y=aContext->mouseClientY+patch->offsY;
        }
      }
      if (MOUSE_EDIT_LABEL==tempMouseMode && (aContext->id==8 || (aContext->id>=32 && aContext->id<127))) {
        updateModuleLabel(patch, obj, aContext->id);
      }
      if ((aContext->id=='[' || aContext->id==']' || aContext->id=='{' || aContext->id=='}' || 
           aContext->id=='-' || aContext->id==8 || (aContext->id>='0' && aContext->id<='9')) &&
          (MOUSE_NONE==mouseMode && MOUSE_KNOB==tempMouseMode)) {
        int temp = projects[currentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob;
        if (aContext->id=='-') {
          temp=-temp;
        }
        if (aContext->id==8) {
          temp/=10;
        }
        if (aContext->id>='0' && aContext->id<='9') {
          temp*=10;
          if (temp<0) {
            temp-=aContext->id-'0';
          } else {
            temp+=aContext->id-'0';
          }
        }
        if (aContext->id=='[') {
          temp--;
        }
        if (aContext->id==']') {
          temp++;
        }
        if (aContext->id=='{') {
          temp-=10;
        }
        if (aContext->id=='}') {
          temp+=10;
        }

        if (temp > 32767) temp = 32767;
        if (temp < -32768) temp = -32768;
        projects[currentProject].dsp[patch->objects[editObject].dsp].io[knobIndex].knob = temp;          
        mainWindowRefresh=true;
      }
    } break;
  default:
    break;
  }
}

EOF
  rv('HANDLE ICON_SIN0, ICON_SIN1, ICON_TRI0, ICON_TRI1, ICON_SAW0, ICON_SAW1;');
  rv('HANDLE ICON_SPS0, ICON_SPS1, ICON_PLS0, ICON_PLS1, ICON_RND0, ICON_RND1;');

  rv('HMENU synFactoryMenu[noofLanguages];');
  push(@initFunctions, "for(i=0;languageCodes[i];i++) { synFactoryMenu[i] = makeMenu(synFactoryMenuDef, mainMenuDef, i, true); }");
  push(@initFunctions, "ICON_SIN0=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOSIN0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_SIN1=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOSIN1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_TRI0=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOTRI0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_TRI1=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOTRI1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_SAW0=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOSAW0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_SAW1=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOSAW1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_SPS0=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOSPS0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_SPS1=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOSPS1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_PLS0=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOPLS0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_PLS1=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCOPLS1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_RND0=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCORND0), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@initFunctions, "ICON_RND1=LoadImage(myInstance, MAKEINTRESOURCE(IDI_VCORND1), IMAGE_ICON, 32, 32, LR_DEFAULTCOLOR);");
  push(@termFunctions, "for(i=0;languageCodes[i];i++) { DestroyMenu(synFactoryMenu[i]); }");

$rc .= <<EOF;
static void patchConfigWindowHandler(ContextPtr aContext) {
  static int maxy=0;
  static int maxysize=0;

  switch(aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      int y=-guiGetVScrollPos(aContext->currentWindow);

      guiClearAreaList(aContext);

      guiSelectFillColor(aContext, 0xFFFFFF);
      guiDraw3DRect(aContext, &(aContext->guiClientRect), 0);

      if (currentProject>=0 && currentPatch>=0) {
        guiSelectPotFont(aContext);
        guiTextTransparent(aContext, true);

        configWindowHeader(aContext,  4, &y, STRING_PATCH_MIDI_PORT);
        int j;
        guiSelectPen1Color(aContext, 0);
        for(j=0;j<27;j++) {
          if (j==projects[currentProject].patches[currentPatch].midiPort) {
            TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiSelectedPortNames+j*5, 5);
          } else {
            TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiUnSelectedPortNames+j*5, 5);
          }
          configWindowAddHitArea(aContext, 30+(j%9)*25, y+(j/9)*14, 30+25+(j%9)*25);
        }
        y+=42;

        configWindowHeader(aContext,  4, &y, STRING_PATCH_MIDI_CHANNEL);
        guiSelectPen1Color(aContext, 0);
        for(j=0;j<17;j++) {
          if (j==projects[currentProject].patches[currentPatch].midiChannel) {
            TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiSelectedChannelNames+j*5, 5);
          } else {
            TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiUnSelectedChannelNames+j*5, 5);
          }
          configWindowAddHitArea(aContext, 30+(j%9)*25, y+(j/9)*14, 30+25+(j%9)*25);
        }
        y+=42;

  //    guiSelectFillColor(aContext, 0xFFFFFF);
  //    guiSelectPen1Color(aContext, -1);
  //    guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom);
        if (y+guiGetVScrollPos(aContext->currentWindow)!=maxy || aContext->guiClientRect.bottom!=maxysize) {
          maxy=y+guiGetVScrollPos(aContext->currentWindow);
          maxysize=aContext->guiClientRect.bottom;
          guiVScrollRange(aContext->currentWindow, 0, maxy, maxysize);
        }
      }

    } break;
  case GUIEVENT_MOUSEBUTTON1DOWN: {
      int index=0;
      int i;

      for(i=0;i<27;i++) {
        index++;
        if (index==aContext->id) {
          projects[currentProject].patches[currentPatch].midiPort=i;
          patchConfigWindowRefresh=true;
        }
      }
      for(i=0;i<17;i++) {
        index++;
        if (index==aContext->id) {
          projects[currentProject].patches[currentPatch].midiChannel=i;
          patchConfigWindowRefresh=true;
        }
      }
    } break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(aContext->currentWindow);
    break;
  default:
    break;
  }
}
EOF

  genWindow('patchConfigWindow', 'PatchConfigWindow', '"Patch Settings"', 'patchConfigWindowHandler', undef, undef, 1, undef);

  push(@stfSave, 'storeStfBegin(loadInfo, "PatchCableType"); storeStfInteger(loadInfo, currentCableType); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp(loadInfo->headerTitle, "PatchCableType")==0) { int c; readStfInteger(loadInfo, &c); setCableType(c); }');
}


##########################################################################
#
# Bitmap editor
#
##########################################################################
sub genBitmapEditor {
}


##########################################################################
#
# Expand String
#
##########################################################################
sub genExpandString {
  $rc .= <<EOF;
static char *expandedString;
static int maxExpandedString;
static const char *expandString(const char *name, const char *data, bool modifiedFlag) {
  int len=strlen(name)+1;

  if (data) len+=strlen(data);
  if (modifiedFlag) len+=3;

  if (len>maxExpandedString) {
    maxExpandedString=len;
    expandedString=(char*)realloc(expandedString, maxExpandedString);
  }
  strcpy(expandedString, name);
  if (modifiedFlag) strcat(expandedString, " * ");
  if (data) strcat(expandedString, data);
  return expandedString;
}
EOF
  push(@termFunctions, "if (expandedString) { free(expandedString); expandedString=NULL; maxExpandedString=0; }\n");

  $rc .= <<EOF;
static const char *expandString(const char *string1, const char *string2, const char *string3, const char *string4, const char *string5) {
  int len=0;

  if (string1) len+=strlen(string1);
  if (string2) len+=strlen(string2);
  if (string3) len+=strlen(string3);
  if (string4) len+=strlen(string4);
  if (string5) len+=strlen(string5);
  len++;

  if (len>maxExpandedString) {
    free(expandedString);
    maxExpandedString=len;
    expandedString=(char*)malloc(maxExpandedString);
  }

  if (string1) strcpy(expandedString, string1);
  if (string2) strcat(expandedString, string2);
  if (string3) strcat(expandedString, string3);
  if (string4) strcat(expandedString, string4);
  if (string5) strcat(expandedString, string5);
  return expandedString;
}
EOF
}


##########################################################################
#
# Project Browser window
#
##########################################################################
sub genProjectBrowser {

  my @projectStructDef=(
    'bool selected;',
    'bool modifiedFlag;',
    'bool unfoldedProjectView;',
    'char *projectName;',
    'char *projectPath;',
    'DspObject *dsp;',
    'SynFactoryPatchPtr patches;',
    'TrackPtr tracks;',
    'SamplePtr samples;',
    'int maxPatches;',
    'int maxTracks;',
    'int maxSamples;',
    'int patchCount;',
  );
#    'int trackCount;',
  genEnumStruct("struct", "Project", \@projectStructDef);

  rv('const char * const newProjectName="new project";');
  rv('ProjectPtr projects;');
  rv('int maxProjects;');
  rv('int currentPlayingProject=-1;');
  rv('int currentProject=-1;');
  rv('int currentPatch=-1;');
  rv('int projectBrowserPos;');

  $rs .= <<EOF;
static void selectProjectBrowserItem(int project, SynFactoryPatchPtr patch, int object, int track, bool ctrlPressed) {
  int i,j,k;
  int p=0;

//  projectBrowserPos=-1;
  currentPatch=-1;
  for(i=0;i<maxProjects;i++) {
    if (project==i) {
      currentProject=i;
    }
    if (project==i && patch==NULL && track==-1) {
      projects[i].selected = true;
      projectBrowserPos = p;
      if (projects[i].maxPatches>0) {
        currentPatch=0;
      }
    } else {
      if (!ctrlPressed) {
        projects[i].selected=false;         
      }
    }
    p++;

    // Force selected item visible
    if (project==i && (patch!=NULL || track>=0)) {
      projects[i].unfoldedProjectView = true;
    }

    // Patches
    for(j=0;j<projects[i].maxPatches;j++) {
      if (project==i && patch==projects[i].patches+j) {
        currentPatch=j;
      }
      if (project==i && patch==projects[i].patches+j && object<0) {
        projects[i].patches[j].selected=true;
        if (projects[i].unfoldedProjectView) {
          projectBrowserPos = p;
        }
      } else {
        if (!ctrlPressed) {
          projects[i].patches[j].selected=false;
        }
      }
      if (projects[i].unfoldedProjectView) {
        p++;
      }
      for(k=0;k<projects[i].patches[j].maxObjects;k++) {
        if (project==i && patch==projects[i].patches+j && object==k) {
          projects[i].patches[j].objects[k].selected=true;
        } else {
          if (!ctrlPressed) {
            projects[i].patches[j].objects[k].selected=false;
          }
        }
      }
    }

    // Tracks
    for(j=0;j<projects[i].maxTracks;j++) {
      if (projects[i].tracks[j].inUse) {
        if (project==i && track==j) {
          projects[i].tracks[j].selected=true;
          if (projects[i].unfoldedProjectView) {
            projectBrowserPos = p;
          }
        } else {
          if (!ctrlPressed) {
            projects[i].tracks[j].selected=false;
          }
        }
        if (projects[i].unfoldedProjectView) {
          p++;
        }
      }
    }
  }

//  projects[project].tracks[track].selected=true;
  projectBrowserWindowRefresh=true;
  mainWindowRefresh=true;
  patchConfigWindowRefresh=true;
}

static int createProject(const char *projectName) {
  int n=-1;

  logprintf("createProject\\n");

  n=maxProjects;
  maxProjects++;
  lockMutex(audioMutex);
  lockMutex(midiMutex);
  projects=(ProjectPtr)realloc(projects, maxProjects*sizeof(Project));
  unlockMutex(midiMutex);
  unlockMutex(audioMutex);
  memset(&(projects[n]), 0, sizeof(Project));
  projects[n].dsp=(DspObject*)calloc(maxDspObjects, sizeof(DspObject));
  if (projectName) {
    projects[n].projectName=strdup(projectName);
  }
  currentPatch=-1;
  selectProjectBrowserItem(n, NULL, -1, -1, false);
  transportWindowRefresh=true;

  logprintf("createProject returns %d\\n", n);
  return n;
}
EOF

  $rc .= <<EOF;
static void deleteProject(int project) {
  int i;

  if (currentPlayingProject == project) {
    setPlayMode(PLAYMODE_STOP);
  } 

  for(i=0;i<projects[project].maxPatches;i++) {
    deleteSynFactoryObjects(project, &(projects[project].patches[i]), true);
  }

  lockMutex(audioMutex);
  lockMutex(midiMutex);
  free(projects[project].dsp);
  if (projects[project].patches) {
    free(projects[project].patches);
  }
  if (projects[project].projectName) {
    free(projects[project].projectName);
  }
  maxProjects--;
  for(i=project;i<maxProjects;i++) {
    projects[i]=projects[i+1];
  }
  projects=(ProjectPtr)realloc(projects, maxProjects*sizeof(Project));
  if (currentPlayingProject >= project) {
    currentPlayingProject--;
  }
  if (currentProject>=maxProjects) {
    currentProject=maxProjects-1;
  }
  if (currentProject>=0) {
    currentPatch=projects[currentProject].maxPatches-1;
  } else {
    currentPatch=-1;
  }
  projectBrowserWindowRefresh=true;
  patchConfigWindowRefresh=true;
  mainWindowRefresh=true;
  unlockMutex(midiMutex);
  unlockMutex(audioMutex);
}

static void deleteSelected(void) {
  int i=0;

  while(i<maxProjects) {
    if (projects[i].selected) {
      deleteProject(i);
    } else {
      int j=0;
      while (j<projects[i].maxPatches) {
        deleteSynFactoryObjects(i, &(projects[i].patches[j]), projects[i].patches[j].selected);
        if (projects[i].patches[j].selected) {
          for(int k=j;k<projects[i].maxPatches-1;k++) {
            projects[i].patches[k]=projects[i].patches[k+1];
          }
          projects[i].maxPatches--;
          projects[i].patches=(SynFactoryPatchPtr)realloc(projects[i].patches, projects[i].maxPatches*sizeof(SynFactoryPatch));
          if (currentProject==i && currentPatch>=projects[i].maxPatches) {
            currentPatch=projects[i].maxPatches-1;
          }

          projectBrowserWindowRefresh=true;
          mainWindowRefresh=true;
          patchConfigWindowRefresh=true;
        } else {
          j++;
        }
      }
      i++;
    }
  }
}

static void projectBrowserAddHitArea(ContextPtr aContext, int y) {
  guiAddArea(aContext, aContext->guiClientRect.left, y, aContext->guiClientRect.right, y+14);
}

static void drawExpandBox(ContextPtr aContext, int x, int y, bool expanded) {
  guiSelectFillColor(aContext, -1);
  guiDrawRect(aContext, x, y, x+14, y+12);
  MoveToEx(aContext->hdc, x+4, y+6, NULL);
  LineTo(aContext->hdc, x+11, y+6);
  if (!expanded) {
    MoveToEx(aContext->hdc, x+7, y+3, NULL);
    LineTo(aContext->hdc, x+7, y+10);            
  }
}

static int projectBrowserBackground(bool cursor, bool selected) {
  if (cursor) {
    if (selected) {
      return 0x00AAAA;
    } else {
      return 0xAAAAAA;
    }
  } else {
    if (selected) {
      return 0x00FFFF;
    } else {
      return 0xFFFFFF;
    }
  }
}

static void projectBrowserHandler(ContextPtr aContext) {
  static bool recalcCursor=false;
  static int maxy=0;
  static int maxysize=0;

  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      int textColor;
      int color;
      int y=-guiGetVScrollPos(aContext->currentWindow);
	  int ycursor=0;
      int i;
      int j;

      guiClearAreaList(aContext);

      guiSelectFillColor(aContext, 0xFFFFFF);
      guiSelectPen1Color(aContext, -1);
      guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom);

      guiSelectPotFont(aContext);
      guiTextTransparent(aContext, true);
      guiSelectPen1Color(aContext, 0);

      for(i=0;i<maxProjects;i++) {
        if (currentProject==i) {
          textColor=0;
        } else {
          textColor=0x888888;
        }

        guiSelectPen1Color(aContext, textColor);
        drawExpandBox(aContext, 2, y, projects[i].unfoldedProjectView);

        color=projectBrowserBackground(projectBrowserPos==guiGetAreaListLen(aContext), projects[i].selected);
		if (projectBrowserPos==guiGetAreaListLen(aContext)) ycursor=y;

        guiSelectPen1Color(aContext, -1);
        guiSelectFillColor(aContext, color);
        guiDrawRect(aContext, 18, y, aContext->guiClientRect.right, y+14);
        expandString(projects[i].projectName, (projects[i].modifiedFlag)?" *":NULL, " (", projects[i].projectPath, ")");
        guiSelectPen1Color(aContext, textColor);
        TextOut(aContext->hdc, 20, y, expandedString, strlen(expandedString));
        projectBrowserAddHitArea(aContext, y);
        y+=14;

//          guiSelectFillColor(aContext, 15);
        if (projects[i].unfoldedProjectView) {
//          expandString("path: ", projects[i].projectPath, 0);
//          guiSelectFillColor(aContext, 0xFFFFFF);
//          guiSelectPen1Color(aContext, 0x000000);
//          TextOut(aContext->hdc, 20, y, expandedString, strlen(expandedString));
//          y+=14;

          if (projects[i].maxPatches>0) {
            guiSelectFillColor(aContext, 0xFFFFFF);
            TextOut(aContext->hdc, 40, y, "Patches", 7);
            y+=14;
            for(j=0;j<projects[i].maxPatches;j++) {
              int k;

              color=projectBrowserBackground(projectBrowserPos==guiGetAreaListLen(aContext), projects[i].patches[j].selected);
              if (projectBrowserPos==guiGetAreaListLen(aContext)) ycursor=y;

//              guiSelectFillColor(aContext, 0xFFFFFF);
              if (currentProject==i && currentPatch==j) {
                TextOut(aContext->hdc, 40, y, "==>", 3);
                
              }

              drawExpandBox(aContext, 62, y, projects[i].patches[j].unfoldedPatchView);

              guiSelectPen1Color(aContext, -1);
              guiSelectFillColor(aContext, color);
              guiDrawRect(aContext, 80, y, aContext->guiClientRect.right, y+14);
              expandString(projects[i].patches[j].patchName, NULL, false);
              guiSelectPen1Color(aContext, textColor);
              TextOut(aContext->hdc, 80, y, expandedString, strlen(expandedString));
              projectBrowserAddHitArea(aContext, y);
              y+=14;
              if (projects[i].patches[j].unfoldedPatchView) {
                guiSelectFillColor(aContext, 0xFFFFFF);
                for(k=0;k<projects[i].patches[j].maxObjects;k++) {
                  color=projectBrowserBackground(projectBrowserPos==guiGetAreaListLen(aContext), projects[i].patches[j].objects[k].selected);
                  if (projectBrowserPos==guiGetAreaListLen(aContext)) ycursor=y;
                  guiSelectFillColor(aContext, color);
                  if (objectTitles[projects[i].patches[j].objects[k].objectType]) {
                    guiSelectPen1Color(aContext, -1);
                    guiSelectFillColor(aContext, color);
                    guiDrawRect(aContext, 100, y, aContext->guiClientRect.right, y+14);
                    expandString(objectTitles[projects[i].patches[j].objects[k].objectType], NULL, false);
                    guiSelectPen1Color(aContext, textColor);
                    TextOut(aContext->hdc, 100, y, expandedString, strlen(expandedString));
//                    projectBrowserAddHitArea(aContext, y);
                    y+=14;
                  }
                }
              }
            }
          }
          if (projects[i].maxTracks>0) {
            guiSelectFillColor(aContext, 15);
            TextOut(aContext->hdc, 40, y, "Tracks", 6);
            y+=14;
            for(j=0;j<projects[i].maxTracks;j++) {
              if (projects[i].tracks[j].inUse) {
                if (projects[i].tracks[j].selected) {
                  color=0x00FFFF;
                } else {
                  color=0xFFFFFF;
                }
                if (projectBrowserPos == guiGetAreaListLen(aContext)) {
                  color-=8;
                }
                guiSelectFillColor(aContext, color);
                expandString(projects[i].tracks[j].trackName, NULL, false);
                TextOut(aContext->hdc, 60, y, expandedString, strlen(expandedString));
                projectBrowserAddHitArea(aContext, y);
                y+=14;
              }
            }
          }

          guiSelectFillColor(aContext, 0xFFFFFF);
          TextOut(aContext->hdc, 40, y, "Samples", 7);
          y+=14;
        }
      }


      if (y+guiGetVScrollPos(aContext->currentWindow)!=maxy || aContext->guiClientRect.bottom!=maxysize) {
        maxy=y+guiGetVScrollPos(aContext->currentWindow);
        maxysize=aContext->guiClientRect.bottom;
        guiVScrollRange(aContext->currentWindow, 0, maxy, maxysize);
        recalcCursor=true;
      } if (recalcCursor) {
        if (ycursor<0) {
	        guiSetVScrollPos(aContext->currentWindow, guiGetVScrollPos(aContext->currentWindow)+ycursor);
        } else if (ycursor+14>aContext->guiClientRect.bottom && aContext->guiClientRect.bottom>14) {
  	      guiSetVScrollPos(aContext->currentWindow, guiGetVScrollPos(aContext->currentWindow)+ycursor-aContext->guiClientRect.top);
        }
        recalcCursor=false;
      }
    } break;
  case GUIEVENT_MOUSEBUTTON1DOWN: {
      int i,j;
      int areaCount=0;

      for(i=0;i<maxProjects;i++) {
        areaCount++;
        if (aContext->id==areaCount) {
          if (aContext->mouseClientX<20) {
            projects[i].unfoldedProjectView = !projects[i].unfoldedProjectView;
          }
          selectProjectBrowserItem(i, NULL, -1, -1, ctrlPressed());
          transportWindowRefresh=true;
          break;
        }
        if (projects[i].unfoldedProjectView) {
          for(j=0;j<projects[i].maxTracks;j++) {
            if (projects[i].tracks[j].inUse) {
              areaCount++;
              if (aContext->id==areaCount) {
                selectProjectBrowserItem(i, NULL, -1, j, ctrlPressed());
                break;
              }
            }
          }
          for(j=0;j<projects[i].maxPatches;j++) {
            areaCount++;
            if (aContext->id == areaCount) {
              if (aContext->mouseClientX<80) {
                projects[i].patches[j].unfoldedPatchView = !projects[i].patches[j].unfoldedPatchView;
              }
              selectProjectBrowserItem(i, &(projects[i].patches[j]), -1, -1, ctrlPressed());
              break;
            }
          }
        }
      }
    } break;
  case GUIEVENT_MOUSEBUTTON1DCLK: {
      int i,j;
      int areaCount=0;

      for(i=0;i<maxProjects;i++) {
        areaCount++;
        if (aContext->id == areaCount) {
          projects[i].unfoldedProjectView = !projects[i].unfoldedProjectView;
          projectBrowserWindowRefresh=true;
          break;
        }
        if (projects[i].unfoldedProjectView) {
          for(j=0;j<projects[i].maxTracks;j++) {
            if (projects[i].tracks[j].inUse) {
              areaCount++;
              if (aContext->id==areaCount) {
                selectProjectBrowserItem(i, NULL, -1, j, ctrlPressed());
                break;
              }
            }
          }
          for(j=0;j<projects[i].maxPatches;j++) {
            areaCount++;
            if (aContext->id == areaCount) {
              selectProjectBrowserItem(i, &(projects[i].patches[j]), -1, -1, ctrlPressed());
              break;
            }
          }
        }
      }
    } break;
  case GUIEVENT_CHAR:
    switch(aContext->id) {
    case 3:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        saveStfToClipboard();
      }
      break;
    case 4:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        clone();
      }
      break;
    case 22:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        loadFromClipboard();
      }
      break;
    case 24:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        saveStfToClipboard();
        deleteSelected();
      }
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    case VK_RETURN: {
        int i,j;
        int pos=0;

        for(i=0;i<maxProjects;i++) {
          if (projectBrowserPos == pos) {
            projects[i].unfoldedProjectView = !projects[i].unfoldedProjectView;
            selectProjectBrowserItem(i, NULL, -1, -1, ctrlPressed());
            transportWindowRefresh=true;
            goto leaveProjectBrowserReturn;
          }
          pos++;
          if (projects[i].unfoldedProjectView) {
            for(j=0;j<projects[i].maxTracks;j++) {
              if (projects[i].tracks[j].inUse) {
                if (projectBrowserPos==pos) {
                  selectProjectBrowserItem(i, NULL, -1, j, ctrlPressed());
                  goto leaveProjectBrowserReturn;
                }
                pos++;
              }
            }
            for(j=0;j<projects[i].maxPatches;j++) {
              if (projectBrowserPos==pos) {
                projects[i].patches[j].unfoldedPatchView = !projects[i].patches[j].unfoldedPatchView;
                selectProjectBrowserItem(i, &(projects[i].patches[j]), -1, -1, ctrlPressed());
                goto leaveProjectBrowserReturn;
              }
              pos++;
            }
          }
        }
      } leaveProjectBrowserReturn: break;
    case VK_SPACE: {
        int i,j;
        int pos=0;

        for(i=0;i<maxProjects;i++) {
          if (projectBrowserPos == pos) {
            selectProjectBrowserItem(i, NULL, -1, -1, ctrlPressed());
            transportWindowRefresh=true;
            goto leaveProjectBrowserSpace;
          }
          pos++;
          if (projects[i].unfoldedProjectView) {
            for(j=0;j<projects[i].maxTracks;j++) {
              if (projects[i].tracks[j].inUse) {
                if (projectBrowserPos==pos) {
                  selectProjectBrowserItem(i, NULL, -1, j, ctrlPressed());
                  goto leaveProjectBrowserSpace;
                }
                pos++;
              }
            }
            for(j=0;j<projects[i].maxPatches;j++) {
              if (projectBrowserPos==pos) {
                selectProjectBrowserItem(i, &(projects[i].patches[j]), -1, -1, ctrlPressed());
                goto leaveProjectBrowserSpace;
              }
              pos++;
            }
          }
        }  
      } leaveProjectBrowserSpace: break;
    case VK_UP:
      if (projectBrowserPos>0) {
        projectBrowserPos--;
        projectBrowserWindowRefresh=true;
        recalcCursor=true;
      }
      break;
    case VK_DOWN:
      if (projectBrowserPos<guiGetAreaListLen(aContext)-1) {
        projectBrowserPos++;
        projectBrowserWindowRefresh=true;
        recalcCursor=true;
      }
      break;
    case VK_DELETE:
      deleteSelected();
      break;
    }
    break;
  case GUIEVENT_CLOSE:
    guiHideWindow(aContext->currentWindow);
    break;
  default:
    break;
  }
}
EOF

  genWindow('projectBrowserWindow', 'PROJECTWINDOW', '"Project Browser"', 'projectBrowserHandler', undef, undef, 1, undef);
}


##########################################################################
#
# Color selector window
#
##########################################################################
sub genColorSelectorWindow {
  my $noofCableColors=8;

  $rd .= "static const int noofCableColors=$noofCableColors;\n";
  $rv .= "static int currentCableColorIndex=0;\n";
#  $rv .= "static int cablePenColors[noofCableColors] ={0000,0004,0040,0400,0044,0404,0440,0444};\n";
#  $rv .= "static int cableFillColors[noofCableColors]={0333,0007,0070,0700,0077,0707,0770,0xFFFFFF};\n";
  $rv .= "static int cablePenColors[noofCableColors] ={0x000000,0x000088,0x008800,0x880000,0x008888,0x880088,0x888800,0x888888};\n";
  $rv .= "static int cableFillColors[noofCableColors]={0x666666,0x0000BB,0x00BB00,0xBB0000,0x00BBBB,0xBB00BB,0xBBBB00,0xFFFFFF};\n";
  $rc .= <<EOF;
static void setCurrentCableColor(int index) {
  int i;

  currentCableColorIndex=index;
  colorSelectorWindowRefresh=true;
  for(i=0;i<noofCableColors;i++) {
    markMenuItem(mainMenu, MENU_CABLECOLOR_0+i, i==index);
    markMenuItem(synFactoryMenu, MENU_CABLECOLOR_0+i, i==index);
//    for(j=0;j<noofLanguages;j++) {
      // !!! make platform independed
//      CheckMenuItem(mainMenu[j], MENU_CABLECOLOR_0+i, MF_BYCOMMAND | ((i==index)?MF_CHECKED:MF_UNCHECKED));
//      CheckMenuItem(synFactoryMenu[j], MENU_CABLECOLOR_0+i, MF_BYCOMMAND | ((i==index)?MF_CHECKED:MF_UNCHECKED));
//    }
  }
}
static void cableColorHandler(ContextPtr aContext) {
  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      guiClearAreaList(aContext);
      DrawEdge(aContext->hdc, &(aContext->guiClientRect), EDGE_SUNKEN, BF_MIDDLE);

      GuiRect myRect;
      for(int i=0; i<noofCableColors; i++) {
        myRect.top = ((i*2)/noofCableColors)*32+2;
        myRect.bottom = ((i*2)/noofCableColors)*32+30;
        myRect.left = (i&(noofCableColors/2-1))*32+34;
        myRect.right = (i&(noofCableColors/2-1))*32+62;

        DrawEdge(aContext->hdc, &myRect, (i==currentCableColorIndex)?EDGE_SUNKEN:EDGE_RAISED, BF_MIDDLE | BF_RECT);
        guiSelectPen3Color(aContext, cablePenColors[i]);
        guiSelectFillColor(aContext, cableFillColors[i]);
        Ellipse(aContext->hdc, ((i==currentCableColorIndex)?1:0)+myRect.left+6, ((i==currentCableColorIndex)?1:0)+myRect.top+6, ((i==currentCableColorIndex)?1:0)+myRect.left+22, ((i==currentCableColorIndex)?1:0)+myRect.top+22);
        guiAddArea(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);
      }
      
      myRect.top = 2;
      myRect.bottom = 30;
      myRect.left = 2;
      myRect.right = 30;
      int offs=(showResizeMarkers)?1:0;
      DrawEdge(aContext->hdc, &myRect, (showResizeMarkers)?EDGE_SUNKEN:EDGE_RAISED, BF_MIDDLE | BF_RECT);
      guiSelectPen1Color(aContext, 0x000000);
      guiSelectFillColor(aContext, -1);
      guiDrawRect(aContext, 10+offs, 10+offs, 20+offs, 20+offs);
      guiSelectPen1Color(aContext, 0x000000);
      guiSelectFillColor(aContext, 0xFFFFFF);
      guiDrawRect(aContext,  6+offs,  6+offs, 10+offs, 10+offs);
      guiDrawRect(aContext,  6+offs, 20+offs, 10+offs, 24+offs);
      guiDrawRect(aContext, 20+offs,  6+offs, 24+offs, 10+offs);
      guiDrawRect(aContext, 20+offs, 20+offs, 24+offs, 24+offs);
      guiAddArea(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);

      myRect.top = 34;
      myRect.bottom = 62;
      myRect.left = 2;
      myRect.right = 30;
      DrawEdge(aContext->hdc, &myRect, (showResizeMarkers)?EDGE_SUNKEN:EDGE_RAISED, BF_MIDDLE | BF_RECT);
    } break;
  case GUIEVENT_MOUSEBUTTON1DOWN:
    if (aContext->id>0 && aContext->id<=noofCableColors) {
      setCurrentCableColor(aContext->id-1);
    }
    if (aContext->id>noofCableColors) {
      showResizeMarkers=!showResizeMarkers;
      colorSelectorWindowRefresh=true;
      mainWindowRefresh=true;
    }
    break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(aContext->currentWindow);
    break;
  default:
    break;
  }
}
EOF

  genWindow('colorSelectorWindow', 'COLORSELECTORWINDOW', '"Color Selector"', 'cableColorHandler', ($noofCableColors+2)*16, 64, 0, 1);
  push(@initFunctions, "setCurrentCableColor(0);\n");
}


##########################################################################
#
# Midi Activity Monitor
#
##########################################################################
sub genMidiActivityMonitor {
  $rd .= "static const int midiActivityLedSize=10;\n";

# !!! hack
  $rd .= "static const int maxMidiPorts=26;\n";

  $rc .= <<EOF;
// !!! hack
static const char * const midiPortNames="-ABCDEFGHIJKLMNOPQRSTUVWXYZ";

static const char *rownames="01020304050607080910111213141516RT";
static int midiActivityLedOld[maxMidiPorts][17];
static int midiActivityLedBlink[maxMidiPorts][17];
static bool midiActivityFlag;

static int midiActivityColor[8]={0x666666, 0x667777, 0x668888, 0x66AAAA, 0x66BBBB, 0x66CCCC, 0x66DDDD, 0x66EEEE};
static void midiActivityMonitorDrawLeds(ContextPtr aContext, bool forceUpdate) {
  int x,y;

  guiSelectPen1Color(aContext, -1);
  for(x=0;x<maxMidiPorts;x++) {
    for(y=0;y<17;y++) {
      if (forceUpdate || midiActivityLedBlink[x][y]!=midiActivityLedOld[x][y]) {
        int blink=midiActivityLedBlink[x][y];
        GuiRect myrect={x*midiActivityLedSize+20, y*midiActivityLedSize+16, x*midiActivityLedSize+20+midiActivityLedSize-2, y*midiActivityLedSize+16+midiActivityLedSize-2};
        
        guiSelectFillColor(aContext, midiActivityColor[blink]);
        guiDrawEllipse(aContext, &myrect);
        midiActivityLedOld[x][y]=midiActivityLedBlink[x][y];
      }
      if (midiActivityLedBlink[x][y]>0) {            
        midiActivityLedBlink[x][y]--;
        midiActivityFlag=true; // Must turn led off later
      }
    }
  }
}

static void midiActivityMonitorHandler(ContextPtr aContext) {
  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      int x,y;

      guiSelectFillColor(aContext, 0);
      guiSelectPen1Color(aContext, 0);
      guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom);

      guiSelectPen1Color(aContext, 0xFFFFFF);
      SetTextAlign(aContext->hdc, TA_LEFT | TA_TOP);

      guiSelectPotFont(aContext);
      for(x=0;x<maxMidiPorts;x++) {
        TextOut(aContext->hdc, x*midiActivityLedSize+20, 0, &midiPortNames[x+1], 1);
      }
      guiSelectIOFont(aContext);
      for(y=0;y<17;y++) {
        TextOut(aContext->hdc, 4, y*midiActivityLedSize+16, &rownames[y*2], 2);
      }

      midiActivityMonitorDrawLeds(aContext, true);
    } break;
  case GUIEVENT_TIMERTICK: {
      if (midiActivityFlag) {
//        midiActivityMonitorWindowRefresh=true;
        midiActivityFlag=false;
        midiActivityMonitorDrawLeds(aContext, false);
      }
    }
    break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(aContext->currentWindow);
    break;
  default:
    break;
  }
}

static void snoopMidiActivity(int msg) {
  if ((msg&0xF0)==0xF0) {
    midiActivityLedBlink[(msg>>24)-1][16]=min(6,midiActivityLedBlink[(msg>>24)-1][16]+2);
  } else {
    midiActivityLedBlink[(msg>>24)-1][msg&0x0F]=min(6,midiActivityLedBlink[(msg>>24)-1][msg&0x0F]+2);
  }
  midiActivityFlag=true;
}
EOF

  genWindow('midiActivityMonitorWindow', 'MIDIACTIVITYWINDOW', '"Midi Activity Monitor"', 'midiActivityMonitorHandler', 'midiActivityLedSize*maxMidiPorts+20', 'midiActivityLedSize*17+16', 0, 1);
  push(@initWindows, "SetTimer(midiActivityMonitorWindow, 1, 50, NULL);");
}

##########################################################################
#
# Midi Scope
#
##########################################################################
sub genMidiScope {
  $rc .= <<EOF;

static const int MAX_MIDI_SCOPE_EVENTS=4096; // Needs to be power of 2
static const int FILTERFLAGS=13;
static const int MIDICHANNELS=16;
#define MIDI_MONITOR_MASK (MAX_MIDI_SCOPE_EVENTS-1)

static int monitorTable[MAX_MIDI_SCOPE_EVENTS];
static int currentPosition;
static int lastPosition;
static bool monitorPortFlag[maxMidiPorts];
static bool monitorChannelFlag[16]={
  true,true,true,true,
  true,true,true,true,
  true,true,true,true,
  true,true,true,true
};
static bool filterFlag[FILTERFLAGS]={true, true, true, true, true, true, true, false, false, false, true, true, true};
static const char *midiScopeFilterNames[FILTERFLAGS]={"Note", "CCmsg", "Prog", "PBend", "KyAft", "ChAft", "Song", "Clock", "SMPTE", "ASens", "Tune", "Reset", "Other"};

static void snoopMidiScope(int midiMessage) {
  if (monitorPortFlag[(midiMessage>>24)-1]) {
    if (((midiMessage&0xF0)==0x80 && monitorChannelFlag[midiMessage&0x0F] && filterFlag[0]) // Note Off
     || ((midiMessage&0xF0)==0x90 && monitorChannelFlag[midiMessage&0x0F] && filterFlag[0]) // Note On
     || ((midiMessage&0xF0)==0xB0 && monitorChannelFlag[midiMessage&0x0F] && filterFlag[1]) // CC message
     || ((midiMessage&0xF0)==0xC0 && monitorChannelFlag[midiMessage&0x0F] && filterFlag[2]) // Program change
     || ((midiMessage&0xF0)==0xE0 && monitorChannelFlag[midiMessage&0x0F] && filterFlag[3]) // PitchBend
     || ((midiMessage&0xF0)==0xA0 && monitorChannelFlag[midiMessage&0x0F] && filterFlag[4]) // Key aftertouch
     || ((midiMessage&0xF0)==0xD0 && monitorChannelFlag[midiMessage&0x0F] && filterFlag[5]) // Channel aftertouch
     || ((midiMessage&0xFF)==0xF2 && filterFlag[6])  // Song Position
     || ((midiMessage&0xFF)==0xF3 && filterFlag[6])  // Song Select
     || ((midiMessage&0xFF)==0xF8 && filterFlag[7])  // Clock
     || ((midiMessage&0xFF)==0xF1 && filterFlag[8])  // SMPTE
     || ((midiMessage&0xFF)==0xFE && filterFlag[9])  // Active Sense
     || ((midiMessage&0xFF)==0xF6 && filterFlag[10]) // Tune request
     || ((midiMessage&0xFF)==0xFF && filterFlag[11]) // Reset
     || ((midiMessage&0xFF)==0xFA && filterFlag[6])  // Start
     || ((midiMessage&0xFF)==0xFB && filterFlag[6])  // Continue
     || ((midiMessage&0xFF)==0xFC && filterFlag[6])  // Stop
     || ((midiMessage&0xFF)==0xF4 && filterFlag[12]) // Unknown
     || ((midiMessage&0xFF)==0xF5 && filterFlag[12]) // Unknown
     || ((midiMessage&0xFF)==0xF9 && filterFlag[12]) // Unknown
     || ((midiMessage&0xFF)==0xFD && filterFlag[12]) // Unknown
       ) {
      monitorTable[currentPosition]=midiMessage;
      currentPosition=(currentPosition+1)&MIDI_MONITOR_MASK;

      // Clear next row (to prevent display of very old data)
      monitorTable[currentPosition]=0;
      monitorTable[(currentPosition+1)&MIDI_MONITOR_MASK]=0;
      monitorTable[(currentPosition+2)&MIDI_MONITOR_MASK]=0;
      monitorTable[(currentPosition+3)&MIDI_MONITOR_MASK]=0;
    }
  }
}

static void midiScopeHandler(ContextPtr aContext) {
  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      int y;
      int x;
      int i;

      guiClearAreaList(aContext);
//      GuiRect fillRect=aContext->guiClientRect;
      int scrollpos=(MAX_MIDI_SCOPE_EVENTS-(guiGetVScrollPos(aContext->currentWindow)+15))&(~3);

//      fillRect.bottom-=66;

//      DrawEdge(aContext->hdc, &(aContext->guiClientRect), 0 ,BF_MIDDLE);
      guiSelectPen1Color(aContext, -1);
      guiSelectFillColor(aContext, 0xFFFFFF);
      guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom-66);

      guiTextTransparent(aContext, true);
      guiSelectPen1Color(aContext, 0);

      guiSelectRouterFont(aContext);
      for(y=0;y<20;y++) {
        for(x=0;x<4;x++) {
          if (scrollpos+y*4+x<(MAX_MIDI_SCOPE_EVENTS-4)) {
            char buf[256];
            int pos=(((currentPosition)&(~3))+x-(scrollpos+y*4))&MIDI_MONITOR_MASK;

            switch(monitorTable[pos]&0xF0) {
            case 0x80:
              sprintf(buf, "%c%02d N.Off %3d %3d", midiPortNames[monitorTable[pos]>>24], (monitorTable[pos]&15)+1, (monitorTable[pos]>>8)&255, (monitorTable[pos]>>16)&255);
              break;
            case 0x90:
              sprintf(buf, "%c%02d Note  %3d %3d", midiPortNames[monitorTable[pos]>>24], (monitorTable[pos]&15)+1, (monitorTable[pos]>>8)&255, (monitorTable[pos]>>16)&255);
              break;
            case 0xA0:
              sprintf(buf, "%c%02d KyAft %3d %3d", midiPortNames[monitorTable[pos]>>24], (monitorTable[pos]&15)+1, (monitorTable[pos]>>8)&255, (monitorTable[pos]>>16)&255);
              break;
            case 0xB0:
              sprintf(buf, "%c%02d CCmsg %3d %3d", midiPortNames[monitorTable[pos]>>24], (monitorTable[pos]&15)+1, (monitorTable[pos]>>8)&255, (monitorTable[pos]>>16)&255);
              break;
            case 0xC0:
              sprintf(buf, "%c%02d Prog. %3d", midiPortNames[monitorTable[pos]>>24], (monitorTable[pos]&15)+1, (monitorTable[pos]>>8)&255);
              break;
            case 0xD0:
              sprintf(buf, "%c%02d ChAft %3d", midiPortNames[monitorTable[pos]>>24], (monitorTable[pos]&15)+1, (monitorTable[pos]>>8)&255);
              break;
            case 0xE0:
              sprintf(buf, "%c%02d PBend %6d", midiPortNames[monitorTable[pos]>>24], (monitorTable[pos]&15)+1, ((monitorTable[pos]>>16)&255)*128+((monitorTable[pos]>>8)&255));
              break;
            case 0xF0:
              switch(monitorTable[pos]&0x0F) {
              case 0x01:
                sprintf(buf, "%c-- SMPTE", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x02:
                sprintf(buf, "%c-- SongPos.", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x03:
                sprintf(buf, "%c-- SongSel.", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x04:
                sprintf(buf, "%c-- 0xF4", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x05:
                sprintf(buf, "%c-- 0xF5", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x06:
                sprintf(buf, "%c-- TuneRequest", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x07:
//                sprintf(buf, "%c-- ", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x08:
                sprintf(buf, "%c-- Clock", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x09:
                sprintf(buf, "%c-- 0xF9", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x0A:
                sprintf(buf, "%c-- Start", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x0B:
                sprintf(buf, "%c-- Continue", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x0C:
                sprintf(buf, "%c-- Stop", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x0D:
                sprintf(buf, "%c-- 0xFD", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x0E:
                sprintf(buf, "%c-- Active", midiPortNames[monitorTable[pos]>>24]);
                break;
              case 0x0F:
                sprintf(buf, "%c-- Reset", midiPortNames[monitorTable[pos]>>24]);
                break;
              }
              break;
            default:
              continue;
            }
            TextOut(aContext->hdc, x*142+8, aContext->guiClientRect.bottom-(88+y*14), buf, strlen(buf));
          }
        }
      }

      // Draw port selection buttons
      guiSelectFillColor(aContext, -1);
      guiSelectTitleFont(aContext);
      for(i=0;i<maxMidiPorts;i++) {
        guiAddArea(aContext, i*22, aContext->guiClientRect.bottom-44, i*22+22, aContext->guiClientRect.bottom-22);
        GuiRect rect;
        rect.left=i*22;
        rect.top=aContext->guiClientRect.bottom-44;
        rect.right=i*22+22;
        rect.bottom=aContext->guiClientRect.bottom-22;
        guiDraw3DRect(aContext, &rect, monitorPortFlag[i]?EDGE_SUNKEN:EDGE_RAISED);
        TextOut(aContext->hdc, i*22+4, aContext->guiClientRect.bottom-40, &midiPortNames[i+1], 1);
      }

      // When program has limited midi support (sf lite?) fill rest with blanks
      for(;i<26;i++) {
        GuiRect blankRect;
        blankRect.left=i*22;
        blankRect.top=aContext->guiClientRect.bottom-44;
        blankRect.right=i*22+22;
        blankRect.bottom=aContext->guiClientRect.bottom-22;
        guiDraw3DRect(aContext, &blankRect, EDGE_ETCHED);
      }

      // Draw channel selection buttons
      guiSelectTitleFont(aContext);
      for(i=0;i<MIDICHANNELS;i++) {
        char buf[16];
        GuiRect rect;
        rect.left=i*35;
        rect.top=aContext->guiClientRect.bottom-22;
        rect.right=i*35+35;
        rect.bottom=aContext->guiClientRect.bottom;
        guiDraw3DRect(aContext, &rect, monitorChannelFlag[i]?EDGE_SUNKEN:EDGE_RAISED);
        sprintf(buf, "%d", i+1);
        TextOut(aContext->hdc, i*35+4, aContext->guiClientRect.bottom-18, buf, strlen(buf));
        guiAddArea(aContext, rect.left, rect.top, rect.right, rect.bottom);
      }

      {
        GuiRect blankRect;
        blankRect.left=16*35;
        blankRect.right=aContext->guiClientRect.right;
        blankRect.top=aContext->guiClientRect.bottom-22;
        blankRect.bottom=aContext->guiClientRect.bottom;
        guiDraw3DRect(aContext, &blankRect, EDGE_ETCHED);
      }

      // Draw filter buttons
      guiSelectRouterFont(aContext);
      for(i=0;i<FILTERFLAGS;i++) {
        GuiRect rect;
        rect.left=i*44;
        rect.top=aContext->guiClientRect.bottom-66;
        rect.right=i*44+44;
        rect.bottom=aContext->guiClientRect.bottom-44;
        guiDraw3DRect(aContext, &rect, filterFlag[i]?EDGE_SUNKEN:EDGE_RAISED);
        TextOut(aContext->hdc, i*44+4, aContext->guiClientRect.bottom-62, midiScopeFilterNames[i], strlen(midiScopeFilterNames[i]));
        guiAddArea(aContext, rect.left, rect.top, rect.right, rect.bottom);
      }
    } break;
  case GUIEVENT_MOUSEBUTTON1DOWN: {
      if (aContext->id>0) {
        if (aContext->id>maxMidiPorts+MIDICHANNELS) {
          filterFlag[aContext->id-(maxMidiPorts+MIDICHANNELS+1)]=!filterFlag[aContext->id-(maxMidiPorts+MIDICHANNELS+1)];
          midiScopeWindowRefresh=true;
        } else if (aContext->id>maxMidiPorts) {
          monitorChannelFlag[aContext->id-(maxMidiPorts+1)]=!monitorChannelFlag[aContext->id-(maxMidiPorts+1)];
          midiScopeWindowRefresh=true;
        } else {
          monitorPortFlag[aContext->id-1]=!monitorPortFlag[aContext->id-1];
          midiScopeWindowRefresh=true;
        }
      }
    } break;
  case GUIEVENT_TIMERTICK: {
      if (currentPosition!=lastPosition) {
        lastPosition=currentPosition;
        midiScopeWindowRefresh=true;
      }
    } break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(aContext->currentWindow);
    break;
  default:
    break;
  }
}
EOF

  genWindow('midiScopeWindow', 'MIDISCOPEWINDOW', '"Midi Scope"', 'midiScopeHandler', '572+GetSystemMetrics(SM_CXVSCROLL)', 360, 1, 1);

  push(@initWindows, "guiVScrollRange(midiScopeWindow, 0, MAX_MIDI_SCOPE_EVENTS, 16);");
  push(@initWindows, "guiVScrollWindowTo(midiScopeWindow, MAX_MIDI_SCOPE_EVENTS);");
  push(@initWindows, "SetTimer(midiScopeWindow, 1, 100, NULL);");

  push(@stfSave, '{ int i; storeStfBegin(loadInfo, "MidiIScopeSettings"); for(i=0;i<maxMidiPorts;i++) { storeStfInteger(loadInfo, monitorPortFlag[i]?1:0); } for(i=0;i<16;i++) { storeStfInteger(loadInfo, monitorChannelFlag[i]?1:0); } for(i=0;i<FILTERFLAGS;i++) { storeStfInteger(loadInfo, filterFlag[i]?1:0); } storeStfEnd(loadInfo); }');
  push(@stfLoad, 'if (strcmp(loadInfo->headerTitle, "MidiIScopeSettings")==0) { int i,f; for(i=0;i<maxMidiPorts;i++) { readStfInteger(loadInfo, &f); monitorPortFlag[i]=(f!=0); } for(i=0;i<16;i++) { readStfInteger(loadInfo, &f); monitorChannelFlag[i]=(f!=0); } for(i=0;i<FILTERFLAGS;i++) { readStfInteger(loadInfo, &f); filterFlag[i]=(f!=0); } }');
}


##########################################################################
#
# MIDI router
#
##########################################################################
sub genMidiRouter {
  my @midiStructDef=('unsigned char fromPort;', 'unsigned char command;', 'unsigned char data1;', 'unsigned char data2;');
  genEnumStruct("struct", "MidiMessage", \@midiStructDef);

  rv('int noofMidiInDevices;');
  rv('int noofMidiOutDevices;');
  rv('const char **midiInDeviceNames;');
  rv('const char **midiOutDeviceNames;');
  rv('int *midiInPorts;');
  rv('int *midiOutPorts;');

  rv('HMIDIIN *midiInHandles;');
  rv('HMIDIOUT *midiOutHandles;');

  mutex('midiMutex');

  $rc .= <<EOF;
//
// Midi message format
//                     7..0  MIDI command
//               15..8       MIDI data 1
//        23..16             MIDI data 2
// 29..24                    port (1-27)
//
static void routeMidiMessage(int msg) {  
  lockMutex(midiMutex);
  
  // Send to SynFactory patch
  patchProcessMidi(msg);

  // Send to midi monitor
  snoopMidiActivity(msg);
  snoopMidiScope(msg);

  unlockMutex(midiMutex);
}

static void CALLBACK midiInProc(HMIDIIN hMidiIn, UINT wMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2) {
  if (wMsg==MIM_DATA) {
    routeMidiMessage((dwParam1&0x00FFFFFF)|(midiInPorts[dwInstance]<<24));
  }
}

static void setMidiInPort(int device, int port) {
  if (device>=0 && device<noofMidiInDevices) {
    lockMutex(midiMutex);
    if (port==0 && midiInPorts[device]>0) {
      midiInReset(midiInHandles[device]);
      midiInClose(midiInHandles[device]);
      midiInHandles[device]=NULL;
    }
    if (port>0 && midiInPorts[device]==0) {
      if (midiInOpen(&(midiInHandles[device]), device, (DWORD)midiInProc, device, CALLBACK_FUNCTION)==MMSYSERR_NOERROR) {
        midiInStart(midiInHandles[device]);
      } else {
        midiInHandles[device]=NULL;
      }
    }
    midiInPorts[device]=port;
    configWindowRefresh=true;
    unlockMutex(midiMutex);
  }
}

static void setMidiOutPort(int device, int port) {
  if (device>=0 && device<noofMidiOutDevices) {
    lockMutex(midiMutex);
    if (port==0) {
//      midiOutReset(midiOutHandles[device]);
//      midiOutClose(midiOutHandles[device]);
      midiOutPorts[device]=port;
    } else if (midiOutPorts[device]==0) {
      midiOutPorts[device]=port;
//      midiOutOpen(&(midiOutHandles[device]), device, (DWORD)midiInProc, device, CALLBACK_FUNCTION);
//      midiOutStart(midiOutHandles[device]);
    }
    configWindowRefresh=true;
    unlockMutex(midiMutex);
  }
}

static void openMidiRouter(void) {
  MIDIINCAPS midiInInfo;
  MIDIOUTCAPS midiOutInfo;
  int i;
  
  noofMidiInDevices=midiInGetNumDevs();
  midiInDeviceNames=(const char **)malloc((noofMidiInDevices+1)*sizeof(const char *));
  midiInPorts=(int *)calloc(noofMidiInDevices, sizeof(int));
  midiInHandles=(HMIDIIN *)calloc(noofMidiInDevices, sizeof(HMIDIIN));
  for(i=0;i<noofMidiInDevices;i++) {
    if (MMSYSERR_NOERROR==midiInGetDevCaps(i, &midiInInfo, (UINT)sizeof(MIDIINCAPS))) {
      midiInDeviceNames[i]=strdup(midiInInfo.szPname);
    } else {
      midiInDeviceNames[i]="(unknown)";
    }
  }
  midiInDeviceNames[noofMidiInDevices]=NULL;

  noofMidiOutDevices=midiOutGetNumDevs();
  midiOutDeviceNames=(const char **)malloc((noofMidiOutDevices+1)*sizeof(const char *));
  midiOutPorts=(int *)calloc(noofMidiOutDevices, sizeof(int));
  midiOutHandles=(HMIDIOUT *)calloc(noofMidiOutDevices, sizeof(HMIDIOUT));
  for(i=0;i<noofMidiOutDevices;i++) {
    if (MMSYSERR_NOERROR==midiOutGetDevCaps(i, &midiOutInfo, (UINT)sizeof(MIDIOUTCAPS))) {
      midiOutDeviceNames[i]=strdup(midiOutInfo.szPname);
    } else {
      midiOutDeviceNames[i]="(unknown)";
    }
  }
  midiOutDeviceNames[noofMidiOutDevices]=NULL;
}

static void closeMidiRouter(void) {
  int i;

  for(i=0;i<noofMidiInDevices;i++) {
    setMidiInPort(i, 0);
  }
  for(i=0;i<noofMidiOutDevices;i++) {
    setMidiOutPort(i, 0);
  }
}
EOF

  push(@initFunctions, 'openMidiRouter();');
  push(@termFunctions, 'closeMidiRouter();');
  push(@stfSave, '{ int i; for(i=0;i<noofMidiInDevices;i++) { storeStfBegin(loadInfo, "MidiInDevice"); storeStfInteger(loadInfo, i+1); storeStfInteger(loadInfo, midiInPorts[i]); storeStfEnd(loadInfo); } }');
  push(@stfSave, '{ int i; for(i=0;i<noofMidiOutDevices;i++) { storeStfBegin(loadInfo, "MidiOutDevice"); storeStfInteger(loadInfo, i+1); storeStfInteger(loadInfo, midiOutPorts[i]); storeStfEnd(loadInfo); } }');
  push(@stfLoad, 'if (strcmp(loadInfo->headerTitle, "MidiInDevice")==0) { int i,p; readStfInteger(loadInfo, &i); readStfInteger(loadInfo, &p); setMidiInPort(i-1, p); }');
  push(@stfLoad, 'if (strcmp(loadInfo->headerTitle, "MidiOutDevice")==0) { int i,p; readStfInteger(loadInfo, &i); readStfInteger(loadInfo, &p); setMidiOutPort(i-1, p); }');

}


##########################################################################
#
# HTML viewer
#
##########################################################################
sub genRenderHtml {

$rc .= <<EOF;
#define FONT_SIZE_SMALL      0x00
#define FONT_SIZE_NORMAL     0x08
#define FONT_SIZE_BIG        0x10
#define FONT_SIZE_HUGE       0x18
#define FONT_FIXEDPITCH      0x20

#define FONTMASK_UNDERLINE   0x01
#define FONTMASK_BOLD        0x02
#define FONTMASK_ITALIC      0x04
#define FONTMASK_SIZE        0x18
#define FONTMASK_FIXEDPITCH  0x20
const int FontSizes[4]={12,14,18,28};


static HFONT Fonts[64];


//
// Render a html encoded page to the screen
// inputs
//   aContext - Current runtime context
//   destRect - Region which must be used to render HTML page
//   ypos - Current scroll pos (negative to scrollup)
//   page - HTML page to render
// outputs
//   ypos - maximum Ypos used to render HTML
//
static void renderHTMLPage(ContextPtr aContext, RECT *destRect, int *ypos, const char *page) {
//  char textBuffer[TEXTBUFFERSIZE];
  bool forceonespace=false; // If hit a cr/lf pair in none TT mode insert space instead.
  bool PREflag=false;
  bool tableMode=false;
  int fontmask=FONT_SIZE_NORMAL;
  int curX=destRect->left;
  int maxY=0;
  int nrchars;

  //
  // Restrict painting in provided space by 'destRect'
  //
  SaveDC(aContext->hdc);
  HRGN clipRgn=CreateRectRgnIndirect((RECT *)destRect);
  SelectObject(aContext->hdc, clipRgn);
  DeleteObject(clipRgn);

  //
  // Set correct painting modes
  //
  SetBkMode(aContext->hdc, TRANSPARENT);
  SetTextAlign(aContext->hdc, TA_LEFT | TA_TOP);
  (*ypos)+=destRect->top;

  SetTextColor(aContext->hdc, PALETTERGB(0,0,0));

  if (page) while (*page) {
    SIZE tSize;

    SelectObject(aContext->hdc, Fonts[fontmask]);
    nrchars=0;

    while (page[nrchars] && ((unsigned char)page[nrchars])>31 && page[nrchars]!='<' && page[nrchars]!='&') {
      nrchars++;
    }

    if (forceonespace) {
      if (curX>12) {
        GetTextExtentPoint32(aContext->hdc, " ", 1, &tSize);
        if (tSize.cx+curX > destRect->right) {
          (*ypos)+=maxY;
          maxY=0;
          curX=destRect->left;
//          TextOut(aContext->hdc, curX, (*ypos), page, nrchars);
        } else {
          TextOut(aContext->hdc, curX, (*ypos), " ", 1);
        }
        curX+=tSize.cx;
        maxY=max(maxY, tSize.cy);
      }
      forceonespace=false;
    }

    // Word wrap
    if (nrchars>0) {
      for(;;) {
        GetTextExtentPoint32(aContext->hdc, page, nrchars, &tSize);
        if (tSize.cx+curX >= destRect->right) {
          int backup=nrchars;
          nrchars--;
          while(nrchars>1 && page[nrchars-1]!=' ') {
            nrchars--;
          }
          if (nrchars<=1) {
            if (curX>0) {
              // Force newline
              (*ypos)+=maxY;
              maxY=0;
              curX=destRect->left;
            }
            nrchars=backup;
            TextOut(aContext->hdc, curX, (*ypos), page, nrchars);
            break;
          }
        } else {
          TextOut(aContext->hdc, curX, (*ypos), page, nrchars);
          break;
        }
      }

      curX+=tSize.cx;
      maxY=max(maxY, tSize.cy);
      page+=nrchars;
    }

    if (*page=='<') {
      char htmltag[32];
      int htmlcnt=0;

      page++;
      while (*page && *page!='>' && htmlcnt<31) {
        htmltag[htmlcnt++]=*page;
        page++;
      }
      htmltag[htmlcnt]='\\0';
        if (stricmp(htmltag, "FONT SIZE=-1")==0) {
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_SMALL;
        }
        if (stricmp(htmltag, "FONT SIZE=+1")==0) {
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_BIG;
        }
        if (stricmp(htmltag, "FONT SIZE=+2")==0) {
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_HUGE;
        }
        if (stricmp(htmltag, "/FONT")==0) {
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_NORMAL;
        }
        if (stricmp(htmltag, "H1")==0) {
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_HUGE | FONTMASK_BOLD | FONTMASK_UNDERLINE;
        }
        if (stricmp(htmltag, "H2")==0) {
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_BIG /*| FONTMASK_BOLD*/ | FONTMASK_UNDERLINE;
        }
        if (stricmp(htmltag, "H3")==0) {
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_NORMAL /*| FONTMASK_BOLD*/ | FONTMASK_UNDERLINE;
        }
        if ((stricmp(htmltag, "/H1")==0) || (stricmp(htmltag, "/H2")==0) || (stricmp(htmltag, "/H3")==0)) {
          fontmask &= ~FONTMASK_UNDERLINE;
          fontmask &= ~FONTMASK_BOLD;
          fontmask &= ~FONTMASK_SIZE;
          fontmask |= FONT_SIZE_NORMAL;
        }
        if (stricmp(htmltag, "TT")==0) {
          fontmask |= FONT_FIXEDPITCH;
        }
        if (stricmp(htmltag, "/TT")==0) {
          fontmask &= ~FONT_FIXEDPITCH;
        }
        if (stricmp(htmltag, "PRE")==0) {
          PREflag=true;
        }
        if (stricmp(htmltag, "/PRE")==0) {
          PREflag=false;
        }

        if (stricmp(htmltag, "BLUE")==0) {
          SetTextColor(aContext->hdc, PALETTERGB(0,0,255));
        }
        if (stricmp(htmltag, "RED")==0) {
          SetTextColor(aContext->hdc, PALETTERGB(255,0,0));
        }
        if (stricmp(htmltag, "BLACK")==0) {
          SetTextColor(aContext->hdc, PALETTERGB(0,0,0));
        }


        if (stricmp(htmltag, "BR")==0) {
          (*ypos)+=maxY;
          maxY=0;
          curX=destRect->left;
        }
/*        if (stricmp(htmltag, "P")==0) {
          if (curX==destRect->left) {
            curX+=12;
          }
        }*/
        if (stricmp(htmltag, "/P")==0 && tableMode==false) {
          (*ypos)+=maxY+16;
          maxY=0;
          curX=destRect->left;
        }
        if (stricmp(htmltag, "/UL")==0) {
          (*ypos)+=maxY+16;
          maxY=0;
          curX=destRect->left;
        }
        if (stricmp(htmltag, "HR")==0) {
          if (curX>4) {
            (*ypos)+=maxY;
            maxY=0;
            curX=destRect->left;
          }
          MoveToEx(aContext->hdc, destRect->left+4, (*ypos), NULL);
          LineTo(aContext->hdc, destRect->right-4, (*ypos));
          (*ypos)+=4; 
        }
        if (stricmp(htmltag, "LI")==0) {
          if (curX>0) {
            (*ypos)+=maxY+16;
            maxY=0;
            curX=destRect->left;
          }
          GetTextExtentPoint32(aContext->hdc, "* ", 2, &tSize);
          TextOut(aContext->hdc, curX, (*ypos), "* ", 2);
          curX+=tSize.cx;
          maxY=max(maxY, tSize.cy);
        }
        //
        // Tables
        if (stricmp(htmltag, "/TR")==0) {
          (*ypos)+=maxY;
          maxY=0;
          curX=destRect->left;
        }
        if (stricmp(htmltag, "TH")==0) {
          tableMode=true;
          fontmask ^= FONTMASK_BOLD;
        }
        if (stricmp(htmltag, "/TH")==0) {
          tableMode=false;
          fontmask ^= FONTMASK_BOLD;
          curX = ((curX+47)/48)*48;
        }
        if (stricmp(htmltag, "TD")==0) {
          tableMode=true;
        }
        if (stricmp(htmltag, "/TD")==0) {
          tableMode=false;
          curX = ((curX+47)/48)*48;
        }
        //
        // Font types
        if (stricmp(htmltag, "B")==0) {
          fontmask |= FONTMASK_BOLD;
        }
        if (stricmp(htmltag, "/B")==0) {
          fontmask &= ~FONTMASK_BOLD;
        }
        if (stricmp(htmltag, "U")==0) {
          fontmask |= FONTMASK_UNDERLINE;
        }
        if (stricmp(htmltag, "/U")==0) {
          fontmask &= ~FONTMASK_UNDERLINE;
        }
        if (stricmp(htmltag, "I")==0) {
          fontmask |= FONTMASK_ITALIC;
        }
        if (stricmp(htmltag, "/I")==0) {
          fontmask &= ~FONTMASK_ITALIC;
        }
      if (*page=='>') {
        page++;
      }
    } else {
      if (*page=='&') {
        char htmltag[32];
        int htmlcnt=0;

        page++;
        while (*page && *page!=';' && htmlcnt<31) {
          htmltag[htmlcnt++]=*page;
          page++;
        }
        htmltag[htmlcnt]='\\0';

          if (stricmp(htmltag, "amp")==0) {
            GetTextExtentPoint32(aContext->hdc, "&", 1, &tSize);
            if (tSize.cx+curX > destRect->right) {
              (*ypos)+=maxY;
              maxY=0;
              curX=destRect->left;
            }
            TextOut(aContext->hdc, curX, (*ypos), "&", 1);
            curX+=tSize.cx;
            maxY=max(maxY, tSize.cy);
          }

          if (stricmp(htmltag, "lt")==0) {
            GetTextExtentPoint32(aContext->hdc, "<", 1, &tSize);
            if (tSize.cx+curX > destRect->right) {
              (*ypos)+=maxY;
              maxY=0;
              curX=destRect->left;
            }
            TextOut(aContext->hdc, curX, (*ypos), "<", 1);
            curX+=tSize.cx;
            maxY=max(maxY, tSize.cy);
          }

          if (stricmp(htmltag, "gt")==0) {
            GetTextExtentPoint32(aContext->hdc, ">", 1, &tSize);
            if (tSize.cx+curX > destRect->right) {
              (*ypos)+=maxY;
              maxY=0;
              curX=destRect->left;
            }
            TextOut(aContext->hdc, curX, (*ypos), ">", 1);
            curX+=tSize.cx;
            maxY=max(maxY, tSize.cy);
          }

          if (stricmp(htmltag, "nbsp")==0) {
            GetTextExtentPoint32(aContext->hdc, " ", 1, &tSize);
            if (tSize.cx+curX > destRect->right) {
              (*ypos)+=maxY;
              maxY=0;
              curX=destRect->left;
            }
            TextOut(aContext->hdc, curX, (*ypos), " ", 1);
            curX+=tSize.cx;
            maxY=max(maxY, tSize.cy);
          }

        if (*page==';') {
          page++;
        }
      }
      if ((*page==10) || (*page==13)) {
        if (PREflag) {
          (*ypos)+=maxY;
          maxY=0;
          curX=destRect->left;
          page++;
        } else {
          forceonespace=true;
        }
      }
      while (((unsigned char)*page)<32 && ((unsigned char)*page)>0) {
        page++;
      }
    }
  }

  (*ypos)-=destRect->top;
  RestoreDC(aContext->hdc, -1);
}

static void initHTMLEngine(void) {
  int fontcnt;
  for(fontcnt=0; fontcnt<64; fontcnt++) {
    Fonts[fontcnt] = CreateFont(
      FontSizes[(fontcnt&FONTMASK_SIZE)>>3], 0, 0, 0, (fontcnt&FONTMASK_BOLD)?FW_BOLD:FW_THIN,
      (fontcnt&FONTMASK_ITALIC)?TRUE:FALSE, (fontcnt&FONTMASK_UNDERLINE)?TRUE:FALSE, FALSE,
      DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS,
      PROOF_QUALITY, ((fontcnt&FONTMASK_FIXEDPITCH)?FIXED_PITCH:VARIABLE_PITCH) | FF_DONTCARE, (fontcnt&FONTMASK_FIXEDPITCH)?"Courier New":"Arial");
  }
}

static void termHTMLEngine(void) {
  int fontcnt;
  for(fontcnt=0; fontcnt<64; fontcnt++) {
    DeleteObject(Fonts[fontcnt]);
  }
}
EOF

  push(@initFunctions, "initHTMLEngine();");
  push(@termFunctions, "termHTMLEngine();");

}


sub genSystemIncludes {
# system
  if ($platformOS eq 'Win32') {
    include('<windows.h>');
    include('<windowsx.h>');
    include('<commdlg.h>');
    include('<stdlib.h>');

    # VC6.0 specific includes
    $ri .= "#ifdef _MSC_VER\n";
    include('<process.h>');
    $ri .= "#pragma comment(linker,\"/opt:nowin98 /MAP /MAPINFO:FIXUPS\")\n";
    $ri .= "#pragma warning(disable:4100)\n";
    $ri .= "#pragma warning(disable:4711)\n";
    $ri .= "#endif\n";
  }
  if ($platformOS eq 'Unix') {
    include('<unistd.h>');
    include('<stdlib.h>');
    include('<stdarg.h>');
    include('<pthread.h>');
  }
  include('<stdio.h>');
  include('<math.h>');
  include('<string.h>');

# graphics
  if ($platformGraphics eq 'GDI') {
    include('"resource.h"');
  }
  if ($platformGraphics eq 'X') {
    $ri .= "#include <X11/Xlib.h>\n";
    $ri .= "#include <X11/Xutil.h>\n";
  }
  if ($platformGraphics eq 'Xt') {
#    $ri .= "#include <X11/Xlib.h>\n";
#    $ri .= "#include <X11/Xutil.h>\n";
    $ri .= "#include <X11/Intrinsic.h>\n";
    $ri .= "#include <X11/StringDefs.h>\n";
    $ri .= "#include <X11/Xaw/Form.h>\n";
    $ri .= "#include <X11/Xaw/Command.h>\n";
  }

# video

# audio
}

sub genLogging {
  $rs .= readCompleteFile("log_functions.txt");
  push(@initFunctions, 'createLogfile("studiofactory.log");');
  push(@termFunctions, 'logprintf("closing studiofactory logfile.\n");');
}

sub genStdWindowProc {
  $rc .= <<EOF;
static void fillIdFromAreaList(ContextPtr aContext, int x, int y) {
  int len=GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALISTUSEDLEN);
  RECT *list=(RECT*)GetWindowLong(aContext->currentWindow, WNDEXTRA_AREALIST);

  if (NULL!=list) {
    int i;

    aContext->id=0;
    for(i=0;i<len;i++) {
      if (x>=list[i].left && x<=list[i].right &&
          y>=list[i].top && y<=list[i].bottom) {
        aContext->id=i+1;
      }
    }    
  }  
}

static void dispatchMouseButton(HWND hwnd, GuiEvent aGuiEvent, int x, int y) {
  EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);
  if (NULL != eventFunc) {
    Context myContext;

    myContext.currentWindow=hwnd;
    myContext.userData=(void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
    myContext.mouseClientX=x;
    myContext.mouseClientY=y;
    POINT p={x,y};
    ClientToScreen(hwnd, &p);
    myContext.mouseX=p.x;
    myContext.mouseY=p.y;
    myContext.guiEvent=aGuiEvent;
    fillIdFromAreaList(&myContext, x, y);
    eventFunc(&myContext);
  }
}

#ifndef WM_MOUSEWHEEL
#define WM_MOUSEWHEEL 0x020A
#endif

static int virtualMouseSpeed = 1;
static LRESULT CALLBACK stdWindowProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
  Context myContext;
  switch(uMsg) {
    case WM_CREATE:
      break;
    case WM_ACTIVATE: {
        EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);

        if (NULL != eventFunc) {
          myContext.currentWindow=hwnd;
          myContext.userData=(void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
          POINT p;
          GetCursorPos(&p);
          myContext.mouseX=p.x;
          myContext.mouseY=p.y;
          ScreenToClient(hwnd, &p);
          myContext.mouseClientX=p.x;
          myContext.mouseClientY=p.y;
          if (LOWORD(wParam)==WA_INACTIVE) {
            myContext.guiEvent=GUIEVENT_LOSTFOCUS;
          } else {
            myContext.guiEvent=GUIEVENT_GETFOCUS;
          }
          eventFunc(&myContext);
        }
      } break;
    case WM_SIZE: {
        EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);

        if (NULL != eventFunc) {
          myContext.currentWindow=hwnd;
          myContext.userData=(void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
          myContext.guiEvent=GUIEVENT_RESIZE;
          eventFunc(&myContext);
          InvalidateRect(hwnd, NULL, FALSE);
        }
      } break;
    case WM_DESTROYCLIPBOARD: {
        if (clipboardHandle) {
          GlobalFree(clipboardHandle);
          clipboardHandle=NULL;
        }
//        MessageBox(mainWindow, "DestroyClipboard", "DestroyClipboard", MB_OK);
      } break;
    case WM_COMMAND: {
        EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);

        if (NULL != eventFunc) {
          myContext.currentWindow=hwnd;
          myContext.userData=(void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
          switch(HIWORD(wParam)) {
          case 0: // Menu or button
            if (0==lParam) {
              POINT p;
              GetCursorPos(&p);
              myContext.mouseX=p.x;
              myContext.mouseY=p.y;
              ScreenToClient(hwnd, &p);
              myContext.mouseClientX=p.x;
              myContext.mouseClientY=p.y;
              // Menu
              myContext.guiEvent=GUIEVENT_MENU;
              myContext.id=LOWORD(wParam);
              eventFunc(&myContext);
            } else {
              // Control
    //            dispatchControlEvent(hwnd, (HWND)lParam, HIWORD(wParam));
            }
            break;
          case 1: // Accelarator
            break;
          default: // Control
    //          dispatchControlEvent(hwnd, (HWND)lParam, HIWORD(wParam));
            break;
          }
        }
      } break;
    case WM_MOUSEWHEEL: {
        SCROLLINFO myScrollInfo;
        myScrollInfo.cbSize=sizeof(myScrollInfo);
        myScrollInfo.fMask=SIF_POS | SIF_TRACKPOS | SIF_PAGE | SIF_RANGE;
        GetScrollInfo(hwnd, SB_VERT, &myScrollInfo);

//        guiScrollWindow(hwnd, -((short)HIWORD(wParam))/4);

        myScrollInfo.nPos -= ((short)HIWORD(wParam))/4;

        if (myScrollInfo.nPos<myScrollInfo.nMin) myScrollInfo.nPos=myScrollInfo.nMin;
        if (myScrollInfo.nPos>myScrollInfo.nMax) myScrollInfo.nPos=myScrollInfo.nMax;
        myScrollInfo.cbSize=sizeof(myScrollInfo);
        myScrollInfo.fMask=SIF_POS;
        SetScrollInfo(hwnd, SB_VERT, &myScrollInfo, TRUE);
        InvalidateRect(hwnd, NULL, FALSE);
      } break;
    case WM_VSCROLL: {
        SCROLLINFO myScrollInfo;
        myScrollInfo.cbSize=sizeof(myScrollInfo);
        myScrollInfo.fMask=SIF_POS | SIF_TRACKPOS | SIF_PAGE | SIF_RANGE;
        GetScrollInfo(hwnd, SB_VERT, &myScrollInfo);
      
        switch(LOWORD(wParam)) {
        case SB_PAGEUP:
          myScrollInfo.nPos-=myScrollInfo.nPage;
          break;
        case SB_LINEUP:
          myScrollInfo.nPos-=16;
          break;
        case SB_PAGEDOWN:
          myScrollInfo.nPos+=myScrollInfo.nPage;
          break;
        case SB_LINEDOWN:
          myScrollInfo.nPos+=16;
          break;
        case SB_THUMBTRACK:
          myScrollInfo.nPos=myScrollInfo.nTrackPos;
          break;
        default:
          break;
        }

        if (myScrollInfo.nPos<myScrollInfo.nMin) myScrollInfo.nPos=myScrollInfo.nMin;
        if (myScrollInfo.nPos>myScrollInfo.nMax) myScrollInfo.nPos=myScrollInfo.nMax;
        myScrollInfo.cbSize=sizeof(myScrollInfo);
        myScrollInfo.fMask=SIF_POS;
        SetScrollInfo(hwnd, SB_VERT, &myScrollInfo, TRUE);
        InvalidateRect(hwnd, NULL, FALSE);
      } break;
    case WM_PAINT: {
        PAINTSTRUCT ps;
        HDC hdc=BeginPaint(hwnd, &ps);
        EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);

        if (NULL != eventFunc) {
          GetClientRect(hwnd, &(myContext.guiClientRect));
          myContext.currentWindow = hwnd;
          myContext.userData = (void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
          myContext.hdc = hdc;
          POINT p;
          GetCursorPos(&p);
          myContext.mouseX=p.x;
          myContext.mouseY=p.y;
          ScreenToClient(hwnd, &p);
          myContext.mouseClientX=p.x;
          myContext.mouseClientY=p.y;
          myContext.guiEvent = GUIEVENT_REFRESH;
          eventFunc(&myContext);

          // Restore HDC to stock objects
          SelectObject(myContext.hdc, GetStockObject(SYSTEM_FONT));
          SelectObject(myContext.hdc, GetStockObject(BLACK_PEN));
          SelectObject(myContext.hdc, GetStockObject(BLACK_BRUSH));
        }

        EndPaint(hwnd, &ps);    
      } break;
    case WM_LBUTTONDOWN: {
        dispatchMouseButton(hwnd, GUIEVENT_MOUSEBUTTON1DOWN, GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
      } break;
    case WM_LBUTTONUP: {
        dispatchMouseButton(hwnd, GUIEVENT_MOUSEBUTTON1UP, GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
      } break;
    case WM_LBUTTONDBLCLK: {
        dispatchMouseButton(hwnd, GUIEVENT_MOUSEBUTTON1DCLK, GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
      } break;
    case WM_RBUTTONDOWN: {
        dispatchMouseButton(hwnd, GUIEVENT_MOUSEBUTTON2DOWN, GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
      } break;
    case WM_RBUTTONUP: {
        dispatchMouseButton(hwnd, GUIEVENT_MOUSEBUTTON2UP, GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
      } break;
    case WM_RBUTTONDBLCLK: {
        dispatchMouseButton(hwnd, GUIEVENT_MOUSEBUTTON2DCLK, GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
      } break;
    case WM_KEYDOWN: {
        POINT pt;
        switch(wParam) {
        case VK_NUMPAD4:
          GetCursorPos(&pt);
          pt.x -= virtualMouseSpeed/2+1;
          SetCursorPos(pt.x, pt.y); 
          break;
        case VK_NUMPAD6:
          GetCursorPos(&pt);
          pt.x += virtualMouseSpeed/2+1;
          SetCursorPos(pt.x, pt.y); 
          break;
        case VK_NUMPAD8:
          GetCursorPos(&pt);
          pt.y -= virtualMouseSpeed/2+1;
          SetCursorPos(pt.x, pt.y); 
          break;
        case VK_NUMPAD2:
          GetCursorPos(&pt);
          pt.y += virtualMouseSpeed/2+1;
          SetCursorPos(pt.x, pt.y); 
          break;
        default: {
            EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);

            if (NULL != eventFunc) {
              myContext.currentWindow = hwnd;
              myContext.userData = (void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
              POINT p;
              GetCursorPos(&p);
              myContext.mouseX=p.x;
              myContext.mouseY=p.y;
              ScreenToClient(hwnd, &p);
              myContext.mouseClientX=p.x;
              myContext.mouseClientY=p.y;
              myContext.id=wParam;
              myContext.guiEvent = GUIEVENT_KEYDOWN;
              eventFunc(&myContext);
            }
          } break;
        }
        if (virtualMouseSpeed<16) virtualMouseSpeed++;
      } break;
    case WM_KEYUP: {
/*        switch(wParam) {
        case VK_ESCAPE:
          break;
        default: {
            EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);
            if (NULL != eventFunc) {
              ContextDef myContext=mainContext;
              GuiEvent guiEvent;

              guiEvent.guiEventType=GUIEVENT_KEYUP;
              myContext.currentWindow=hwnd;
              myContext.userData=(void*)GetWindowLong(hwnd, WNDEXTRA_USER_DATA);
              myContext.key=wParam;
              eventFunc(&myContext, &guiEvent);
            }
          } break;
        }*/
        virtualMouseSpeed = 1;
      } break;
    case WM_CHAR: {
        EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);
        if (NULL != eventFunc) {
          myContext.currentWindow = hwnd;
          myContext.userData = (void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
          POINT p;
          GetCursorPos(&p);
          myContext.mouseX=p.x;
          myContext.mouseY=p.y;
          ScreenToClient(hwnd, &p);
          myContext.mouseClientX=p.x;
          myContext.mouseClientY=p.y;
          myContext.id=wParam;
          myContext.keyRepeatFlag=(lParam&0x40000000)?true:false;
          myContext.guiEvent = GUIEVENT_CHAR;
          eventFunc(&myContext);
        }
      } break;
    case WM_TIMER: {
        EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);

        if (NULL != eventFunc) {
          myContext.currentWindow = hwnd;
          myContext.userData = (void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
          POINT p;
          GetCursorPos(&p);
          myContext.mouseX=p.x;
          myContext.mouseY=p.y;
          ScreenToClient(hwnd, &p);
          myContext.mouseClientX=p.x;
          myContext.mouseClientY=p.y;
          myContext.guiEvent = GUIEVENT_TIMERTICK;
          myContext.hdc = GetDC(hwnd);
          eventFunc(&myContext);
          // Restore HDC to stock objects
          SelectObject(myContext.hdc, GetStockObject(SYSTEM_FONT));
          SelectObject(myContext.hdc, GetStockObject(BLACK_PEN));
          SelectObject(myContext.hdc, GetStockObject(BLACK_BRUSH));
          ReleaseDC(hwnd, myContext.hdc);
        }
      } break;
    case WM_CLOSE: {
        EventFunc eventFunc=(EventFunc)GetWindowLong(hwnd, WNDEXTRA_EVENTFUNC);

        if (NULL != eventFunc) {
          myContext.currentWindow = hwnd;
          myContext.userData = (void*)GetWindowLong(hwnd, WNDEXTRA_USERDATA);
          myContext.guiEvent = GUIEVENT_CLOSE;
          eventFunc(&myContext);
        }
      } break;
    case WM_DESTROY: {
        RECT *list=(RECT*)GetWindowLong(hwnd, WNDEXTRA_AREALIST);

        if (list) {
          free(list);
        }
      } break;
    default:
      return DefWindowProc(hwnd, uMsg, wParam, lParam);
  }
  refreshWindows();
  return 0;
}
EOF
}

my @wndExtraFields=qw(WNDEXTRA_EVENTFUNC WNDEXTRA_DOUBLEBUFFER WNDEXTRA_AREALIST WNDEXTRA_AREALISTUSEDLEN WNDEXTRA_AREALISTMAXLEN WNDEXTRA_USERDATA);
sub genWindowClass {
  $rv .= "static const char * const mainWindowClass=\"StudioFactoryMainClass\";\n";
  $rv .= "static ATOM mainClassAtom;\n";

  my $wndExtraSize = scalar(@wndExtraFields)*4;
  my $i;
  for($i=0;$i<scalar(@wndExtraFields);$i++) {
    $rv .= "static const int $wndExtraFields[$i] = " . ($i*4) . ";\n";
  }

  $rc .= <<EOF;
  WNDCLASSEX myClass;
  myClass.cbSize=sizeof(myClass);
  myClass.style=CS_DBLCLKS; /* Enable double click processing */
  myClass.lpfnWndProc=stdWindowProc;
  myClass.cbClsExtra=0;
  myClass.cbWndExtra=$wndExtraSize;
  myClass.hInstance=myInstance;
  myClass.hIcon=LoadIcon(myInstance, MAKEINTRESOURCE(IDI_MAINICON));
  myClass.hCursor=LoadCursor(NULL, IDC_ARROW);
  myClass.hbrBackground=NULL; // (HBRUSH)GetStockObject(GRAY_BRUSH);
  myClass.lpszMenuName=NULL;
  myClass.lpszClassName=mainWindowClass;
  myClass.hIconSm=NULL;
  mainClassAtom=RegisterClassEx(&myClass);
EOF
}

sub genLoadWindow {
  my $name = shift;
  my $saveName = shift;
  my $fixedSize = shift;

  if (defined($fixedSize)) {
  push(@synLoad, <<EOF);
  if ((section==SECTION_WINDOWS) && strcmp(loadInfo->headerTitle, \"$saveName\")==0) {
    int v,x,y;

    readInteger(loadInfo, &v,  true);
    readInteger(loadInfo, &x,  true);
    readInteger(loadInfo, &y,  true);
    SetWindowPos($name, NULL, x, y, 0, 0, SWP_NOZORDER | SWP_NOSIZE);
    ShowWindow($name, (v)?SW_SHOW:SW_HIDE);
  }
EOF
  } else {
  push(@synLoad, <<EOF);
  if ((section==SECTION_WINDOWS) && strcmp(loadInfo->headerTitle, \"$saveName\")==0) {
    int v,x,y,xs,ys;

    readInteger(loadInfo, &v,  true);
    readInteger(loadInfo, &x,  true);
    readInteger(loadInfo, &y,  true);
    readInteger(loadInfo, &xs, true);
    readInteger(loadInfo, &ys, true);
    SetWindowPos($name, NULL, x, y, xs, ys, SWP_NOZORDER);
    ShowWindow($name, (v)?SW_SHOW:SW_HIDE);
  }
EOF
  }
}

my $generatedLoadAndSaveWindowStf=0;
sub genWindow {
  my $name = shift;
  my $saveName = shift;
  my $title = shift;
  my $handlerFunction = shift;
  my $windowXSize = shift;
  my $windowYSize = shift;
  my $vscroll = shift;
  my $fixedSize = shift;

  genLoadWindow($name, $saveName, $fixedSize);

  if ($generatedLoadAndSaveWindowStf == 0) {
  $rc .= <<EOF;
void loadWindowStf(LoadInfo *loadInfo, const char *aTitle, HWND windowHandle, bool fixedSize) {
  if (strcmp(aTitle, loadInfo->headerTitle)==0) {
    if (fixedSize) {
      int v,x,y;

      readStfInteger(loadInfo, &v);
      readStfInteger(loadInfo, &x);
      readStfInteger(loadInfo, &y);
      SetWindowPos(windowHandle, NULL, x, y, 0, 0, SWP_NOZORDER | SWP_NOSIZE);
      ShowWindow(windowHandle, (v)?SW_SHOW:SW_HIDE);
    } else {
      int v,x,y,xs,ys;

      readStfInteger(loadInfo, &v);
      readStfInteger(loadInfo, &x);
      readStfInteger(loadInfo, &y);
      readStfInteger(loadInfo, &xs);
      readStfInteger(loadInfo, &ys);
      SetWindowPos(windowHandle, NULL, x, y, xs, ys, SWP_NOZORDER);
      ShowWindow(windowHandle, (v)?SW_SHOW:SW_HIDE);
    }
  }
}

void saveWindowStf(LoadInfo *loadInfo, const char *aTitle, HWND windowHandle, bool fixedSize) {
  WINDOWPLACEMENT winpos;

  winpos.length = sizeof(WINDOWPLACEMENT);
  GetWindowPlacement(windowHandle, &winpos);
  if (fixedSize) {
    storeStfBegin(loadInfo, aTitle);
    storeStfInteger(loadInfo, IsWindowVisible(windowHandle)?1:0);
    storeStfInteger(loadInfo, winpos.rcNormalPosition.left);
    storeStfInteger(loadInfo, winpos.rcNormalPosition.top);
    storeStfEnd(loadInfo);
  } else {
    storeStfBegin(loadInfo, aTitle);
    storeStfInteger(loadInfo, IsWindowVisible(windowHandle)?1:0);
    storeStfInteger(loadInfo, winpos.rcNormalPosition.left);
    storeStfInteger(loadInfo, winpos.rcNormalPosition.top);
    storeStfInteger(loadInfo, winpos.rcNormalPosition.right-winpos.rcNormalPosition.left);
    storeStfInteger(loadInfo, winpos.rcNormalPosition.bottom-winpos.rcNormalPosition.top);
    storeStfEnd(loadInfo);
  }
}
EOF
    $generatedLoadAndSaveWindowStf=1;
  }

#  push(@stfSave, "sprintf(myTempBuf, \"windowpos\");");

  # backwards compatibility (versions before 1.13 don't have resizable scopewindow)
  if ($saveName eq 'SCOPEWINDOW2') {
    genLoadWindow($name, 'SCOPEWINDOW', 1);
  }

  if (defined($fixedSize)) {
    push(@stfLoad, "loadWindowStf(loadInfo, \"$saveName\", $name, true);");
    push(@stfSave, "saveWindowStf(loadInfo, \"$saveName\", $name, true);");
    $fixedSize='';
  } else {
    push(@stfLoad, "loadWindowStf(loadInfo, \"$saveName\", $name, false);");
    push(@stfSave, "saveWindowStf(loadInfo, \"$saveName\", $name, false);");
    $fixedSize=' | WS_THICKFRAME | WS_MINIMIZEBOX | WS_MAXIMIZEBOX';
  }
  if ($vscroll == 1) {
    $vscroll = ' | WS_VSCROLL';
  } else {
    $vscroll = '';
  }
  if (!defined($windowXSize)) {
    $windowXSize=400;
  } else {
    $windowXSize.='+2*GetSystemMetrics(SM_CXFIXEDFRAME)';
  }
  if (!defined($windowYSize)) {
    $windowYSize=400;
  } else {
    $windowYSize.='+GetSystemMetrics(SM_CYSMCAPTION) + 2*GetSystemMetrics(SM_CYFIXEDFRAME)'
  }

  rv("bool ${name}Refresh;");
  push(@refreshFunctions, "if (${name}Refresh) { guiRefreshWindow(${name}, NULL); ${name}Refresh=false; }");

  if ($platformGraphics eq 'GDI') {
    rv("HWND $name;");
    if ($name eq 'mainWindow') {
      push(@initWindows, "$name = CreateWindowEx(0, mainWindowClass, $title, WS_SYSMENU | WS_CLIPCHILDREN | WS_OVERLAPPED | WS_VISIBLE$fixedSize$vscroll, CW_USEDEFAULT, CW_USEDEFAULT, $windowXSize, $windowYSize, NULL, NULL, myInstance, NULL);\n  SetWindowLong($name, WNDEXTRA_EVENTFUNC, (LONG)$handlerFunction);\n  UpdateWindow($name);");
    } else {
      push(@initWindows, "$name = CreateWindowEx(WS_EX_TOOLWINDOW, mainWindowClass, $title, WS_SYSMENU | WS_CLIPCHILDREN | WS_OVERLAPPED$fixedSize$vscroll, CW_USEDEFAULT, CW_USEDEFAULT, $windowXSize, $windowYSize, mainWindow, NULL, myInstance, NULL);\n  SetWindowLong($name, WNDEXTRA_EVENTFUNC, (LONG)$handlerFunction);");
    }
    push(@termWindows, "DestroyWindow($name);");
  }
  if ($platformGraphics eq 'Xt') {
    rv("Widget $name;");
#     push(@initWindows, "$name = XtCreateWidget(\"$name\", , mainWindow,,);");
    push(@termWindows, "XtDestroyWidget($name);");
  }
}

sub genMainWindow {
  genWindow('mainWindow', 'MAINWINDOW', "\"StudioFactory $studioFactoryVersion - http://www.syntiac.com\"", 'mainWindowHandler', 400, 400, 0, undef);
  push(@termWindows, "SetMenu(mainWindow, NULL);");
  push(@initWindows, "SetTimer(mainWindow, 0, 100, NULL);");
}

sub genMainWindowProcessing {
$rc .= <<EOF;
static void mainWindowHandler(ContextPtr aContext) {
  switch(aContext->guiEvent) {
  case GUIEVENT_GETFOCUS:
  case GUIEVENT_LOSTFOCUS:
    if (currentProject>=0 && currentPatch>=0) {
      processSynFactory(aContext, &(projects[currentProject].patches[currentPatch]));
    } else {
      processTimeline(aContext);
    }
    break;
  case GUIEVENT_REFRESH:
    guiClearAreaList(aContext);
//    lastMainWindowArea=0;
//    processTimeline(aContext);
    if (currentProject>=0 && currentPatch>=0) {
      processSynFactory(aContext, &(projects[currentProject].patches[currentPatch]));
    } else {
      processTimeline(aContext);
    }
    break;
  case GUIEVENT_MENU:
    switch(aContext->id) {
    case MENU_NEW_PROJECT:
      createProject(newProjectName);
      break;
    case MENU_NEW_SYNFACTORY:
      createSynfactoryPatch();
      break;
    case MENU_NEW_TRACK:
      createTrack();
      break;
    case MENU_OPEN_FILE:
      openFileSelect(aContext, false);
      break;
    case MENU_ADD_FILE:
      openFileSelect(aContext, true);
      break;
    case MENU_CLOSE:
      // Implement !!!
      break;
    case MENU_CLOSE_PROJECT:
      // !!! Check for modified and give warning!
      if (currentProject>=0) {
        deleteProject(currentProject);
      }
      break;
    case MENU_SAVE:
      break;
    case MENU_SAVE_AS:
      saveFileSelect(aContext, false);
      break;
    case MENU_EXIT:
      quitFlag=true;
      break;
    case MENU_CUT:
      saveStfToClipboard();
      deleteSelected();
      break;
    case MENU_COPY:
      saveStfToClipboard();
      break;
    case MENU_PASTE:
      loadFromClipboard();
      break;
    case MENU_DELETE:
      deleteSelected();
      break;
    case MENU_CLONE:
      clone();
      break;
    case MENU_TOGGLELABEL:
      toggleLabel();
      break;
    case MENU_CABLECOLOR_0:
    case MENU_CABLECOLOR_1:
    case MENU_CABLECOLOR_2:
    case MENU_CABLECOLOR_3:
    case MENU_CABLECOLOR_4:
    case MENU_CABLECOLOR_5:
    case MENU_CABLECOLOR_6:
    case MENU_CABLECOLOR_7:
      setCurrentCableColor(aContext->id-MENU_CABLECOLOR_0);
      break;
    case MENU_PROJECTBROWSER:
      guiShowWindow(projectBrowserWindow);
      break;
    case MENU_COLORSELECTOR:
      guiShowWindow(colorSelectorWindow);
      break;
    case MENU_TRANSPORT:
      guiShowWindow(transportWindow);
      break;
    case MENU_OUTPUTSCOPE:
      guiShowWindow(scopeWindow);
      break;
    case MENU_AUDIOMIXER:
      guiShowWindow(audioMixerWindow);
      break;
    case MENU_MIDIACTIVITY:
      guiShowWindow(midiActivityMonitorWindow);
      break;
    case MENU_MIDISCOPE:
      guiShowWindow(midiScopeWindow);
      break;
    case MENU_REWIND:
      rewindPressed=2;
      resetPlayback();
      transportWindowRefresh=true;
      break;
    case MENU_STOP:
      setPlayMode(PLAYMODE_STOP);
      break;
    case MENU_PLAY:
      setPlayMode(PLAYMODE_PLAY);
      break;
    case MENU_RECORD:
      if (getRecordMode()!=RECORDMODE_STOP) {
        setRecordMode(RECORDMODE_STOP);
      } else {
        setRecordMode(RECORDMODE_RECORD);
      }
      break;
    case MENU_GLOBAL_SETTINGS:
      guiShowWindow(configWindow);
      break;
    case MENU_PATCH_SETTINGS:
      guiShowWindow(patchConfigWindow);
      break;
    case MENU_MENUHELP:
      displayHelp("Menu help", helpTextMainMenu);
      break;
    case MENU_ABOUT:
      displayHelp("About StudioFactory", aboutHelpText);
      break;
    default:
      if (currentProject>=0 && currentPatch>=0) {
        processSynFactory(aContext, &(projects[currentProject].patches[currentPatch]));
      } else {
        processTimeline(aContext);
      }
      break;
    }
    break;
  case GUIEVENT_MOUSEBUTTON1DOWN:
  case GUIEVENT_MOUSEBUTTON1UP:
  case GUIEVENT_MOUSEBUTTON1DCLK:
  case GUIEVENT_MOUSEBUTTON2DOWN:
  case GUIEVENT_MOUSEBUTTON2UP:
  case GUIEVENT_MOUSEBUTTON2DCLK:
  case GUIEVENT_TIMERTICK:
  case GUIEVENT_CHAR:
    switch(aContext->id) {
    case 3:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        saveStfToClipboard();
      }
      break;
    case 4:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        clone();
      }
      break;
    case 22:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        loadFromClipboard();
      }
      break;
    case 24:
      // Ignore repeated events when key is held down
      if (!aContext->keyRepeatFlag) {
        saveStfToClipboard();
        deleteSelected();
      }
      break;
    default:
      if (currentProject>=0 && currentPatch>=0) {
        processSynFactory(aContext, &(projects[currentProject].patches[currentPatch]));
      } else {
        processTimeline(aContext);
      }
    }
//    processTimeline(aContext);

    break;
  case GUIEVENT_KEYDOWN: {
      switch(aContext->id) {
      case VK_F1:
        if (currentProject>=0 && currentPatch>=0) {
          processSynFactory(aContext, &(projects[currentProject].patches[currentPatch]));
        } else {
          processTimeline(aContext);
        }
        break;
      case VK_F2:
        guiShowWindow(projectBrowserWindow);      
        break;
      default:
        if (currentProject>=0 && currentPatch>=0) {
          processSynFactory(aContext, &(projects[currentProject].patches[currentPatch]));
        } else {
          processTimeline(aContext);
        }
        break;
      }
    } break;
  case GUIEVENT_CLOSE:
    quitFlag=true;
    break;
  default:
    break;
  }
}
EOF
}

sub genMainLoopGdi {
  $rv .= "static int quitFlag=0;\n";

  $rc .= "  while (!quitFlag) {\n";
  $rc .= "    MSG myMsg;\n";
  $rc .= "    switch (GetMessage(&myMsg,NULL,0,0)) {\n";
  $rc .= "    case FALSE:\n";
  $rc .= "      quitFlag=1;\n";
  $rc .= "      break;\n";
  $rc .= "    case TRUE:\n";
# //      if (0==IsDialogMessage(configPanelWindow, &myMsg)) {
#  $rc .=       if (!TranslateMDISysAccel((HWND)GetWindowLong(context->mainWindow, WNDEXTRA_MDI_CLIENT), &myMsg)) {
  $rc .= "      TranslateMessage(&myMsg);\n";
  $rc .= "      DispatchMessage(&myMsg);\n";
#  $rc .= "        }\n";
  $rc .= "      break;\n";
  $rc .= "    default:\n";
  $rc .= "      quitFlag=1;\n";
  $rc .= "      break;\n";
  $rc .= "    }\n";
  $rc .= "  }\n";
}

sub genMainLoopXt {
}

sub genMainLoop {
  if ($platformGraphics eq 'GDI') {
    genMainLoopGdi()
  }
  if ($platformGraphics eq 'Xt') {
    genMainLoopXt()
  }
}

sub genRefreshWindowFunction {
  $rs .= "static void refreshWindows(void) {\n";
  for(my $i=0;$i<scalar(@refreshFunctions);$i++) {
    $rs .= ("  " . $refreshFunctions[$i] . "\n");
  }
  $rs .= "}\n";
}

sub genInitFunctions {
  for(my $i=0;$i<scalar(@initFunctions);$i++) {
    $rc .= ("  " . $initFunctions[$i] . "\n");
  }
}

sub genTermFunctions {
  for(my $i=scalar(@termFunctions);$i>0;$i--) {
    $rc .= ("  " . $termFunctions[$i-1] . "\n");
  }
}

sub genInitWindows {
  for(my $i=0;$i<scalar(@initWindows);$i++) {
    $rc .= ("  " . $initWindows[$i] . "\n");
  }
}

sub genTermWindows {
  for(my $i=scalar(@termWindows);$i>0;$i--) {
    $rc .= ("  " . $termWindows[$i-1] . "\n");
  }
}

sub genMain {
  if ($platformOS eq 'Win32') {
    $ri .= "static HINSTANCE myInstance;\n";
    $rc .= "int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, PSTR szCmdLine, int iCmdShow) {\n";
    $rc .= "  int i;\n\n";
    $rc .= "  myInstance = hInstance;\n";
    genWindowClass();
  }
  if ($platformOS eq 'Unix') {
    $rc .= "int main(int argc, char **argv) {\n";
    $rc .= "  int i;\n\n";
  }
  if ($platformGraphics eq 'Xt') {
    rv('Widget applicationShell;');
    rv('XtAppContext application;');
    $rc .= "applicationShell = XtAppInitialize(&application, \"StudioFactory\", (XrmOptionDescList) NULL, 0, &argc, argv, (String *) NULL, (ArgList) NULL, 0);\n";
  }
  genInitFunctions();
  genInitWindows();
  $rc .= <<EOF;
  setLanguageCode("EN");
  if (loadFromFilename(NULL, "autoload.stf", false)!=LOADERROR_OK) {
    if (loadFromFilename(NULL, "autoload.syf", false)!=LOADERROR_OK) {
      if (loadFromFilename(NULL, "autoload.syn", false)!=LOADERROR_OK) {
        displayHelp("About StudioFactory", aboutHelpText);
        setAudioOutDevice(2);
      }
    }
  }
EOF
  genMainLoop();
  $rc .= "  saveStfToFilename(NULL, \"autoload.stf\", true, saveAllProjects);\n";
  $rc .= "  setPlayMode(PLAYMODE_STOP);\n"; # !!!
  $rc .= "  while(maxProjects>0) {\n";
  $rc .= "    deleteProject(0);\n";
  $rc .= "  }\n";
  genTermWindows();
  genTermFunctions();
  $rc .= "  return 0;\n}\n";
}

sub genHelpWindow() {
  $rv .= "static const char * const StudioFactoryHelp=\"StudioFactory Help\";\n";
  $rv .= "static const char *currentHelpPage=aboutHelpText;\n";

$rd .= <<EOF;
static const char * const aboutHelpText=
  "<P><FONT SIZE=+2><U>StudioFactory</U></FONT><BR>"
  "The desktop Audio/Video studio</P><HR>"
  "<P><B>Version: </B>$studioFactoryVersion<BR>"
  "<P><B>Website: </B>http://www.syntiac.com/studiofactory.html<BR>"
  "<B>Feedback: </B>studio\@syntiac.com</P>"
  "<P><B>Software design and programming: </B>Peter Wendrich (pwsoft\@syntiac.com)<BR>"
  "<B>Thanx to: </B>Dave, Antonio, Patrick and Raymond for patches and ideas</P><HR>"
  "<P><B>license: </B>This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.</P>"
  "<P><B>Disclamer: </B>This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.</P>";
EOF

$rc .= <<EOF;
static bool displayHelp(const char *title, const char *helppage) {
  if (NULL!=onlineHelpWindow) {
    guiSetWindowTitle(onlineHelpWindow, (NULL!=title)?title:StudioFactoryHelp); // !!! not platofrm independed
    currentHelpPage=helppage;
    guiShowWindow(onlineHelpWindow);
    guiVScrollWindowTo(onlineHelpWindow, 0);
    return true;
  }

  return false;
}

static void onlineHelpHandler(ContextPtr aContext) {
  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      int scrollPos=guiGetVScrollPos(aContext->currentWindow);
      int curY=-scrollPos;

      DrawEdge(aContext->hdc, &(aContext->guiClientRect), EDGE_SUNKEN, BF_MIDDLE);
      aContext->guiClientRect.left+=4;
      aContext->guiClientRect.right-=4;
      renderHTMLPage(aContext, &(aContext->guiClientRect), &curY, currentHelpPage);
      guiVScrollRange(aContext->currentWindow, 0, curY+scrollPos, aContext->guiClientRect.bottom-16);

    } break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(aContext->currentWindow);
    break;
  default:
    break;
  }
}
EOF

  genWindow('onlineHelpWindow', 'HELPWINDOW', 'StudioFactoryHelp', 'onlineHelpHandler', undef, undef, 1, undef);
}

##########################################################################
#
# Transport Control
#
##########################################################################
sub genTransportControl() {
  genSection('Transport Control');
  
$rc .= <<EOF;

static int rewindPressed=0;
static const int vuLength=14;
static const int VUtbl[vuLength]={
   4000, 6000, 8000,10000,12000,14000,16000,
  18000,20000,22000,24000,26000,28000,30000};
static const int vuColors[vuLength][2]={
  {0x004400, 0x44CC44}, {0x004400, 0x44CC44}, {0x004400, 0x44CC44}, {0x004400, 0x44CC44}, {0x004400, 0x44CC44},
  {0x004400, 0x44CC44}, {0x004400, 0x44CC44}, {0x004400, 0x44CC44}, {0x004400, 0x44CC44}, {0x004400, 0x44CC44}, {0x004400, 0x44CC44},
  {0x000044, 0x4444CC}, {0x000044, 0x4444CC}, {0x000044, 0x4444CC},
};

// Draw the 4 transport icons: STOP PAUSE RECORD PLAY
static void DrawTransportIcons(ContextPtr aContext, int xoffs, int yoffs) {
  RECT myRect;
  int pressed;
  int grayed=true;

  if (currentProject>=0) {
    grayed=false;
  }

  // Rewind icon
  {
    pressed=(rewindPressed>0)?1:0;//((CaptureMode==CAPTURE_STOP)?1:0);
    myRect.left=xoffs+2;
    myRect.right=xoffs+30;
    myRect.top=yoffs+2;
    myRect.bottom=yoffs+30;
    DrawEdge(aContext->hdc, &myRect, (rewindPressed>0)?EDGE_SUNKEN:EDGE_RAISED, BF_MIDDLE | BF_RECT);
    guiAddArea(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);
    guiSelectPen1Color(aContext, (grayed)?0x888888:0xBB0000);
    guiSelectFillColor(aContext, (grayed)?0x888888:0xBB0000);
    POINT pt[3]={{pressed+myRect.right-6, pressed+yoffs+8}, {pressed+myRect.left+6, pressed+yoffs+16}, {pressed+myRect.right-6, pressed+yoffs+24}};
    Polygon(aContext->hdc, pt, 3);
    myRect.left+=pressed+4;
    myRect.right+=pressed-22;
    myRect.top+=pressed+6;
    myRect.bottom+=pressed-6;
    guiDrawRect(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);
  }

  // Draw stop icon
  {
    pressed=(getPlayMode()==PLAYMODE_STOP)?1:0;
    myRect.left=32+xoffs+2;
    myRect.right=32+xoffs+30;
    myRect.top=yoffs+2;
    myRect.bottom=yoffs+30;
    DrawEdge(aContext->hdc, &myRect, (getPlayMode()==PLAYMODE_STOP)?EDGE_SUNKEN:EDGE_RAISED, BF_MIDDLE | BF_RECT);
    guiAddArea(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);
    myRect.left+=pressed+6;
    myRect.right+=pressed-6;
    myRect.top+=pressed+6;
    myRect.bottom+=pressed-6;
    guiSelectPen1Color(aContext, (grayed)?0x888888:0xBB0000);
    guiSelectFillColor(aContext, (grayed)?0x888888:0xBB0000);
    guiDrawRect(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);
  }

  // Draw play icon
  {
    pressed=(getPlayMode()==PLAYMODE_PLAY)?1:0;
    myRect.left=64+xoffs+2;
    myRect.right=64+xoffs+30;
    myRect.top=yoffs+2;
    myRect.bottom=yoffs+30;
    DrawEdge(aContext->hdc, &myRect, (getPlayMode()==PLAYMODE_PLAY)?EDGE_SUNKEN:EDGE_RAISED, BF_MIDDLE | BF_RECT);
    guiAddArea(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);
    guiSelectPen1Color(aContext, (grayed)?0x888888:0x00AA00);
    guiSelectFillColor(aContext, (grayed)?0x888888:0x00AA00);
    POINT pt[3]={{pressed+myRect.left+6, pressed+yoffs+8}, {pressed+myRect.right-6, pressed+yoffs+16}, {pressed+myRect.left+6, pressed+yoffs+24}};
    Polygon(aContext->hdc, pt, 3);
  }

#if 1
  // Draw record icon
  pressed=(getRecordMode()==RECORDMODE_RECORD)?1:0;
  myRect.left=96+xoffs+2;
  myRect.right=96+xoffs+30;
  myRect.top=yoffs+2;
  myRect.bottom=yoffs+30;
  DrawEdge(aContext->hdc, &myRect, (pressed)?EDGE_SUNKEN:EDGE_RAISED, BF_MIDDLE | BF_RECT);
  guiAddArea(aContext, myRect.left, myRect.top, myRect.right, myRect.bottom);
  guiSelectPen1Color(aContext, 0x0000BB);
  guiSelectFillColor(aContext, 0x0000BB);
  Ellipse(aContext->hdc, pressed+96+xoffs+8, pressed+yoffs+8, pressed+96+xoffs+24, pressed+yoffs+24);
#endif
}

static void drawTransportVU(ContextPtr aContext) {
  // VU-meter
  guiSelectPen1Color(aContext, -1);
  for(int i=0; i<vuLength; i++) {
    guiSelectFillColor(aContext, vuColors[i][VU_Left>VUtbl[i]]);
    guiDrawRect(aContext, 4, 57-i*4, 14, 60-i*4);
    guiSelectFillColor(aContext, vuColors[i][VU_Right>VUtbl[i]]);
    guiDrawRect(aContext, 16, 57-i*4, 26, 60-i*4);
  }
}

static void drawTransportWindow(ContextPtr aContext) {
  DrawEdge(aContext->hdc, &(aContext->guiClientRect), EDGE_SUNKEN, BF_MIDDLE);

  drawTransportVU(aContext);

  // transport ICONS
  DrawTransportIcons(aContext, 32, 0);

  // Pattern position
//  SelectObject(aContext->hdc, GetStockObject(BLACK_PEN));
  guiSelectPen1Color(aContext, 0);
  guiSelectPotFont(aContext);
//		  SetBkMode(context->hdc, OPAQUE);
	SetBkMode(aContext->hdc, TRANSPARENT);
  SetTextAlign(aContext->hdc, TA_TOP | TA_LEFT);
//      SelectObject(context->hdc, GetStockObject(SYSTEM_FIXED_FONT));

  char tmpstr[256];
//  sprintf(tmpstr, " SongPos: %d  Pattern: %d  Row: %d", copySongPos, songOrders[copySongPos], copyRow);
//  TextOut(aContext->hdc, 32, 32, tmpstr, strlen(tmpstr));
  sprintf(tmpstr, " Tempo: %d / %d  Time: %02d:%02d:%02d", copyTempo, copyTicks, copyPlayTime/3600, (copyPlayTime/60)%60, copyPlayTime%60);
  TextOut(aContext->hdc, 32, 48, tmpstr, strlen(tmpstr));
}

static void transportHandler(ContextPtr aContext) {
  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH:
    guiClearAreaList(aContext);
    drawTransportWindow(aContext);
    break;
  case GUIEVENT_MOUSEBUTTON1DOWN:
    switch(aContext->id) {
    case 1:
      rewindPressed=2;
      resetPlayback();
      transportWindowRefresh=true;
      break;
    case 2:
      setPlayMode(PLAYMODE_STOP);
      break;
    case 3:
      setPlayMode(PLAYMODE_PLAY);
      break;
    case 4:
      if (getRecordMode()!=RECORDMODE_STOP) {
        setRecordMode(RECORDMODE_STOP);
      } else {
        setRecordMode(RECORDMODE_RECORD);
      }
      break;
    default:
      break;
    }
    break;
  case GUIEVENT_TIMERTICK:
//    VU_Left = InterlockedExchange(&VU_newLeft, -1);
//    VU_Right = InterlockedExchange(&VU_newRight, -1);

    if (VU_Left!=VU_newLeft || VU_Right!=VU_newRight) {
      VU_Left=VU_newLeft;
      VU_Right=VU_newRight;
      drawTransportVU(aContext);

    }

    flushCaptureFile();

    if (rewindPressed>0) {
      rewindPressed--;
      if (rewindPressed==0) {
        transportWindowRefresh=true;
      }
    }
    
//    if (copySongPos!=curSongPos || copyRow!=curRow || (curPlayTime/44100)!=copyPlayTime) {
//      copyPlayTime=curPlayTime/44100;
//      copySongPos=curSongPos;
//      copyRow=curRow;
//      copyTempo=curTempo;
//      copyTicks=curTicks;

//      RECT myRect={32,32,224,64};
//      guiRefreshWindow(aContext->currentWindow, &myRect);
//    }
    break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(aContext->currentWindow);
    break;
  default:
    break;
  }
}
EOF
  $rv .= "static int VU_Left=0;\nstatic int VU_Right=0;\n";
  $rv .= "static int VU_newLeft=0;\nstatic int VU_newRight=0;\n";
  $rv .= "static int copyPlayTime;\nstatic int copySongPos;\nstatic int copyRow;\nstatic int copyTempo;\nstatic int copyTicks;\n";

  genWindow('transportWindow', 'TRANSPORTWINDOW', '"Transport Control"', 'transportHandler', 256, 64, 0, 1);
  push(@initWindows, "SetTimer(transportWindow, 0, 100, NULL);");
}

##########################################################################
#
# Audio Output Scope
#
##########################################################################
sub genAudioOutputScope() {
  genSection('Audio Output Scope');

  $rv .= "static const int SCOPEBUFSIZE=1024;\n";
  $rv .= "static bool scopeVisible=false;\n";
  $rv .= "static int scopeMode=0;\n";
  $rv .= "static signed short DSP_ScopeL[SCOPEBUFSIZE];     // Scope buffer for left channel\n";
  $rv .= "static signed short DSP_ScopeR[SCOPEBUFSIZE];     // Scope buffer for right channel\n";
  $rv .= "static volatile unsigned int DSP_ScopePosition=0; // Current position within buffers\n";

$rc .= <<EOF;
static void drawScopeTopBottom(ContextPtr aContext) {
  int cnt;
  int yRange;

  RECT clientRect;

  GetClientRect(aContext->currentWindow, &clientRect);

//  clientRect=myRect;
  guiSelectPen1Color(aContext, 0x000000);
  guiSelectFillColor(aContext, 0x000000);
  guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom);
//  FillRect(aContext->hdc, &clientRect, (HBRUSH)GetStockObject(BLACK_BRUSH));
  yRange=((clientRect.bottom-16)/4);

  guiSelectPen1Color(aContext, 0xFFFFFF);
  MoveToEx(aContext->hdc, 0, yRange-(DSP_ScopeL[0]*yRange)/32768, NULL);
  for(cnt=1; cnt<SCOPEBUFSIZE; cnt++) {
    LineTo(aContext->hdc, (cnt*clientRect.right)/SCOPEBUFSIZE, yRange-(DSP_ScopeL[cnt]*yRange)/32768);
  }

  MoveToEx(aContext->hdc, 0, 3*yRange+16-(DSP_ScopeR[0]*yRange)/32768, NULL);
  for(cnt=1; cnt<SCOPEBUFSIZE; cnt++) {
    LineTo(aContext->hdc, (cnt*clientRect.right)/SCOPEBUFSIZE, 3*yRange+16-(DSP_ScopeR[cnt]*yRange)/32768);
  }

  // Draw base lines
  guiSelectPen1Color(aContext, 0x00BB00);
  MoveToEx(aContext->hdc, 0,   yRange, NULL);
  LineTo(aContext->hdc, clientRect.right,   yRange);
  MoveToEx(aContext->hdc, 0, 3*yRange+16, NULL);
  LineTo(aContext->hdc, clientRect.right, 3*yRange+16);

  guiSelectTitleFont(aContext);
  SetBkMode(aContext->hdc, TRANSPARENT);
  SetTextAlign(aContext->hdc, TA_BOTTOM | TA_LEFT);
  guiSelectPen1Color(aContext, 0x00BB00);
  TextOut(aContext->hdc, 0,   yRange, "Left", 4);
  TextOut(aContext->hdc, 0, 3*yRange+16, "Right", 5);

  DSP_ScopePosition = 0;
}


static void drawScopeLeftRight(ContextPtr aContext) {
  int cnt;
  int yRange;
  int xRange;

  RECT clientRect;

  GetClientRect(aContext->currentWindow, &clientRect);

//  clientRect=myRect;
  guiSelectPen1Color(aContext, 0x000000);
  guiSelectFillColor(aContext, 0x000000);
  guiDrawRect(aContext, aContext->guiClientRect.left, aContext->guiClientRect.top, aContext->guiClientRect.right, aContext->guiClientRect.bottom);
  yRange=(clientRect.bottom/2);
  xRange=((clientRect.right-16)/2);

  guiSelectPen1Color(aContext, 0xFFFFFF);
  MoveToEx(aContext->hdc, 0, yRange-(DSP_ScopeL[0]*yRange)/32768, NULL);
  for(cnt=1; cnt<SCOPEBUFSIZE; cnt++) {
    LineTo(aContext->hdc, (cnt*xRange)/SCOPEBUFSIZE, yRange-(DSP_ScopeL[cnt]*yRange)/32768);
  }

  MoveToEx(aContext->hdc, xRange+16, yRange-(DSP_ScopeR[0]*yRange)/32768, NULL);
  for(cnt=1; cnt<SCOPEBUFSIZE; cnt++) {
    LineTo(aContext->hdc, (cnt*xRange)/SCOPEBUFSIZE+xRange+16, yRange-(DSP_ScopeR[cnt]*yRange)/32768);
  }

  // Draw base lines
  guiSelectPen1Color(aContext, 0x00BB00);
  MoveToEx(aContext->hdc, 0,   yRange, NULL);
  LineTo(aContext->hdc, xRange,   yRange);
  MoveToEx(aContext->hdc, xRange+16,   yRange, NULL);
  LineTo(aContext->hdc, clientRect.right,   yRange);

//  guiSelectPen1Color(aContext, 7);
  guiSelectTitleFont(aContext);
  SetBkMode(aContext->hdc, TRANSPARENT);
  SetTextAlign(aContext->hdc, TA_BOTTOM | TA_LEFT);
  guiSelectPen1Color(aContext, 0x00BB00);
//  SetTextColor(aContext->hdc, PALETTERGB(128,255,128));
  TextOut(aContext->hdc, 0,   yRange, "Left", 4);
  TextOut(aContext->hdc, xRange+16,   yRange, "Right", 5);

  DSP_ScopePosition = 0;
}
EOF

$rc .= <<EOF;
static void outputScopeHandler(ContextPtr aContext) {
  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      switch(scopeMode) {
      case 0:
        drawScopeTopBottom(aContext);
        break;
      case 1:
        drawScopeLeftRight(aContext);
        break;
      default:
        scopeMode=0;
        drawScopeTopBottom(aContext);
        break;
      }
    } break;
//  case GUIEVENT_MOUSEBUTTON1DOWN:
  case GUIEVENT_MOUSEBUTTON2DOWN:
    scopeMode=1-scopeMode;
    scopeWindowRefresh=true;
    break;
  case GUIEVENT_TIMERTICK:
    if (DSP_ScopePosition==4*SCOPEBUFSIZE) {
      scopeWindowRefresh=true;
    }
    break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(scopeWindow);
    break;
  default:
    break;
  }
}
EOF

  genWindow('scopeWindow', 'SCOPEWINDOW2', '"Audio Output Scope"', 'outputScopeHandler', undef, undef, 0, undef);
  push(@initWindows, "SetTimer(scopeWindow, 0, 100, NULL);");
  push(@stfSave, 'storeStfBegin(loadInfo, "AudioOutputScopeMode"); storeStfInteger(loadInfo, scopeMode); storeStfEnd(loadInfo);');
  push(@stfLoad, 'if (strcmp(loadInfo->headerTitle, "AudioOutputScopeMode")==0) { readStfInteger(loadInfo, &scopeMode); }');
}

##########################################################################
#
# Config window
#
##########################################################################
sub genConfig {

$rs .= <<EOF;
static const char * const midiUnSelectedPortNames=   " off   A    B    C    D    E    F    G    H    I    J    K    L    M    N    O    P    Q    R    S    T    U    V    W    X    Y    Z  ";
static const char * const midiSelectedPortNames=     "[off][ A ][ B ][ C ][ D ][ E ][ F ][ G ][ H ][ I ][ J ][ K ][ L ][ M ][ N ][ O ][ P ][ Q ][ R ][ S ][ T ][ U ][ V ][ W ][ X ][ Y ][ Z ]";
static const char * const midiUnSelectedChannelNames=" all   1    2    3    4    5    6    7    8    9    10   11   12   13   14   15   16 ";
static const char * const midiSelectedChannelNames=  "[all][ 1 ][ 2 ][ 3 ][ 4 ][ 5 ][ 6 ][ 7 ][ 8 ][ 9 ][ 10][ 11][ 12][ 13][ 14][ 15][ 16]";

static void configWindowAddHitArea(ContextPtr aContext, int x1, int y, int x2) {
  guiAddArea(aContext, x1, y, x2, y+14);
}
static void configWindowHeader(ContextPtr aContext, int x, int *y, int stringIndex) {
  guiSelectPen1Color(aContext, 0x004466);
  TextOut(aContext->hdc, x, *y, strings[stringIndex][currentLanguage], strlen(strings[stringIndex][currentLanguage]));
  (*y)+=14;
}
static void configWindowHeaderStr(ContextPtr aContext, int x, int *y, const char *str) {
  guiSelectPen1Color(aContext, 0x446688);
  TextOut(aContext->hdc, x, *y, str, strlen(str));
  (*y)+=14;
}
static void configWindowItem(ContextPtr aContext, int *y, bool select, const char *str) {
  guiSelectPen1Color(aContext, 0x000000);
  if (select) {
    TextOut(aContext->hdc, 20, *y, "==>", 3);
  }
  TextOut(aContext->hdc, 40, *y, str, strlen(str));
  configWindowAddHitArea(aContext, aContext->guiClientRect.left, *y, aContext->guiClientRect.right);
  (*y)+=14;
}
static int colorNotMask[3]={0xFFFF00,0xFF00FF,0x00FFFF};
static int colorMask[3]={0x0000FF,0x00FF00,0xFF0000};
static int colorSelector[3][colorSteps]={
  {0x000000,0x000011,0x000022,0x000033,0x000044,0x000055,0x000066,0x000077,0x000088,0x000099,0x0000AA,0x0000BB,0x0000CC,0x0000DD,0x0000EE,0x0000FF},
  {0x000000,0x001100,0x002200,0x003300,0x004400,0x005500,0x006600,0x007700,0x008800,0x009900,0x00AA00,0x00BB00,0x00CC00,0x00DD00,0x00EE00,0x00FF00},
  {0x000000,0x110000,0x220000,0x330000,0x440000,0x550000,0x660000,0x770000,0x880000,0x990000,0xAA0000,0xBB0000,0xCC0000,0xDD0000,0xEE0000,0xFF0000}};
static void configWindowColorBar(ContextPtr aContext, int *y, int current) {
  int i,j;

  guiSelectPen1Color(aContext, -1);
  for(i=0;i<colorSteps;i++) {
    for(j=0;j<3;j++) {
      guiSelectFillColor(aContext, (current&colorNotMask[j])+colorSelector[j][i]);
      guiDrawRect(aContext, 30+i*14, (*y)+20*j, 30+i*14+10, (*y)+12+20*j);
      guiSelectFillColor(aContext, ((current&colorMask[j])==colorSelector[j][i])?0:0xFFFFFF);
      guiDrawRect(aContext, 30+i*14, (*y)+14+20*j, 30+i*14+10, (*y)+16+20*j);
      guiAddArea(aContext, 30+i*14, (*y)+20*j, 30+i*14+10, (*y)+20+20*j);
    }
  }
  (*y)+=60;
}

EOF
$rc .= <<EOF;
static void configWindowHandler(ContextPtr aContext) {
  static int maxy=0;
  static int maxysize=0;

  switch (aContext->guiEvent) {
  case GUIEVENT_REFRESH: {
      int y=-guiGetVScrollPos(aContext->currentWindow);
      int i;

      guiClearAreaList(aContext);

      guiSelectFillColor(aContext, 0xFFFFFF);
      guiDraw3DRect(aContext, &(aContext->guiClientRect), 0);
      guiSelectPotFont(aContext);
      guiTextTransparent(aContext, true);

      configWindowHeader(aContext,  4, &y, STRING_GENERAL_SETTINGS);
      configWindowHeader(aContext, 12, &y, STRING_LANGUAGE);
      for(i=0;languageCodes[i];i++) {
        configWindowItem(aContext, &y, currentLanguage==i, strings[STRING_LANGUAGE_NAME][i]);
      }
      configWindowHeader(aContext, 12, &y, STRING_CABLE_TYPE);
      for(i=0;cableTypeList[i]!=STRING_;i++) {
        configWindowItem(aContext, &y, currentCableType==i, strings[cableTypeList[i]][currentLanguage]);
      }      
      configWindowHeader(aContext,  4, &y, STRING_COLORS);
      configWindowHeader(aContext, 12, &y, STRING_COLOR_PATCH_BACKGROUND);
      configWindowColorBar(aContext, &y, defcolor_patchBackground);
      configWindowHeader(aContext, 12, &y, STRING_COLOR_PATCH_MODULES);
      configWindowColorBar(aContext, &y, defcolor_patchModules);
      configWindowHeader(aContext, 12, &y, STRING_COLOR_PATCH_KNOBS);
      configWindowColorBar(aContext, &y, defcolor_patchKnobs);
      configWindowHeader(aContext, 12, &y, STRING_COLOR_PATCH_HIGHLIGHT);
      configWindowColorBar(aContext, &y, defcolor_patchHighlight);
      configWindowHeader(aContext, 12, &y, STRING_COLOR_PATCH_SHADOW);
      configWindowColorBar(aContext, &y, defcolor_patchShadow);


      configWindowHeader(aContext,  4, &y, STRING_AUDIO_SETTINGS);
      configWindowHeader(aContext, 12, &y, STRING_AUDIO_INPUT);
      configWindowHeader(aContext, 20, &y, STRING_AUDIO_IN_DEVICE);
      for(i=0;audioInDeviceNames[i];i++) {
        configWindowItem(aContext, &y, currentAudioInDevice==i, audioInDeviceNames[i]);
      }
      configWindowHeader(aContext, 12, &y, STRING_AUDIO_OUTPUT);
      configWindowHeader(aContext, 20, &y, STRING_AUDIO_OUT_DEVICE);
      for(i=0;audioOutDeviceNames[i];i++) {
        configWindowItem(aContext, &y, currentAudioOutDevice==i, audioOutDeviceNames[i]);
      }
      if (currentAudioOutDevice>1) {
        configWindowHeader(aContext, 20, &y, STRING_NOOF_BUFFERS);
        for(i=2;i<9;i++) {
          char tmp[32];
          itoa(i, tmp, 10);
          configWindowItem(aContext, &y, DSP_NewNoofBuffers==i, tmp);
        }
      }
      configWindowHeader(aContext, 20, &y, STRING_BUFFER_SIZE);
      for(i=0;audioBufferLengthTable[i];i++) {
        char tmp[32];
        itoa(audioBufferLengthTable[i], tmp, 10);
        configWindowItem(aContext, &y, nextAudioOutBufferSize==audioBufferLengthTable[i], tmp);
      }
      configWindowHeader(aContext, 20, &y, STRING_LATENCY);
      {
        char tmp[32];
        if (currentAudioOutDevice>1) {
          sprintf(tmp, "%d ms / %d ms", (nextAudioOutBufferSize*1000)/88200, (DSP_NewNoofBuffers*nextAudioOutBufferSize*1000)/88200);
        } else {
          sprintf(tmp, "%d ms", (nextAudioOutBufferSize*1000)/88200);
        }
        configWindowHeaderStr(aContext, 20, &y, tmp);
      }
      configWindowHeader(aContext, 12, &y, STRING_JOYSTICK);
      for(i=0;i<4;i++) {
        configWindowItem(aContext, &y, joystickMode==i, strings[STRING_JOYSTICK0+i][currentLanguage]);
      }

      configWindowHeader(aContext,  4, &y, STRING_MIDI_SETTINGS);
      if (midiInDeviceNames) {
        configWindowHeader(aContext, 12, &y, STRING_MIDI_IN_DEVICES);
        for(i=0;midiInDeviceNames[i];i++) {
          int j;
          configWindowHeaderStr(aContext, 20, &y, midiInDeviceNames[i]);
          guiSelectPen1Color(aContext, 0);
          for(j=0;j<27;j++) {
            if (j==midiInPorts[i]) {
              TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiSelectedPortNames+j*5, 5);
            } else {
              TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiUnSelectedPortNames+j*5, 5);
            }
            configWindowAddHitArea(aContext, 30+(j%9)*25, y+(j/9)*14, 30+25+(j%9)*25);
          }
          y+=42;
        }
      }
      if (midiOutDeviceNames) {
        configWindowHeader(aContext, 12, &y, STRING_MIDI_OUT_DEVICES);
        for(i=0;midiOutDeviceNames[i];i++) {
          int j;
          configWindowHeaderStr(aContext, 20, &y, midiOutDeviceNames[i]);
          guiSelectPen1Color(aContext, 0);
          for(j=0;j<27;j++) {
            if (j==midiOutPorts[i]) {
              TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiSelectedPortNames+j*5, 5);
            } else {
              TextOut(aContext->hdc, 30+(j%9)*25, y+(j/9)*14, midiUnSelectedPortNames+j*5, 5);
            }
            configWindowAddHitArea(aContext, 30+(j%9)*25, y+(j/9)*14, 30+25+(j%9)*25);
          }
          y+=42;
        }
      }

      if (y+guiGetVScrollPos(aContext->currentWindow)!=maxy || aContext->guiClientRect.bottom!=maxysize) {
        maxy=y+guiGetVScrollPos(aContext->currentWindow);
        maxysize=aContext->guiClientRect.bottom;
        guiVScrollRange(aContext->currentWindow, 0, maxy, maxysize);
      }
    } break;
  case GUIEVENT_MOUSEBUTTON1DOWN: {
      int index=0;
      int i,j;

      for(i=0;languageCodes[i];i++) {
        index++;
        if (index==aContext->id) {
          setLanguageIndex(i);
        }
      }
      for(i=0;cableTypeList[i]!=STRING_;i++) {
        index++;
        if (index==aContext->id) {
          setCableType(i);
        }
      }
      for(j=0;j<colorSteps;j++) {
        for(i=0;i<3;i++) {
          index++;
          if (index==aContext->id) {
            defcolor_patchBackground&=colorNotMask[i];
            defcolor_patchBackground|=colorSelector[i][j];
            mainWindowRefresh=true;
            configWindowRefresh=true;
          }
        }
      }
      for(j=0;j<colorSteps;j++) {
        for(i=0;i<3;i++) {
          index++;
          if (index==aContext->id) {
            defcolor_patchModules&=colorNotMask[i];
            defcolor_patchModules|=colorSelector[i][j];
            mainWindowRefresh=true;
            configWindowRefresh=true;
          }
        }
      }
      for(j=0;j<colorSteps;j++) {
        for(i=0;i<3;i++) {
          index++;
          if (index==aContext->id) {
            defcolor_patchKnobs&=colorNotMask[i];
            defcolor_patchKnobs|=colorSelector[i][j];
            mainWindowRefresh=true;
            configWindowRefresh=true;
          }
        }
      }
      for(j=0;j<colorSteps;j++) {
        for(i=0;i<3;i++) {
          index++;
          if (index==aContext->id) {
            defcolor_patchHighlight&=colorNotMask[i];
            defcolor_patchHighlight|=colorSelector[i][j];
            mainWindowRefresh=true;
            configWindowRefresh=true;
          }
        }
      }
      for(j=0;j<colorSteps;j++) {
        for(i=0;i<3;i++) {
          index++;
          if (index==aContext->id) {
            defcolor_patchShadow&=colorNotMask[i];
            defcolor_patchShadow|=colorSelector[i][j];
            mainWindowRefresh=true;
            configWindowRefresh=true;
          }
        }
      }
      for(i=0;audioInDeviceNames[i];i++) {
        index++;
        if (index==aContext->id) {
          setAudioInDevice(i);
        }
      }
      for(i=0;audioOutDeviceNames[i];i++) {
        index++;
        if (index==aContext->id) {
          setAudioOutDevice(i);
        }
      }
      if (currentAudioOutDevice>1) {
        for(i=2;i<9;i++) {
          index++;
          if (index==aContext->id) {
            DSP_NewNoofBuffers=i;
            configWindowRefresh=true;
          }
        }
      }
      for(i=0;audioBufferLengthTable[i];i++) {
        index++;
        if (index==aContext->id) {
          setAudioOutBufferSize(audioBufferLengthTable[i]);
        }
      }
      for(i=0;i<4;i++) {
        index++;
        if (index==aContext->id) {
          setJoystickMode(i);
        }
      }
      for(i=0;midiInDeviceNames[i];i++) {
        for(j=0;j<27;j++) {
          index++;
          if (index==aContext->id) {
            setMidiInPort(i, j);
          }
        }
      }
      for(i=0;midiOutDeviceNames[i];i++) {
        for(j=0;j<27;j++) {
          index++;
          if (index==aContext->id) {
            setMidiOutPort(i, j);
          }
        }
      }
    } break;
  case GUIEVENT_KEYDOWN:
    switch(aContext->id) {
    case VK_ESCAPE:
      guiHideWindow(aContext->currentWindow);
      break;
    default:
      break;
    } break;
  case GUIEVENT_CLOSE:
    guiHideWindow(configWindow);
    break;
  default:
    break;
  }
}
EOF

  genWindow('configWindow', 'CONFIGWINDOW', '"Configuration and Settings"', 'configWindowHandler', undef, undef, 1, undef);
}


##########################################################################
#
# Load/Save routines
#
##########################################################################
sub genLoadSaveSupport {
  genSection('Load/Save Support routines');

  $rd .= "static const int maxHeaderTitleSize=32;\n";

  my @loaderrorEnum=qw( LOADERROR_OK LOADERROR_FILENOTOPEN LOADERROR_OTHERFORMAT );
  genEnumStruct("enum", "LoadError", \@loaderrorEnum, ',');

  my @loadInfoStruct=(
    'LoadError loadError;',
    'FILE *file;',
    'unsigned char *buffer;',
    'int bufferSize;',
    'int bufferPos;',
    'int headerPos;',
    'int dataSize;',
    'char headerTitle[maxHeaderTitleSize];',
    'bool saveSelectedOnly;',
    'bool generatePatches;', 
    'bool copyPasteFlag;',
    'bool mergeFlag;',
    'int project;',
    'int linkLoadId[maxDspObjects];',
    'int linkCable[maxDspObjects][maxObjectIo];',
    'int linkMyObject[maxDspObjects];');
  genEnumStruct('struct', 'LoadInfo', \@loadInfoStruct);

$rc .= <<EOF;
static void loadReset(LoadInfo *loadInfo) {
  if (loadInfo->file) {
    fseek(loadInfo->file, 0, SEEK_SET);
  }
  loadInfo->bufferPos=0;
}

static void storeStfInteger(LoadInfo *loadInfo, int aValue) {
  unsigned char myBuffer[4];

  myBuffer[0]=(aValue >> 24) & 255;
  myBuffer[1]=(aValue >> 16) & 255;
  myBuffer[2]=(aValue >> 8) & 255;
  myBuffer[3]=aValue & 255;
  if (loadInfo->file) fwrite(myBuffer, 4, 1, loadInfo->file);
  if (loadInfo->buffer) memcpy(loadInfo->buffer+loadInfo->bufferPos, myBuffer, 4);
  loadInfo->bufferPos+=4;
}

static void storeStfBuffer(LoadInfo *loadInfo, const char *aBuffer, int aBufferSize) {
  if (aBuffer) {
    storeStfInteger(loadInfo, aBufferSize);
    if (loadInfo->file) fwrite(aBuffer, aBufferSize, 1, loadInfo->file);
    if (loadInfo->buffer) memcpy(loadInfo->buffer+loadInfo->bufferPos, aBuffer, aBufferSize);
    loadInfo->bufferPos+=aBufferSize;
  } else {
    storeStfInteger(loadInfo, 0);
  }
}

static void storeStfString(LoadInfo *loadInfo, const char *aString) {
  if (aString) {
    storeStfBuffer(loadInfo, aString, strlen(aString));
  } else {
    storeStfBuffer(loadInfo, NULL, 0);
  }
}

static bool readLittleEndianLong(LoadInfo *loadInfo, int *aValuePtr) {
  unsigned char myBuffer[4];

  if (loadInfo->file && fread(myBuffer, 4, 1, loadInfo->file)==1) {
    (*aValuePtr)=(myBuffer[3]<<24)+(myBuffer[2]<<16)+(myBuffer[1]<<8)+(myBuffer[0]);
    return true;
  } if (loadInfo->buffer && loadInfo->bufferPos<loadInfo->bufferSize) {
    (*aValuePtr) =(loadInfo->buffer[loadInfo->bufferPos++]);
    (*aValuePtr)+=(loadInfo->buffer[loadInfo->bufferPos++]<<8);
    (*aValuePtr)+=(loadInfo->buffer[loadInfo->bufferPos++]<<16);
    (*aValuePtr)+=(loadInfo->buffer[loadInfo->bufferPos++]<<24);    
    return true;
  } else {
    (*aValuePtr)=0;
    return false;
  }
}

static bool readStfInteger(LoadInfo *loadInfo, int *aValuePtr) {
  unsigned char myBuffer[4];

  if (loadInfo->file && fread(myBuffer, 4, 1, loadInfo->file)==1) {
    (*aValuePtr)=(myBuffer[0]<<24)+(myBuffer[1]<<16)+(myBuffer[2]<<8)+(myBuffer[3]);
    return true;
  } if (loadInfo->buffer && loadInfo->bufferPos<loadInfo->bufferSize) {
    (*aValuePtr) =(loadInfo->buffer[loadInfo->bufferPos++]<<24);
    (*aValuePtr)+=(loadInfo->buffer[loadInfo->bufferPos++]<<16);
    (*aValuePtr)+=(loadInfo->buffer[loadInfo->bufferPos++]<<8);
    (*aValuePtr)+=(loadInfo->buffer[loadInfo->bufferPos++]);    
    return true;
  } else {
    (*aValuePtr)=0;
    return false;
  }
}

static bool readStfString(LoadInfo *loadInfo, char *buffer, int bufferLength) {
  int curPos;
  int loadStringLength;
  int len;
  
  if ((!readStfInteger(loadInfo, &loadStringLength)) || loadStringLength<0) return false;
  if (loadInfo->file) {
    curPos=ftell(loadInfo->file);
    len=min(loadStringLength, bufferLength-1);
    memset(buffer, 0, bufferLength);
    if (len>0) {
      fread(buffer, len, 1, loadInfo->file);
      fseek(loadInfo->file, curPos+loadStringLength, SEEK_SET);
    }
  } else {
    len=min(loadStringLength, bufferLength-1);
    memset(buffer, 0, bufferLength);
    memcpy(buffer, loadInfo->buffer+loadInfo->bufferPos, len);
    loadInfo->bufferPos+=loadStringLength;
  }
  return true;
}

static bool allocateStfString(LoadInfo *loadInfo, char **string) {
  int len;
  char *str;

  if ((!readStfInteger(loadInfo, &len)) || len<0) return false;
  str=(char*)malloc(len+1);
  if (loadInfo->file) {
    memset(str, 0, len+1);
    fread(str, len, 1, loadInfo->file);
  } else {
    str[len]='\\0';
    memcpy(str, loadInfo->buffer+loadInfo->bufferPos, len);
    loadInfo->bufferPos+=len;
  }
  *string=str;
  return true;
}

static void storeStfBegin(LoadInfo *loadInfo, const char *aHeaderTitle) {
  storeStfString(loadInfo, aHeaderTitle);
  if (loadInfo->file) {
    loadInfo->headerPos=ftell(loadInfo->file);
  } else {
    loadInfo->headerPos=loadInfo->bufferPos;
  }
  storeStfInteger(loadInfo, 0);
}

static void storeStfEnd(LoadInfo *loadInfo) {
  int curPos;

  if (loadInfo->file) {
    curPos=ftell(loadInfo->file);
    fseek(loadInfo->file, loadInfo->headerPos, SEEK_SET);
    storeStfInteger(loadInfo, curPos-(loadInfo->headerPos+4));
    fseek(loadInfo->file, curPos, SEEK_SET);
  } else {
    curPos=loadInfo->bufferPos;
    loadInfo->bufferPos=loadInfo->headerPos;
    storeStfInteger(loadInfo, curPos-(loadInfo->headerPos+4));
    loadInfo->bufferPos=curPos;
  }
}

static bool readStfHeader(LoadInfo *loadInfo) {
  bool result=readStfString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize);
  readStfInteger(loadInfo, &(loadInfo->dataSize));
  if (loadInfo->file) {
    loadInfo->headerPos = ftell(loadInfo->file);
  } else {
    loadInfo->headerPos = loadInfo->bufferPos;
  }
  return result;
}

EOF

}

sub genLoadSave {
  genSection('Load/Save');

$rc .= <<EOF;
//
// dest - buffer used to store the string
// maxlen - maximum length of the string (excluding the terminating '0') so dest must be maxlen+1 in size
// sf - if reading from file this is the file descriptor
// iBuffer - if reading from a character buffer this is a pointer to a pointer to the buffer.
// useComma - Due to backwards compatibility, some commands use comma seperated values.
//
// return value  0 no string read.
//               >0 single string read (returned value is string length).
//
static int readString(LoadInfo *loadInfo, char *dest, int maxlen, bool useComma) {
  char *tmpdest=dest;
  if (tmpdest) {
    int chr;
    if (loadInfo->file) {
      chr=getc(loadInfo->file);
      // Skip leading spaces and control chars
      while ((chr!=EOF) && ((chr<=32) || (chr==',' && useComma))) {
        chr=getc(loadInfo->file);
      }
      while ((chr!=EOF) && (chr>32) && (chr!=',' || (!useComma))) {
        if (maxlen>0) {
          *tmpdest=(char)chr;
          tmpdest++;
          maxlen--;
        }
        chr=getc(loadInfo->file);
      }
    }
    if (loadInfo->buffer) {
      // Skip leading spaces and control chars
      while ((loadInfo->buffer[loadInfo->bufferPos]>0) && (loadInfo->buffer[loadInfo->bufferPos]<=32 || (loadInfo->buffer[loadInfo->bufferPos]==',' && useComma))) {
        loadInfo->bufferPos++;
      }
      while (loadInfo->buffer[loadInfo->bufferPos]>32 && (loadInfo->buffer[loadInfo->bufferPos]!=',' || (!useComma))) {
        if (maxlen>0) {
          *tmpdest=loadInfo->buffer[loadInfo->bufferPos];          
          tmpdest++;
          maxlen--;
        }
        loadInfo->bufferPos++;
      }

      // Just like filebased method we consume last read character
      if (loadInfo->buffer[loadInfo->bufferPos] && (loadInfo->buffer[loadInfo->bufferPos]<=32 || (loadInfo->buffer[loadInfo->bufferPos]==',' && useComma))) {
        loadInfo->bufferPos++;
      }
    }

    *tmpdest='\\0';
    return strlen(dest);
  }
  return 0;
}

//
// Read integer from file or character buffer.
// returns 0 when no integer read.
//         1 when single integer read.
//
static int readInteger(LoadInfo *loadInfo, int *i, bool useComma) {
  char tmpbuf[32];
  if (readString(loadInfo, tmpbuf, 31, useComma)>0) {
    *i=atoi(tmpbuf);
    return 1;
  } else {
    *i=0;
  }
  return 0;
}

//
// Read single char from input file or character buffer.
// returns 0 when no character read.
//         1 when single character read.
//
static int readChar(LoadInfo *loadInfo, int *chr) {
  if (loadInfo->file) {
    *chr=getc(loadInfo->file);
    return 1;
  }
  if (loadInfo->buffer) {
    *chr=loadInfo->buffer[loadInfo->bufferPos++];
    return 1;
  }
  return 0;
}

static void loadClearRewireTables(LoadInfo *loadInfo) {
  memset(loadInfo->linkLoadId, 0, sizeof(loadInfo->linkLoadId));
  memset(loadInfo->linkMyObject, 0, sizeof(loadInfo->linkMyObject));
  memset(loadInfo->linkCable, 0, sizeof(loadInfo->linkCable));
}

static void loadRewire(LoadInfo *loadInfo, int patch) {
  // connect the cables
  int idsp,odsp,input;
  for(idsp=0;idsp<maxDspObjects;idsp++) {
    if (loadInfo->linkMyObject[idsp]>=0) {
      for(input=0;input<maxObjectIo;input++) {
        if (loadInfo->linkCable[idsp][input]) {
          for(odsp=0;odsp<maxDspObjects;odsp++) {
            if (loadInfo->linkLoadId[odsp]==loadInfo->linkCable[idsp][input]) {
              currentCableColorIndex=projects[currentProject].patches[patch].objects[loadInfo->linkMyObject[idsp]].connections[input].cableColor;
              selectOutput(&(projects[currentProject].patches[patch]), loadInfo->linkMyObject[odsp], projects[currentProject].patches[patch].objects[loadInfo->linkMyObject[idsp]].connections[input].fromOutput);
              connectCable(&(projects[currentProject].patches[patch]), loadInfo->linkMyObject[idsp], input);
            }
          }
        }
      }
    }
  }
}

typedef enum _SynFileSection {
  SECTION_NONE,
  SECTION_SETTINGS,
  SECTION_WINDOWS,
  SECTION_SAMPLES,
  SECTION_PATCHES
} SynFileSection;
static void logNotSynFile(void) {
  logprintf("Not a SynFactory 1.x (SYN or SYF) file.\\n");
}
static void loadSyn(LoadInfo *loadInfo) {
  SynFileSection section=SECTION_NONE;
  int offsetX=0, offsetY=0;
  int patch=currentPatch;
  int module=-1;
  bool createdProject=false;

  if (readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)<=0 ||
      strcmp(loadInfo->headerTitle, "FILETYPE")!=0 ||
      readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)<=0 ||
      strcmp(loadInfo->headerTitle, "SYN")!=0) {
    logNotSynFile();
    return;
  }
  if (readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)>0 &&
      strcmp(loadInfo->headerTitle, "PROGRAM")==0 &&
      readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)>0) {
    logprintf("File generated by '%s'.\\n", loadInfo->headerTitle);
  } else {
    logprintf("Expected 'PROGRAM'.\\n");
    logNotSynFile();
    return;
  }

  loadInfo->loadError=LOADERROR_OK;
  while (readString(loadInfo, loadInfo->headerTitle, maxHeaderTitleSize-1, false)>0) {
  if (strcmp(loadInfo->headerTitle, "{{{SETTINGS")==0) {
    section=SECTION_SETTINGS;
  }
  if (strcmp(loadInfo->headerTitle, "}}}SETTINGS")==0) {
    section=SECTION_NONE;
  }
  if (strcmp(loadInfo->headerTitle, "{{{WINDOWS")==0) {
    section=SECTION_WINDOWS;
  }
  if (strcmp(loadInfo->headerTitle, "}}}WINDOWS")==0) {
    section=SECTION_NONE;
  }
  if ((strcmp(loadInfo->headerTitle, "{{{SAMPLES")==0) || (strcmp(loadInfo->headerTitle, "{{{SAMPLEBANK")==0)) {
    section=SECTION_SAMPLES;
  }
  if ((strcmp(loadInfo->headerTitle, "}}}SAMPLES")==0)  || (strcmp(loadInfo->headerTitle, "}}}SAMPLEBANK")==0)) {
    section=SECTION_NONE;
  }
  if (strcmp(loadInfo->headerTitle, "{{{PATCHES")==0) {
    section=SECTION_PATCHES;
  }
  if (strcmp(loadInfo->headerTitle, "}}}PATCHES")==0) {
    patch=-1;
    module=-1;
    section=SECTION_NONE;
  }
  if (strcmp(loadInfo->headerTitle, "{{{PATCH")==0) {
    section=SECTION_PATCHES; // Pre 2.00 SYN files don't have an PATCHES section. We fake one when '{{{PATCH' is seen.
    if (projects==NULL || (!loadInfo->copyPasteFlag && !loadInfo->mergeFlag && !createdProject)) {
      createProject(newProjectName);
      createdProject=true;
      patch=-1;
    }
    if (patch<0) {
      patch=createSynfactoryPatch();
    }
    loadClearRewireTables(loadInfo);
  }
  if (strcmp(loadInfo->headerTitle, "}}}PATCH")==0) {
    loadRewire(loadInfo, patch);
    patch=-1;
    module=-1;
  }
  if (strcmp(loadInfo->headerTitle, "{{{MODULE")==0 && patch>=0) {
    ObjectType type=OBJ_NONE;
    char modname[32];
    int index=0;
    int i;

    readInteger(loadInfo, &index, true);
    readString(loadInfo, modname, 32, true);
    for(i=0;objectLoadNames[i];i++) {
      if (strcmp(objectLoadNames[i], modname)==0) {
        type=(ObjectType)i;
        break;
      }
    }
    // Try aliases
    if (type==OBJ_NONE) {
      if (strcmp("VCF", modname)==0) type=OBJ_OBVCF;
      if (modname[0]=='A' && modname[1]=='D' && modname[2]>'1' && modname[2]<'9' && modname[3]==0) type=OBJ_ADD;
      if (modname[0]=='A' && modname[1]=='N' && modname[2]=='D' && modname[3]>'2' && modname[4]<'9' && modname[5]==0) type=OBJ_AND;
      if (modname[0]=='O' && modname[1]=='R' && modname[2]>'2' && modname[2]<'9' && modname[3]==0) type=OBJ_OR;
    }
    if (type!=OBJ_NONE) {
      module=addSynFactoryObject(patch, type, false);
      if (module>=0 && projects[currentProject].patches[patch].objects[module].dsp) {
        loadInfo->linkLoadId[projects[currentProject].patches[patch].objects[module].dsp]=index;
        loadInfo->linkMyObject[projects[currentProject].patches[patch].objects[module].dsp]=module;
      }
    }
  }
  if (strcmp(loadInfo->headerTitle, "}}}MODULE")==0) {
    module=-1;
  }
  if (strcmp(loadInfo->headerTitle, "OFFSET")==0 && module>=0 && patch>=0) {
    int x,y;
    readInteger(loadInfo, &x, true);
    readInteger(loadInfo, &y, true);
    projects[currentProject].patches[patch].objects[module].x=x-offsetX;
    projects[currentProject].patches[patch].objects[module].y=y-offsetY;
    
  }
  if (strcmp(loadInfo->headerTitle, "POT") == 0 && module>=0 && patch>=0) {
    int n,v;
    readInteger(loadInfo, &n, true);
    readInteger(loadInfo, &v, true);
    if (projects[currentProject].patches[patch].objects[module].dsp && n>=0 && n<maxObjectIo) {
      projects[currentProject].dsp[projects[currentProject].patches[patch].objects[module].dsp].io[n].knob=v;
    }
  }
  if (strcmp(loadInfo->headerTitle, "INPUT") == 0 && module>=0 && patch>=0) {
    int i=-1,n,o;
    readInteger(loadInfo, &i, true);
    readInteger(loadInfo, &n, true);
    readInteger(loadInfo, &o, true);
    if (i>=0 && i<maxObjectIo) {
      loadInfo->linkCable[projects[currentProject].patches[patch].objects[module].dsp][i]=n;
      projects[currentProject].patches[patch].objects[module].connections[i].fromOutput=o+1;
      projects[currentProject].patches[patch].objects[module].connections[i].cableColor=0;
    }
  }
  if (strcmp(loadInfo->headerTitle, "CABLECOLOR") == 0 && module>=0 && patch>=0) {
    int i,c;
    readInteger(loadInfo, &i, true);
    readInteger(loadInfo, &c, true);
    if (i>=0 && i<maxObjectIo) {
      projects[currentProject].patches[patch].objects[module].connections[i].cableColor=c;
    }
  }
  if (strcmp(loadInfo->headerTitle, "MODULENAME") == 0 && module>=0 && patch>=0) {
    int l=0;
    int i;

    readInteger(loadInfo, &l, true);
    projects[currentProject].patches[patch].objects[module].name=(char*)calloc(1,l+1);
    for(i=0;i<l;i++) {
      int c;
      readChar(loadInfo, &c);
      projects[currentProject].patches[patch].objects[module].name[i]=(char)c;
    }
  }
  if (strcmp(loadInfo->headerTitle, "SCRIPT") == 0  && module>=0 && patch>=0 && projects[currentProject].patches[patch].objects[module].objectType==OBJ_NOTESCRIPT) {
    int l=0;
    int i;

    lockMutex(audioMutex);
    readInteger(loadInfo, &l, true);
    projects[currentProject].dsp[projects[currentProject].patches[patch].objects[module].dsp].scriptInfo->script=(char*)calloc(1,l+1);
    for(i=0;i<l;i++) {
      int c;
      readChar(loadInfo, &c);
      projects[currentProject].dsp[projects[currentProject].patches[patch].objects[module].dsp].scriptInfo->script[i]=(char)c;
    }
    unlockMutex(audioMutex);
  }
  if (strcmp(loadInfo->headerTitle, "SIZE") == 0 && module>=0 && patch>=0) {
    int xs,ys;
    readInteger(loadInfo, &xs, true);
    readInteger(loadInfo, &ys, true);
    projects[currentProject].patches[patch].objects[module].xs=xs;
    projects[currentProject].patches[patch].objects[module].ys=ys;
  }
  if (strcmp(loadInfo->headerTitle, "COLOR") == 0 && module>=0 && patch>=0) {
    int m,b;
    readInteger(loadInfo, &m, true);
    readInteger(loadInfo, &b, true);

    if (m>0 && m<8) { m=cablePenColors[m]; } else if (m>=8 && m<16) { m=cableFillColors[m-8]; } else m=0;
    if (b>0 && b<8) { b=cablePenColors[b]; } else if (b>=8 && b<16) { b=cableFillColors[b-8]; } else b=0;
    projects[currentProject].patches[patch].objects[module].mainColor=m;
    projects[currentProject].patches[patch].objects[module].borderColor=b;
  }
  if (strcmp(loadInfo->headerTitle, "DRUM") == 0 && module>=0 && patch>=0) {
    int l;
    readInteger(loadInfo, &l, true);
    if ((projects[currentProject].patches[patch].objects[module].objectType==OBJ_DRM4 && l==64) ||
        (projects[currentProject].patches[patch].objects[module].objectType==OBJ_DRM8 && l==128)) {
      int i;

      for(i=0;i<l;i++) {
        int f;
        readChar(loadInfo, &f);
        ((unsigned char *)projects[currentProject].dsp[projects[currentProject].patches[patch].objects[module].dsp].buffer)[i]=(unsigned char)((f=='1')?1:0);
      }
    } else {
      logprintf("Can't process DRUM %d command\\n", l);
    }
  }
  if (strcmp(loadInfo->headerTitle, "MODE") == 0 && module>=0 && patch>=0) {
    int mode;

    readInteger(loadInfo, &mode, false);
    setMode(&(projects[currentProject].patches[patch]), module, mode);
//    projects[currentProject].patches[patch].objects[module].mode=mode;
  }


EOF

  for(my $i=0;$i<scalar(@synLoad);$i++) {
    $rc .= $synLoad[$i];
  }

$rc .= <<EOF;
  }
}

static void loadStf(LoadInfo *loadInfo) {
  int patch=currentPatch;
  int module=-1;
  bool savedModified=false;
  // When cloning we want the new objects to be selected, so they can be moved to a new location.
  // selectNew is true when we paste in existing patch. secondObject is a flag set after the first object is created so a existing
  // selection can first be removed. Then this flag prevents the previous inserted object(s) from de-selection.
  bool secondObject=false;
  bool selectNew=true;
  bool firstProject=true;
  readStfHeader(loadInfo);

  loadClearRewireTables(loadInfo);

  if (strcmp(loadInfo->headerTitle, "STUDIOFACTORY100")!=0) {
    logprintf("Not a StudioFactory (STF) file. File doesn't start with STUDIOFACTORY100 chunk.\\n");
    return;
  }

  loadInfo->loadError=LOADERROR_OK;
  loadInfo->project=currentProject;
  while (readStfHeader(loadInfo)) {
EOF

    for(my $i=0;$i<scalar(@stfLoad);$i++) {
      $rc = "$rc    " . $stfLoad[$i] . "\n";
    }

$rc .= <<EOF;
    if (strcmp(loadInfo->headerTitle, "ProjectBegin")==0) {
      if (loadInfo->project<0 || !firstProject || !loadInfo->mergeFlag) {
        loadInfo->project=createProject(NULL);
        if (loadInfo->project>=0) {
          allocateStfString(loadInfo, &projects[loadInfo->project].projectName);
        }
        // We just loaded it from disk so can clear modifiedFlag.
        // One small problem for autoload.stf which can contain unsaved projects.
        // savedModified is set by adding a ProjectModified chunk. Chunk only found in autoload.stf
        savedModified=false;
        patch=-1;
      }
      firstProject=false;
    }
    if (strcmp(loadInfo->headerTitle, "ProjectModified")==0) {
      savedModified=true;
    }
    if (strcmp(loadInfo->headerTitle, "PatchBegin")==0) {
      patch=createSynfactoryPatch();
      selectNew=false;
//      loadClearRewireTables(loadInfo);
    }
    if (strcmp(loadInfo->headerTitle, "PatchMidi")==0 && patch>=0) {
      readStfInteger(loadInfo, &projects[loadInfo->project].patches[patch].midiPort);
      readStfInteger(loadInfo, &projects[loadInfo->project].patches[patch].midiChannel);
    }
    if (strcmp(loadInfo->headerTitle, "ModuleBegin")==0) {
      if (patch<0) {
        // hack
        if (projects[loadInfo->project].maxPatches>0) {
          patch=0;
        } else {
          patch=createSynfactoryPatch();
          selectNew=false;
        }
      }
      ObjectType type=OBJ_NONE;
      char moduleTypeString[32];
      int loadId=-1;
      int i;

      readStfInteger(loadInfo, &loadId);
      readStfString(loadInfo, moduleTypeString, 32);

      for(i=0;objectLoadNames[i];i++) {
        if (strcmp(objectLoadNames[i], moduleTypeString)==0) {
          type=(ObjectType)i;
          break;
        }
      }
      if (type!=OBJ_NONE) {
        module=addSynFactoryObject(patch, type, secondObject && selectNew);
        secondObject=true;
        if (module>=0 && projects[currentProject].patches[patch].objects[module].dsp) {
          loadInfo->linkLoadId[projects[currentProject].patches[patch].objects[module].dsp]=loadId;
          loadInfo->linkMyObject[projects[currentProject].patches[patch].objects[module].dsp]=module;
        }
      }
    }
    if (strcmp(loadInfo->headerTitle, "ModulePosition") == 0 && module>=0 && patch>=0) {
      int x=0,y=0;
      readStfInteger(loadInfo, &x);
      readStfInteger(loadInfo, &y);
      projects[currentProject].patches[patch].objects[module].x=x+projects[currentProject].patches[patch].offsX;
      projects[currentProject].patches[patch].objects[module].y=y+projects[currentProject].patches[patch].offsY;
    }
    if (strcmp(loadInfo->headerTitle, "ModuleLabel") == 0 && module>=0 && patch>=0) {
      int f=0;
      readStfInteger(loadInfo, &f);
      projects[currentProject].patches[patch].objects[module].showLabel=(f!=0);
    }
    if (strcmp(loadInfo->headerTitle, "ModuleCable") == 0 && module>=0 && patch>=0) {
      int i=-1,n,o,c;
      readStfInteger(loadInfo, &i);
      readStfInteger(loadInfo, &n);
      readStfInteger(loadInfo, &o);
      readStfInteger(loadInfo, &c);
      if (i>=0 && i<maxObjectIo) {
        loadInfo->linkCable[projects[currentProject].patches[patch].objects[module].dsp][i]=n;
        projects[currentProject].patches[patch].objects[module].connections[i].fromOutput=o;
        projects[currentProject].patches[patch].objects[module].connections[i].cableColor=c;
      }
    }
    if (strcmp(loadInfo->headerTitle, "ModuleKnob") == 0 && module>=0 && patch>=0) {
      int i=-1,v;
      readStfInteger(loadInfo, &i);
      readStfInteger(loadInfo, &v);
      if (i>=0 && i<maxObjectIo && projects[currentProject].patches[patch].objects[module].dsp) {
        projects[currentProject].dsp[projects[currentProject].patches[patch].objects[module].dsp].io[i].knob=v;
      }
    }
    if (strcmp(loadInfo->headerTitle, "ModuleName") == 0 && module>=0 && patch>=0) {
      allocateStfString(loadInfo, &(projects[currentProject].patches[patch].objects[module].name));
      if (projects[currentProject].patches[patch].objects[module].dsp) {
        updateModuleLabel(&(projects[currentProject].patches[patch]), module, 0);
      }
    }
    if (strcmp(loadInfo->headerTitle, "ModuleScript") == 0 && module>=0 && patch>=0 && projects[currentProject].patches[patch].objects[module].objectType==OBJ_NOTESCRIPT) {
      lockMutex(audioMutex);
      allocateStfString(loadInfo, &(projects[currentProject].dsp[projects[currentProject].patches[patch].objects[module].dsp].scriptInfo->script));
      unlockMutex(audioMutex);
    }
    if (strcmp(loadInfo->headerTitle, "ModuleBuffer") == 0 && module>=0 && patch>=0) {
      readStfString(loadInfo, (char *)projects[currentProject].dsp[projects[currentProject].patches[patch].objects[module].dsp].buffer, objectBufferSize(projects[currentProject].patches[patch].objects[module].objectType));
    }
    if (strcmp(loadInfo->headerTitle, "ModuleSize") == 0 && module>=0 && patch>=0 && projects[currentProject].patches[patch].objects[module].dsp==0) {
      readStfInteger(loadInfo, &(projects[currentProject].patches[patch].objects[module].xs));
      readStfInteger(loadInfo, &(projects[currentProject].patches[patch].objects[module].ys));
    }
    if (strcmp(loadInfo->headerTitle, "ModuleColor") == 0 && module>=0 && patch>=0 && projects[currentProject].patches[patch].objects[module].dsp==0) {
      readStfInteger(loadInfo, &(projects[currentProject].patches[patch].objects[module].mainColor));
      readStfInteger(loadInfo, &(projects[currentProject].patches[patch].objects[module].borderColor));
    }
    if (strcmp(loadInfo->headerTitle, "ModuleMode") == 0 && module>=0 && patch>=0) {
      int m=0;
      readStfInteger(loadInfo, &m);
      setMode(&(projects[currentProject].patches[patch]), module, m);
    }
    if (strcmp(loadInfo->headerTitle, "ModuleEnd")==0) {
      module=-1;
    }
    if (strcmp(loadInfo->headerTitle, "PatchEnd")==0) {
      loadRewire(loadInfo, patch);
      loadClearRewireTables(loadInfo);
      module=-1;
      patch=-1;
    }
    if (strcmp(loadInfo->headerTitle, "ProjectEnd")==0) {
      if (patch>=0) {
        loadRewire(loadInfo, patch);
        loadClearRewireTables(loadInfo);
      }
      module=-1;
      patch=-1;
      // We just loaded it from disk so can clear modifiedFlag.
      // One small problem for autoload.stf which can contain unsaved projects.
      // savedModified is set by adding a ProjectModified chunk. Chunk only found in autoload.stf
      projects[currentProject].modifiedFlag=savedModified;
    }
    if (strcmp(loadInfo->headerTitle, "STUDIOFACTORYEND")==0) {
      if (patch>=0) {
        loadRewire(loadInfo, patch);
      }
      // leave loop (end of file)
      break;
    }
    if (loadInfo->file) {
      fseek(loadInfo->file, loadInfo->headerPos+loadInfo->dataSize, SEEK_SET);
    } else {
      loadInfo->bufferPos=loadInfo->headerPos+loadInfo->dataSize;
    }
  }
}

//
// Read buffer and optional compare header of this buffer to template.
//
bool readAndCompare(LoadInfo *loadInfo, void *buf, int len, const void *cmpstr, int cmplen) {
  // read data
  if (fread(buf, 1, len, loadInfo->file)!=(unsigned)len) {
    return false;
  }
  // compare data
  if (cmplen && (memcmp(buf, cmpstr, cmplen)!=0)) {
    return false;
  }
  return true;
}

static const unsigned char riffId[4]={'R','I','F','F'};
static const unsigned char waveId[4]={'W','A','V','E'};
static void loadWav(LoadInfo *loadInfo) {
  unsigned char tmp[256];
  int fileSize=0;

  if (!readAndCompare(loadInfo, tmp, 4, riffId, 4)) {
    logprintf("Not a WAV file. The RIFF header is missing.\\n");
    return;
  }
  if (readLittleEndianLong(loadInfo, &fileSize)) {
    logprintf("Data size specified in RIFF header %d (8 less as filesize in bytes).\\n", fileSize);
  } else {
    logprintf("Not a WAV file. The RIFF header length field is missing.\\n");
    return;
  }
  if (!readAndCompare(loadInfo, tmp, 4, waveId, 4)) {
    logprintf("Not a WAV file. The header was found, but is not of expected type WAVE.\\n");
    return;
  }
}

static void loadTry(LoadInfo *loadInfo) {
  if (loadInfo->loadError == LOADERROR_OTHERFORMAT) {
    loadReset(loadInfo);
    loadSyn(loadInfo);
  }
  if (loadInfo->loadError == LOADERROR_OTHERFORMAT) {
    loadReset(loadInfo);
    loadStf(loadInfo);
  }
  if (loadInfo->loadError == LOADERROR_OTHERFORMAT) {
    loadReset(loadInfo);
//    loadWav(loadInfo);
  }
}

static LoadError loadFromFilename(ContextPtr aContext, const char *filename, bool merge) {
  LoadInfo loadInfo;

  memset(&loadInfo, 0, sizeof(LoadInfo));
  loadInfo.loadError=LOADERROR_FILENOTOPEN;
  loadInfo.file=fopen(filename, "rb");
  logprintf("loadFromFilename '%s'\\n", filename);
  if (loadInfo.file) {
    logprintf("File opened ok.\\n");
    loadInfo.loadError=LOADERROR_OTHERFORMAT;
    loadInfo.copyPasteFlag=false;
    loadInfo.mergeFlag=merge;
    loadTry(&loadInfo);
    fclose(loadInfo.file);
  } else {
    logprintf("Error opening file.\\n");
  }

  return loadInfo.loadError;
}

static LoadError loadFromBuffer(const unsigned char *buffer, int bufferSize) {
  LoadInfo loadInfo;

  memset(&loadInfo, 0, sizeof(LoadInfo));
  loadInfo.loadError=LOADERROR_OTHERFORMAT;
  loadInfo.buffer=(unsigned char *)buffer; // I promise no writing, but try telling the compiler that :-)
  loadInfo.bufferSize=bufferSize;
  loadInfo.copyPasteFlag=true;
  loadInfo.mergeFlag=false;
  loadTry(&loadInfo);

  return loadInfo.loadError;
}

static void saveStfSynFactoryPatch(LoadInfo *loadInfo, int patch) {
  bool save=true;
  bool saveAll=true;
  int i;
  
  if (loadInfo->saveSelectedOnly) {
    int i;

    save=false;
    saveAll=false;
    if (projects[loadInfo->project].patches[patch].selected) saveAll=true;
    for(i=0;i<projects[loadInfo->project].patches[patch].maxObjects;i++) {
      if (projects[loadInfo->project].patches[patch].objects[i].selected) {
        save=true;
        break;
      }
    }
  }
  if (save || saveAll) {
    if (loadInfo->generatePatches) {
      storeStfBegin(loadInfo, "PatchBegin");
      storeStfEnd(loadInfo);
      storeStfBegin(loadInfo, "PatchMidi");
      storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].midiPort);
      storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].midiChannel);
      storeStfEnd(loadInfo);
    }
    for(i=0;i<projects[loadInfo->project].patches[patch].maxObjects;i++) {
      if (saveAll || projects[loadInfo->project].patches[patch].objects[i].selected) {
        int j;
        storeStfBegin(loadInfo, "ModuleBegin");
        storeStfInteger(loadInfo, i+1);
        storeStfString(loadInfo, objectLoadNames[projects[loadInfo->project].patches[patch].objects[i].objectType]);
        storeStfEnd(loadInfo);
        storeStfBegin(loadInfo, "ModulePosition");
        // Give +16,+16 offset when copy/pasting so cloning and copy/paste in same patch will not exactly position the modules on top of the existing ones.
        storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].x+(loadInfo->generatePatches?0:16)-projects[loadInfo->project].patches[patch].offsX);
        storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].y+(loadInfo->generatePatches?0:16)-projects[loadInfo->project].patches[patch].offsY);
        storeStfEnd(loadInfo);
        if (projects[loadInfo->project].patches[patch].objects[i].showLabel) {
          storeStfBegin(loadInfo, "ModuleLabel");
          storeStfInteger(loadInfo, 1);
          storeStfEnd(loadInfo);          
        }
        for(j=0;j<maxObjectIo;j++) {
          if (projects[loadInfo->project].patches[patch].objects[i].connections[j].fromObject) {
            storeStfBegin(loadInfo, "ModuleCable");
            storeStfInteger(loadInfo, j);
            storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].connections[j].fromObject);
            storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].connections[j].fromOutput);
            storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].connections[j].cableColor);
            storeStfEnd(loadInfo);
          }
        }
        for(j=0;j<maxObjectIo;j++) {
          if (knobName(projects[loadInfo->project].patches[patch].objects[i].objectType, j)) {
            storeStfBegin(loadInfo, "ModuleKnob");
            storeStfInteger(loadInfo, j);
            storeStfInteger(loadInfo, projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].io[j].knob);
            storeStfEnd(loadInfo);
          }
        }
        if (projects[loadInfo->project].patches[patch].objects[i].name) {
          storeStfBegin(loadInfo, "ModuleName");
          storeStfString(loadInfo, projects[loadInfo->project].patches[patch].objects[i].name);
          storeStfEnd(loadInfo);
        }
        if (projects[loadInfo->project].patches[patch].objects[i].objectType==OBJ_NOTESCRIPT &&
            projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].scriptInfo->script) {
          storeStfBegin(loadInfo, "ModuleScript");
          storeStfString(loadInfo, projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].scriptInfo->script);
          storeStfEnd(loadInfo);
        } else {
          if (objectBufferKeep(projects[loadInfo->project].patches[patch].objects[i].objectType)) {
            storeStfBegin(loadInfo, "ModuleBuffer");
            storeStfBuffer(loadInfo, (const char *)projects[loadInfo->project].dsp[projects[loadInfo->project].patches[patch].objects[i].dsp].buffer, objectBufferSize(projects[loadInfo->project].patches[patch].objects[i].objectType));
            storeStfEnd(loadInfo);
          }
        }
        if (projects[loadInfo->project].patches[patch].objects[i].dsp==0) {
          storeStfBegin(loadInfo, "ModuleSize");
          storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].xs);
          storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].ys);
          storeStfEnd(loadInfo);
          storeStfBegin(loadInfo, "ModuleColor");
          storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].mainColor);
          storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].borderColor);
          storeStfEnd(loadInfo);
        }
        if (projects[loadInfo->project].patches[patch].objects[i].mode) {
          storeStfBegin(loadInfo, "ModuleMode");
          storeStfInteger(loadInfo, projects[loadInfo->project].patches[patch].objects[i].mode);
          storeStfEnd(loadInfo);
        }
        storeStfBegin(loadInfo, "ModuleEnd");
        storeStfEnd(loadInfo);
      }
    }
    if (loadInfo->generatePatches) {
      storeStfBegin(loadInfo, "PatchEnd");
      storeStfEnd(loadInfo);
    }
  }
}

static void saveStfProject(LoadInfo *loadInfo) {
  int i;

  if (!projects[loadInfo->project].selected && loadInfo->saveSelectedOnly) {
    int currentPatch=-1;

    // If selection is only in one patch or track, only the modules and objects are saved.
    // In all other cases everything below the project level is generated (but only for the ones necessary to contain selected items)
    for(i=0;i<projects[loadInfo->project].maxPatches && (!loadInfo->generatePatches);i++) {
      int j;
      if (projects[loadInfo->project].patches[i].selected) {
        loadInfo->generatePatches=true;
      } else {
        for(j=0;j<projects[loadInfo->project].patches[i].maxObjects;j++) {
          if (projects[loadInfo->project].patches[i].objects[j].selected) {
            if (currentPatch<0) {
              currentPatch=i;
            } else if (currentPatch!=i) {
              loadInfo->generatePatches=true;
              break;
            }
          }
        }
      }
    }
  } else {
    loadInfo->generatePatches=true;
    storeStfBegin(loadInfo, "ProjectBegin");
    storeStfString(loadInfo, projects[loadInfo->project].projectName);
    storeStfEnd(loadInfo);
  }
  for(i=0;i<projects[loadInfo->project].maxPatches;i++) {
    saveStfSynFactoryPatch(loadInfo, i);
  }
  if (!loadInfo->saveSelectedOnly) {
    storeStfBegin(loadInfo, "ProjectEnd");
    storeStfEnd(loadInfo);
  }
}

static const int saveNoProjects=-3;
static const int saveSelectedOnly=-2;
static const int saveAllProjects=-1;
static void saveStfData(LoadInfo *loadInfo, bool saveSettings) {
  storeStfBegin(loadInfo, "STUDIOFACTORY100");
  storeStfEnd(loadInfo);
  if (saveSettings) {
EOF
    for(my $i=0;$i<scalar(@stfSave);$i++) {
      $rc = "$rc    " . $stfSave[$i] . "\n";
    }
$rc .= <<EOF;
  }
  if (loadInfo->project>=0) {
    if (loadInfo->project<maxProjects) {
      saveStfProject(loadInfo);
    } else {
      logprintf("Error saveStfData project %d is invalid.\\n", loadInfo->project);
    }
  } else if (loadInfo->project==saveAllProjects) {
    int i;

    for(i=0;i<maxProjects;i++) {
      loadInfo->project=i;
      saveStfProject(loadInfo);
    }
  }
  storeStfBegin(loadInfo, "STUDIOFACTORYEND");
  storeStfEnd(loadInfo);
}

static LoadError saveStfToFilename(ContextPtr aContext, const char *filename, bool saveSettings, int project) {
  LoadInfo loadInfo;

  memset(&loadInfo, 0, sizeof(LoadInfo));
  loadInfo.loadError=LOADERROR_FILENOTOPEN;
  loadInfo.file=fopen(filename, "wb");
  if (loadInfo.file) {
    loadInfo.loadError=LOADERROR_OK;
    loadInfo.project=project;
    saveStfData(&loadInfo, saveSettings);
    fclose(loadInfo.file);
  }

  return loadInfo.loadError;
}
EOF

$rd .= "static LoadError loadFromClipboard(void);\n"; # !!! Forward declaration
$rd .= "static void saveStfToClipboard(void);\n";     # !!! Forward declaration
$rd .= "static void clone(void);\n";                  # !!! Forward declaration

$rc .= <<EOF;
static LoadError loadFromClipboard(void) {
  LoadError error=LOADERROR_OTHERFORMAT;
  if (OpenClipboard(mainWindow)) {
    HANDLE xhandle=GetClipboardData(CF_PRIVATEFIRST);
    if (xhandle) {
      const unsigned char *buffer;
      logprintf("pasteFromClipboard binary format\\n");
      buffer=(const unsigned char *)GlobalLock(xhandle);
      error=loadFromBuffer(buffer, GlobalSize(xhandle));
      GlobalUnlock(xhandle);
    } else {
      xhandle=GetClipboardData(CF_TEXT);
      if (xhandle) {
        const unsigned char *buffer;
        logprintf("pasteFromClipboard text format\\n");
        buffer=(const unsigned char *)GlobalLock(xhandle);
        error=loadFromBuffer(buffer, GlobalSize(xhandle));
        GlobalUnlock(xhandle);
      }
    }
    CloseClipboard();
  }
  return error;
}

static void saveStfToClipboard(void) {
  LoadInfo loadInfo;

  if (OpenClipboard(mainWindow)) {
    EmptyClipboard();

    memset(&loadInfo, 0, sizeof(LoadInfo));
    loadInfo.saveSelectedOnly=true;
    loadInfo.project=currentProject;
    saveStfData(&loadInfo, false);

    // Reserve buffer of required length
    clipboardHandle=GlobalAlloc(GHND, loadInfo.bufferPos);
    logprintf("copySelectedToClipboard: buflen=%d xhandle=%d\\n", loadInfo.bufferPos, (int)(void*)clipboardHandle);
    if (clipboardHandle) {
      void *dest=GlobalLock(clipboardHandle);

      loadInfo.bufferPos=0;
      loadInfo.buffer=(unsigned char *)dest;
      saveStfData(&loadInfo, false);
      GlobalUnlock(clipboardHandle);
      SetClipboardData(CF_PRIVATEFIRST, clipboardHandle);
      CloseClipboard();
    }
  }
}

static void clone(void) {
  LoadInfo loadInfo;

  memset(&loadInfo, 0, sizeof(LoadInfo));
  loadInfo.saveSelectedOnly=true;
  loadInfo.project=currentProject;
  saveStfData(&loadInfo, false);

  loadInfo.buffer=(unsigned char *)malloc(loadInfo.bufferPos);
  loadInfo.bufferPos=0;
  saveStfData(&loadInfo, false);
  loadInfo.bufferSize=loadInfo.bufferPos;
  loadInfo.bufferPos=0;
  loadStf(&loadInfo);
  free(loadInfo.buffer);
}

static const char * const openFileFilter=
  "All supported types\\0*.669;*.di;*.dmf;*.mod;*.mtm;*.okt;*.s3m;*stf;*.stm;*syf;*.syn;*.wav\\0"
  "669 - 669 modules\\0*.669\\0"
  "DI - Digital Illusion encoded mod-modules\\0*.di\\0"
  "DMF - DMF modules\\0*.dmf\\0"
  "MID - Standard midi files\\0*.mid\\0"
  "MOD - ProTracker/NoiseTracker modules\\0*.mod\\0"
  "MTM - MultiTracker Modules\\0*.mtm\\0"
  "OKT - Oktalyzer modules\\0*.okt\\0"
  "S3M - ScreamTracker 3.x modules\\0*.s3m\\0"
  "STF - StudioFactory Files\\0*.stf\\0"
  "STM - ScreamTracker 1.0 modules\\0*.stm\\0"
  "SYN - SynFactory 1.x Files\\0*.syf;*.syn\\0"
  "WAV - Wavefiles\\0*.wav\\0"
  "All files\\0*.*\\0"
  "\\0";
static const char * const SaveFileFilter=
  "STF - StudioFactory File\\0*.stf\\0"
  "\\0";

//  "SYN - SynFactory 1.x File\\0*.syn\\0"

static const char * const captureFileFilter=
  "WAV - Wavefiles\\0*.wav\\0"
  "MP3 - MP3 files\\0*.mp3\\0"
  "\\0";

#define MAXFILEPATH               1024
#define MAXFILETITLE               512
static char openFilePath[MAXFILEPATH];
static char openFileTitle[MAXFILETITLE];
static void openFileSelect(ContextPtr aContext, bool merge) {
  OPENFILENAME ofn;

  ofn.lStructSize = sizeof(OPENFILENAME);
  ofn.hwndOwner = mainWindow;
  ofn.hInstance = NULL;
  ofn.lpstrFilter = openFileFilter;
  ofn.lpstrCustomFilter = NULL;
  ofn.nMaxCustFilter = 0;
  ofn.nFilterIndex = 0;
  ofn.lpstrFile = openFilePath;
  ofn.nMaxFile = MAXFILEPATH;
  ofn.lpstrFileTitle = openFileTitle;
  ofn.nMaxFileTitle = MAXFILETITLE;
  ofn.lpstrInitialDir = NULL;
  ofn.lpstrTitle = NULL;
  ofn.Flags = OFN_NOCHANGEDIR | OFN_EXPLORER | OFN_HIDEREADONLY;
  ofn.nFileOffset = 0;
  ofn.nFileExtension = 0;
  ofn.lpstrDefExt = "syn";
  ofn.lCustData = NULL;
  ofn.lpfnHook = NULL;
  ofn.lpTemplateName = NULL;
  if (GetOpenFileName(&ofn)) {
    loadFromFilename(aContext, openFilePath, merge);
//    setTitle();
  }
}

static void saveFileSelect(ContextPtr aContext, bool saveSettingsOnly) {
  static char saveFilePath[MAXFILEPATH];
  if ((!saveSettingsOnly) && currentProject>=0 && projects[currentProject].projectPath) {
    strncpy(saveFilePath, projects[currentProject].projectPath, MAXFILEPATH);
    saveFilePath[MAXFILEPATH-1]='\\0';
  } else {
    saveFilePath[0]='\\0';
  }
  OPENFILENAME ofn;

  ofn.lStructSize = sizeof(OPENFILENAME);
  ofn.hwndOwner = mainWindow;
  ofn.hInstance = NULL;
  ofn.lpstrFilter = SaveFileFilter;
  ofn.lpstrCustomFilter = NULL;
  ofn.nMaxCustFilter = 0;
  ofn.nFilterIndex = 0;
  ofn.lpstrFile = saveFilePath;
  ofn.nMaxFile = MAXFILEPATH;
  ofn.lpstrFileTitle = NULL;
  ofn.nMaxFileTitle = 0;
  ofn.lpstrInitialDir = NULL;
  ofn.lpstrTitle = NULL;
  ofn.Flags = OFN_OVERWRITEPROMPT | OFN_NOCHANGEDIR | OFN_HIDEREADONLY;
  ofn.nFileOffset = 0;
  ofn.nFileExtension = 0;
  ofn.lpstrDefExt = "stf";
  ofn.lCustData = NULL;
  ofn.lpfnHook = NULL;
  ofn.lpTemplateName = NULL;
  if (GetSaveFileName(&ofn)) {
    saveStfToFilename(aContext, saveFilePath, false, currentProject);
  }
}

static char currentCaptureFilePath[MAXFILEPATH];
static char currentCaptureFileTitle[MAXFILETITLE];
static int lastCapturFormat=1;
static bool captureFileSelect(void) {
  OPENFILENAME ofn;

  if (NULL==captureSample.file) {
    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.hwndOwner = mainWindow;
    ofn.hInstance = NULL;
    ofn.lpstrFilter = captureFileFilter;
    ofn.lpstrCustomFilter = NULL;
    ofn.nMaxCustFilter = 0;
    ofn.nFilterIndex = lastCapturFormat;
    ofn.lpstrFile = currentCaptureFilePath;
    ofn.nMaxFile = MAXFILEPATH;
    ofn.lpstrFileTitle = currentCaptureFileTitle;
    ofn.nMaxFileTitle = MAXFILETITLE;
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrTitle = NULL;
    ofn.Flags = OFN_OVERWRITEPROMPT | OFN_NOCHANGEDIR | OFN_HIDEREADONLY;
    ofn.nFileOffset = 0;
    ofn.nFileExtension = 0;
    ofn.lpstrDefExt = "wav";
    ofn.lCustData = NULL;
    ofn.lpfnHook = NULL;
    ofn.lpTemplateName = NULL;

    if (GetSaveFileName(&ofn)) {
      lastCapturFormat=ofn.nFilterIndex;
      switch(ofn.nFilterIndex) {
      case 1:
        captureSample.encoding=SAMPLE_ENCODING_WAV;
        break;
      case 2:
        captureSample.encoding=SAMPLE_ENCODING_MP3;
        break;
      default:
        captureSample.encoding=SAMPLE_ENCODING_WAV;
        break;
      }
      captureSample.sampleRate=44100;
      captureSample.noofChannels=2;
      captureSample.noofBits=16;
      createSampleFile(&captureSample, currentCaptureFilePath);
    }
  }
  return (captureSample.file!=NULL);
}
EOF

  rd('static bool captureFileSelect(void);'); # !!! forward decl
  rv('HANDLE clipboardHandle=NULL;');

}

##########################################################################
#
# Scripting
#
##########################################################################
sub script {

#
# Here -> linear memory 
#

# Dict layout
#   <0 term string max 32 chars ?> <offs to : def> <ptr to prev. dict entry or NULL>
#
#   <offs to prev. dict entry or NULL> <offs to : def> <0 term string>

  my @scriptingEngineDef=(
    'int rSP;',
    'int dSP;',
    'int workVars[26];',
    'int returnStack[scriptReturnStackSize];',
    'char *script;'
  );
}


sub beginGen {
  @refreshFunctions=();
  @initFunctions=();
  @termFunctions=();
  @initWindows=();
  @termWindows=();
  @synLoad=();
  @synSave=();
  @stfLoad=();
  @stfSave=();
  $ri = '';
  $rd = '';
  $rv = '';
  $rs = '';
  $rc = '';
}

sub genAll {
  my $cppFileName = shift;
  
  $platformOS = shift;
  $platformGraphics = shift;
  $platformAudio = shift;
  my $f;

  if (open($f, ">$cppFileName")) {
    beginGen();
    genSystemIncludes();
    genLogging();
    genLanguages();
    genGuiMessages();
    genContext();
    genGui();
    genStrings();
    $rv .= readCompleteFile("studiofactory_menuhelp.txt");
    $rv .= readCompleteFile("studiofactory_objectshelp.txt");

    genMenus();
#    genSamples();
    executeFile('studiofactory_4chscope.pl');
    genDsp();
    executeFile('studiofactory_joystick_win32.pl');
    genLoadSaveSupport();

    genExpandString();
    genMainWindow();
    genRenderHtml();
    genAudioRouter();

    genStdWindowProc();
    genHelpWindow();
    genColorSelectorWindow();
    genTransportControl();
    genAudioOutputScope();
    executeFile('studiofactory_audio_mixer.pl');
    genMidiActivityMonitor();
    genMidiScope();

    genSynfactoryPatchEditor();
    executeFile('studiofactory_samples.pl');
    executeFile('studiofactory_sequencer.pl');

    genProjectBrowser();

    genMidiRouter();

    genConfig();
    genLoadSave();

    genMainWindowProcessing();
    genRefreshWindowFunction();
    genMain();

    print $f $ri;
    print $f $rd;
    print $f $rv;
    print $f $rs;
    print $f $rc;
    close($f)
  } else {
    print "Couldn't generate $cppFileName\n";
  }
}


#
#  FileName, platform/OS, Graphic Driver, Audio Driver
#
genAll('studiofactory_mingw32_gdi.cpp', 'Win32', 'GDI', 'Win32');
#genAll('studiofactory_mingw32_gtk.cpp', 'Win32', 'Gtk', 'Win32');
#genAll('studiofactory_cygwin_gtk.cpp', 'Unix', 'Gtk', 'Win32');
#genAll('studiofactory_cygwin_xt.cpp', 'Unix', 'Xt', 'Win32');

# Compile windows version (mingw)
system('windres -o studiofactoryrc.o studiofactory.rc');
system('gcc -Wall -Wwrite-strings -O3 -mcpu=i586 -mwindows -mno-cygwin -fno-rtti -fno-exceptions -o studiofactory_mingw32_gdi.exe studiofactoryrc.o studiofactory_mingw32_gdi.cpp -lwinmm -lwsock32');
system('strip studiofactory_mingw32_gdi.exe');

# Compile Gtk version (cygwin)
#system('gcc -Wall -Wwrite-strings -O3 -mcpu=i586 -fno-rtti -fno-exceptions -o studiofactory_cygwin_gtk.exe studiofactory_cygwin_gtk.cpp -lwinmm -lwsock32');

# Compile Xt version (cygwin)
#system('gcc -Wall -Wwrite-strings -O3 -mcpu=i586 -fno-rtti -fno-exceptions -o studiofactory_cygwin_xt.exe studiofactory_cygwin_xt.cpp -lwinmm -lwsock32');

#system('gcc -Wall -mwindows -mno-cygwin -lwinmm -lwsock32 -o studiofactory_mingw32_gdi.exe studiofactoryrc.o studiofactory_mingw32_gdi.cpp');
#system('gcc -O3 -fomit-frame-pointer -fno-exceptions -lwinmm -Ic:/usr/cygwin/usr/X11R6/include -o studiofactory_cygwin_x.exe studiofactory_cygwin_x.cpp');
#system('gcc -O3 -fomit-frame-pointer -fno-exceptions -lwinmm -Ic:/usr/cygwin/usr/X11R6/include -o studiofactory_cygwin_gtk.exe studiofactory_cygwin_gtk.cpp');
#system('strip studiofactory_cygwin_x.exe');

